<?php

namespace Drupal\logger_entity\Plugin\LoggerTarget;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\logger\Attribute\LoggerTarget;
use Drupal\logger\LoggerEntryInterface;
use Drupal\logger\Plugin\LoggerTargetBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Uuid\UuidInterface;

/**
 * Provides a file logger target.
 */
#[LoggerTarget(
  id: 'entity',
  label: new TranslatableMarkup('Entity'),
  description: new TranslatableMarkup('Writes log to the entity storage. Not recommended for production.'),
  weight: 40
)]
class Entity extends LoggerTargetBase implements ContainerFactoryPluginInterface {
  /**
   * The entity type to store logs.
   *
   * @var string
   */
  const ENTITY_TYPE = 'logger_entry';

  /**
   * UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected UuidInterface $uuid;

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    // Store the container and lazily fetch services when needed.
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->uuid = $container->get('uuid');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function persist(LoggerEntryInterface $entry): void {
    $data = $entry->getData();
    $channel = $data['channel'];
    $message = $data['message'] ?? $data['message_raw'] ?? '';
    $this->entityTypeManager->getStorage(self::ENTITY_TYPE)->create([
      'uuid' => $data['uuid'] ?? $this->uuid->generate(),
      'time' => time(),
      'severity' => $entry->getLogLevel(),
      'channel' => $channel,
      'message' => $message,
      'data' => json_encode($data),
    ])->save();
  }

  /**
   * {@inheritdoc}
   */
  public function validateTarget(array $target): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultConfiguration(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigForm() {
    return [
      '#markup' => '',
    ];
  }

}
