<?php

namespace Drupal\Tests\login_disable\Functional;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests configuring Login Disable.
 *
 * @group login_disable
 */
class LoginDisableTest extends BrowserTestBase {

  use StringTranslationTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'login_disable',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests login disable configuration form.
   */
  public function testConfiguration() {
    $settings = $this->config('login_disable.settings');

    $this->drupalLogin($this->drupalCreateUser([
      'administer permissions',
    ]));
    $this->drupalGet('admin/config/people/login-disable');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldValueEquals(
      'login_disable_is_active', $settings->get('login_disable_is_active')
    );
    $this->assertSession()->fieldValueEquals(
      'login_disable_key', $settings->get('login_disable_key')
    );
    $this->assertSession()->fieldValueEquals(
      'login_disable_message', $settings->get('login_disable_message')
    );

    // Post the form.
    $this->drupalGet('admin/config/people/login-disable');
    $this->submitForm([
      'login_disable_is_active' => TRUE,
      'login_disable_key' => 'test-admin',
      'login_disable_message' => 'Test message',
    ], $this->t('Save configuration'));

    // Test the new values.
    $this->drupalGet('admin/config/people/login-disable');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->fieldValueEquals('login_disable_is_active', TRUE);
    $this->assertSession()->fieldValueEquals('login_disable_key', 'test-admin');
    $this->assertSession()->fieldValueEquals('login_disable_message', 'Test message');
  }

  /**
   * Tests disabled login form.
   */
  public function testLoginDisableForm() {
    // Test disabled login form.
    $this->disableLogin();
    $this->drupalGet('user/login');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains($this->t('Member access has been temporarily disabled. Please try again later.'));
  }

  /**
   * Tests bypass disabled logins.
   */
  public function testCanBypassLoginDisable() {
    // Set up a user to test disabled login bypassing.
    $this->disableLogin();
    $this->drupalLogin($this->drupalCreateUser([
      'bypass disabled login',
    ]));
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests bypass disabled logins.
   */
  public function testCannotBypassLoginDisable() {
    // Set up a user to test disabled login halt.
    $this->disableLogin();
    $this->drupalLogin($this->drupalCreateUser());
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Helper method to disables login in config.
   */
  private function disableLogin() : void {
    $this
      ->config('login_disable.settings')
      ->set('login_disable_is_active', TRUE)
      ->save();

    drupal_flush_all_caches();
  }

}
