<?php

namespace Drupal\Tests\login_flow\Kernel;

use Drupal\login_flow\Controller\LoginFlowEmailOnlyController;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;

/**
 * Tests the email login controller.
 *
 * @covers \Drupal\login_flow\Controller\LoginFlowEmailOnlyController
 * @group authenticate_by_mail
 */
class LoginFlowControllerTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'login_flow',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('user', 'users_data');

    $this->installConfig([
      'login_flow',
      'system',
    ]);
  }

  /**
   * Tests login link behavior while already logged in.
   */
  public function testAlreadyLoggedIn() {
    $user1 = $this->createUser();
    $user2 = $this->createUser();

    $this->setCurrentUser($user1);

    /** @var \Drupal\login_flow\Controller\LoginFlowEmailOnlyController */
    $login_controller = \Drupal::classResolver(LoginFlowEmailOnlyController::class);
    $login_controller->authenticate($user2->id(), $timestamp = \time(), \user_pass_rehash($user2, $timestamp));

    $this->assertCount(1, $messages = \Drupal::messenger()->messagesByType(MessengerInterface::TYPE_ERROR));
    $this->assertContainsEquals('You must log out before you can attempt to log in as another user.', $messages);
    $this->assertSame($user1, \Drupal::currentUser()->getAccount(), 'Current user did not change');
  }

  /**
   * Tests login link behavior for blocked users.
   */
  public function testBlockedUser() {
    $user = $this->createUser();
    $user->block()->save();

    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is anonymous');

    /** @var \Drupal\login_flow\Controller\LoginFlowEmailOnlyController */
    $login_controller = \Drupal::classResolver(LoginFlowEmailOnlyController::class);
    $login_controller->authenticate($user->id(), $timestamp = \time(), \user_pass_rehash($user, $timestamp));

    $this->assertCount(1, $messages = \Drupal::messenger()->messagesByType(MessengerInterface::TYPE_ERROR));
    $this->assertContainsEquals('The one-time login link you clicked is invalid.', $messages);
    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is still anonymous');
  }

  /**
   * Tests login link behavior for a user that was deleted.
   */
  public function testDeletedUser() {
    $user = $this->createUser();
    $user->delete();

    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is anonymous');

    /** @var \Drupal\login_flow\Controller\LoginFlowEmailOnlyController */
    $login_controller = \Drupal::classResolver(LoginFlowEmailOnlyController::class);
    $login_controller->authenticate($user->id(), $timestamp = \time(), \user_pass_rehash($user, $timestamp));

    $this->assertCount(1, $messages = \Drupal::messenger()->messagesByType(MessengerInterface::TYPE_ERROR));
    $this->assertContainsEquals('The one-time login link you clicked is invalid.', $messages);
    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is still anonymous');
  }

  /**
   * Tests login links which were generated prior to the user's last login.
   */
  public function testLinkGeneratedBeforeLastLogin() {
    $user = $this->createUser();
    $user->setLastLoginTime(\time())->save();

    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is anonymous');

    /** @var \Drupal\login_flow\Controller\LoginFlowEmailOnlyController */
    $login_controller = \Drupal::classResolver(LoginFlowEmailOnlyController::class);
    $login_controller->authenticate($user->id(), $timestamp = \time() - 1, \user_pass_rehash($user, $timestamp));

    $this->assertCount(1, $messages = \Drupal::messenger()->messagesByType(MessengerInterface::TYPE_ERROR));
    $this->assertContainsEquals('You have tried to use a one-time login link that has either been used or is no longer valid.', $messages);
    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is still anonymous');
  }

  /**
   * Tests login links which were generated in the future.
   */
  public function testLinkGeneratedInTheFuture() {
    $user = $this->createUser();

    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is anonymous');

    /** @var \Drupal\login_flow\Controller\LoginFlowEmailOnlyController */
    $login_controller = \Drupal::classResolver(LoginFlowEmailOnlyController::class);
    $login_controller->authenticate($user->id(), $timestamp = \time() + 10, \user_pass_rehash($user, $timestamp));

    $this->assertCount(1, $messages = \Drupal::messenger()->messagesByType(MessengerInterface::TYPE_ERROR));
    $this->assertContainsEquals('You have tried to use a one-time login link that has either been used or is no longer valid.', $messages);

    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is still anonymous');
  }

  /**
   * Tests login links with invalid hashes.
   */
  public function testLinkWithInvalidHash() {
    $user = $this->createUser();

    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is anonymous');

    /** @var \Drupal\login_flow\Controller\LoginFlowEmailOnlyController */
    $login_controller = \Drupal::classResolver(LoginFlowEmailOnlyController::class);
    $login_controller->authenticate($user->id(), \time(), \user_pass_rehash($user, 0));

    $this->assertCount(1, $messages = \Drupal::messenger()->messagesByType(MessengerInterface::TYPE_ERROR));
    $this->assertContainsEquals('You have tried to use a one-time login link that has either been used or is no longer valid.', $messages);

    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is still anonymous');
  }

  /**
   * Tests valid login links.
   */
  public function testValidLink() {
    $user = $this->createUser();

    $this->assertTrue(\Drupal::currentUser()->isAnonymous(), 'Current user is anonymous');

    /** @var \Drupal\login_flow\Controller\LoginFlowEmailOnlyController */
    $login_controller = \Drupal::classResolver(LoginFlowEmailOnlyController::class);
    // Set token time timestamp to 1 hour in past.
    $login_controller->authenticate($user->id(), $timestamp = \time() - 3600, \user_pass_rehash($user, $timestamp));

    $this->assertTrue(\Drupal::currentUser()->isAuthenticated(), 'Current user is now authenticated');
  }

}
