<?php

namespace Drupal\login_flow\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implements Login Flow configuration form.
 */
class LoginFlowSettingsForm extends ConfigFormBase {

  /**
   * Injected Plugin Manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $pluginManager;

  /**
   * Constructs a LoginFlowSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $plugin_manager
   *   The Login Flow plugin manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, PluginManagerInterface $plugin_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->pluginManager = $plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.login_flow'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'login_flow_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    $config_names = [
      'login_flow.settings',
    ];
    return $config_names;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $configuration = $this->config('login_flow.settings');
    $plugin_definitions = $this->pluginManager->getDefinitions();
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    $form['plugins'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Name'),
        $this->t('Enabled'),
        $this->t('Summary'),
        $this->t('Operations'),
        $this->t('Weight'),
      ],
      '#empty' => $this->t('Sorry, there are no plugins found.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'table-sort-weight',
        ],
      ],
    ];

    $plugin_settings = [];
    $plugins = $configuration->get('plugins');
    if (!empty($plugins)) {
      foreach ($plugins as $plugin) {
        $plugin_settings[$plugin['plugin_id']] = $plugin;
      }
    }

    // Add weight and enabled status from settings and then sort by weight.
    foreach ($plugin_definitions as $plugin_id => $plugin) {
      $plugin_definitions[$plugin_id]['enabled'] = (empty($plugin_settings[$plugin_id]['enabled'])) ? 0 : $plugin_settings[$plugin_id]['enabled'];
      $plugin_definitions[$plugin_id]['weight'] = (empty($plugin_settings[$plugin_id]['weight'])) ? 0 : $plugin_settings[$plugin_id]['weight'];
    }
    uasort($plugin_definitions, 'Drupal\Component\Utility\SortArray::sortByWeightElement');

    foreach ($plugin_definitions as $plugin_id => $plugin) {
      $config_object = $this->config('login_flow.plugin.' . $plugin_id);
      $plugin_config = (empty($config_object->get('configuration'))) ? [] : $config_object->get('configuration');
      $plugin_instance = $this->pluginManager->createInstance($plugin_id, $plugin_config);

      $form['plugins'][$plugin_id]['#attributes']['class'][] = 'draggable';
      $form['plugins'][$plugin_id]['#weight'] = $plugin['weight'];

      $form['plugins'][$plugin_id]['name'] = [
        '#markup' => $this->t('<strong>@label:</strong> @description', [
          '@label' => $plugin['label'],
          '@description' => $plugin['description'],
        ]),
      ];

      $form['plugins'][$plugin_id]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enabled'),
        '#default_value' => $plugin['enabled'],
      ];

      // This shouldn't be disabled.
      if ($plugin_id == 'login_flow_password') {
        $form['plugins'][$plugin_id]['enabled']['#disabled'] = TRUE;
      }

      $form['plugins'][$plugin_id]['summary'] = [
        '#prefix' => '<span class="login-flow-plugin-summary ' . str_replace('_', '-', $plugin_id) . '">',
        '#markup' => $plugin_instance->getConfigurationSummary(),
        '#suffix' => '</span>',
      ];

      $form['plugins'][$plugin_id]['settings'] = [
        '#type' => 'link',
        '#title' => $this->t('Edit'),
        '#url' => Url::fromRoute('login_flow.login_flow_plugin_settings_form', [
          'id' => $plugin_id,
        ]),
        '#attributes' => [
          'class' => ['use-ajax', 'button', 'button--small'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => 800,
          ]),
        ],
      ];

      $form['plugins'][$plugin_id]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @label', ['@label' => $plugin['label']]),
        '#title_display' => 'invisible',
        '#default_value' => $plugin['weight'],
        '#attributes' => ['class' => ['table-sort-weight']],
      ];
    }

    $form['username_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username title'),
      '#default_value' => $configuration->get('username_title'),
      '#description' => $this->t('The title for the username field.'),
    ];

    $form['username_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username description'),
      '#default_value' => $configuration->get('username_description'),
      '#description' => $this->t('The description for the username field.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $plugins = [];
    $plugins_value = $form_state->getValue('plugins');
    foreach ($plugins_value as $plugin_id => $settings) {
      $plugins[] = ['plugin_id' => $plugin_id] + $settings;
    }

    $this->config('login_flow.settings')
      ->set('plugins', $plugins)
      ->set('username_title', $form_state->getValue('username_title'))
      ->set('username_description', $form_state->getValue('username_description'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
