<?php

namespace Drupal\login_flow\Plugin\LoginFlow;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\login_flow\Attribute\LoginFlow;
use Drupal\login_flow\Plugin\LoginFlowBase;
use Drupal\login_flow\Plugin\LoginFlowInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a base implementation for an Login Flow plugin.
 *
 * @see \Drupal\login_flow\Attribute\LoginFlow
 * @see \Drupal\login_flow\LoginFlowManager
 * @see \Drupal\login_flow\LoginFlowInterface
 * @see plugin_api
 */
#[LoginFlow(
  id: 'login_flow_email',
  label: new TranslatableMarkup('Email'),
  description: new TranslatableMarkup('Allow someone to login using an email login link.'),
  uses_drupal_password: FALSE,
  supports_password_reset: FALSE,
  will_redirect: TRUE,
)]
final class EmailOnly extends LoginFlowBase implements LoginFlowInterface {

  /**
   * Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Mail Manager Service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * Creates a new Role Paywall User Role instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Mail\MailManagerInterface $mail_manager
   *   The mail manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    MailManagerInterface $mail_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->mailManager = $mail_manager;

  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.mail'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigurationSummary() {
    $summary = [];
    if (!empty($this->configuration['roles'])) {
      $allowed_roles = $this->configuration['roles'];
      $roles_entities = $this->entityTypeManager->getStorage('user_role')->loadMultiple($allowed_roles);
      $role_labels = [];

      foreach ($roles_entities as $role) {
        $role_labels[] = $role->label();
      }
      $summary[] = $this->t('Roles: @roles', ['@roles' => implode(', ', $role_labels)]);
    }

    if (!empty($this->configuration['domains'])) {
      $domains = $this->configuration['domains'];
      $summary[] = $this->t('Domains: @domains', ['@domains' => implode(', ', $domains)]);
    }

    if (empty($summary)) {
      $summary[] = $this->t('No restrictions');
    }

    return implode('<br>', $summary);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    // The default configuration is no roles selected.
    return [
      'roles' => [],
      'domains' => [],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm($form, FormStateInterface $form_state) {
    parent::buildConfigurationForm($form, $form_state);

    $roles = [];
    $roles_entities = $this->entityTypeManager->getStorage('user_role')->loadMultiple();

    foreach ($roles_entities as $role_id => $role) {
      $roles[$role_id] = $role->label();
    }

    $form['roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles that can login via email'),
      '#options' => $roles,
      '#description' => $this->t('Leave empty to allow all roles.'),
      '#default_value' => empty($this->configuration['roles']) ? [] : $this->configuration['roles'],
    ];

    $form['domains'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed email domains'),
      '#description' => $this->t('One domain per line. Leave empty to allow all domains.'),
      '#default_value' => empty($this->configuration['domains']) ? '' : implode("\n", $this->configuration['domains']),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $allowed_roles = array_filter($form_state->getValue('roles'));
    $this->configuration['roles'] = array_keys($allowed_roles);
    $this->configuration['domains'] = array_filter(explode("\n", $form_state->getValue('domains')));
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(UserInterface $account, FormStateInterface $form_state) {
    if (!empty($this->configuration['roles'])) {
      $account_roles = $account->getRoles();
      $allowed_roles = $this->configuration['roles'];
      $intersection = array_intersect($account_roles, $allowed_roles);

      if (!empty($intersection)) {
        return TRUE;
      }
    }

    if (!empty($this->configuration['domains'])) {
      $email = ($account->isAnonymous()) ? $form_state->getValue('name') : $account->getEmail();
      $email_parts = explode('@', $email);
      $domain = end($email_parts);

      if (in_array($domain, $this->configuration['domains'])) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function validateLogin(UserInterface $account, FormStateInterface $form_state) {
    if (!$account->isAnonymous()) {
      if ($to = $account->getEmail()) {
        $this->mailManager->mail('login_flow', 'login', $to, $account->getPreferredLangcode(), [
          'user' => $account,
        ]);
      }
    }

    $this->messenger()->addStatus($this->t('If %name is a valid account, an email was sent with a one-time login link.', [
      '%name' => $form_state->getValue('name'),
    ]));

    return '<front>';
  }

  /**
   * {@inheritdoc}
   */
  public function processPasswordReset(UserInterface $account) {
    $this->messenger()->addStatus($this->t('Unable to reset password - not supported.'));
  }

  /**
   * {@inheritdoc}
   */
  public function processLogout(UserInterface $account) {
    user_logout();
  }

}
