<?php

namespace Drupal\login_flow\Plugin\LoginFlow;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\login_flow\Attribute\LoginFlow;
use Drupal\login_flow\Plugin\LoginFlowBase;
use Drupal\login_flow\Plugin\LoginFlowInterface;
use Drupal\user\UserAuthenticationInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a base implementation for an Login Flow plugin.
 *
 * @see \Drupal\login_flow\Attribute\LoginFlow
 * @see \Drupal\login_flow\LoginFlowManager
 * @see \Drupal\login_flow\LoginFlowInterface
 * @see plugin_api
 */
#[LoginFlow(
  id: 'login_flow_password',
  label: new TranslatableMarkup('Password'),
  description: new TranslatableMarkup('Authenticate via Drupal user password.'),
  uses_drupal_password: TRUE,
  supports_password_reset: TRUE,
)]
final class Password extends LoginFlowBase implements LoginFlowInterface {

  /**
   * The user authentication object.
   *
   * @var \Drupal\user\UserAuthenticationInterface
   */
  protected $userAuth;

  /**
   * Constructs a Password object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\user\UserAuthenticationInterface $user_auth
   *   The user authentication service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, UserAuthenticationInterface $user_auth) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->userAuth = $user_auth;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('user.auth'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigurationSummary() {
    return 'This should remain at the bottom of the list.';
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm($form, FormStateInterface $form_state) {
    parent::buildConfigurationForm($form, $form_state);
    $form['message'] = [
      '#markup' => '<p>' . $this->t('This plug-in has no options to configure.') . '</p>',
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {}

  /**
   * {@inheritdoc}
   */
  public function validateLogin(UserInterface $account, FormStateInterface $form_state) {
    $password = $form_state->getValue('pass');
    if (empty($password)) {
      $form_state->setErrorByName('pass', $this->t('Password field is required.'));
    }
    $authenticated = $this->userAuth->authenticateAccount($account, $password);
    if ($authenticated) {
      $this->setUserAccountId($account->id());
    }
    return $authenticated;
  }

  /**
   * {@inheritdoc}
   */
  public function processPasswordReset(UserInterface $account) {
    if ($account) {
      // Mail one time login URL and instructions using current language.
      $mail = _user_mail_notify('password_reset', $account);
      if (!empty($mail)) {
        $this->getLogger('user')
          ->info('Password reset instructions mailed to %name at %email.', [
            '%name' => $account->getAccountName(),
            '%email' => $account->getEmail(),
          ]);
      }
    }

    // Make sure the status text is displayed even if no email was sent. This
    // message is deliberately the same as the success message for privacy.
    $this->messenger()->addStatus($this->t('If a valid account is found, an email will be sent with instructions to reset your password.'));
  }

  /**
   * {@inheritdoc}
   */
  public function processLogout(UserInterface $account) {
    user_logout();
  }

}
