<?php

namespace Drupal\login_flow\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\user\UserInterface;

/**
 * Provides an interface for a Login Flow plugin.
 *
 * @see \Drupal\login_flow\Attribute\LoginFlow
 * @see \Drupal\login_flow\LoginFlowManager
 * @see \Drupal\login_flow\LoginFlowBase
 * @see plugin_api
 */
interface LoginFlowInterface extends PluginInspectionInterface, PluginFormInterface {

  /**
   * Get the plugin configuration.
   *
   * @return array
   *   Array of settings for this plugin.
   */
  public function getConfiguration();

  /**
   * Get a summary of the plugin configuration.
   *
   * @return string
   *   A summary of the current settings for this plugin.
   */
  public function getConfigurationSummary();

  /**
   * Return the default configuration values for this plugin.
   *
   * @return array
   *   Associative array of default configuration values.
   */
  public function defaultConfiguration();

  /**
   * Checks if the Login Flow is applicable for the given account.
   *
   * @param \Drupal\user\UserInterface $account
   *   The user account, if one has been found.
   * @param object $form_state
   *   The form state object.
   *
   * @return bool
   *   TRUE if this Login Flow plug-in is applicable, FALSE otherwise.
   */
  public function isApplicable(UserInterface $account, FormStateInterface $form_state);

  /**
   * Validate the user login request for the account.
   *
   * @param \Drupal\user\UserInterface $account
   *   The user account.
   * @param object $form_state
   *   The form state object.
   *
   * @return \Drupal\Core\Url|string|bool
   *   Return either a URL or route for a redirect or boolean login result.
   */
  public function validateLogin(UserInterface $account, FormStateInterface $form_state);

  /**
   * Set the id of the user account to be logged in.
   */
  public function setUserAccountId(int $user_id);

  /**
   * Gets the id of the user account.
   *
   * @return int
   *   The id of the user account.
   */
  public function getUserAccountId();

  /**
   * Process a password reset request for the given account.
   *
   * @param \Drupal\user\UserInterface $account
   *   The user account.
   */
  public function processPasswordReset(UserInterface $account);

  /**
   * Process a logout request for the given account.
   *
   * @param \Drupal\user\UserInterface $account
   *   The user account.
   */
  public function processLogout(UserInterface $account);

  /**
   * Get additional form elements for the login form.
   *
   * @return array
   *   An array of form elements to be added to the login form.
   */
  public function getFormElements();

}
