<?php

namespace Drupal\login_flow\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\login_flow\Attribute\LoginFlow;

/**
 * Manages Login Flow Handler plugins.
 *
 * @see hook_login_flow_info_alter()
 * @see \Drupal\login_flow\Attribute\LoginFlow
 * @see \Drupal\login_flow\LoginFlowManager
 * @see \Drupal\login_flow\LoginFlowBase
 * @see plugin_api
 */
class LoginFlowManager extends DefaultPluginManager {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new LoginFlowManager.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler, ConfigFactoryInterface $config_factory) {
    parent::__construct('Plugin/LoginFlow', $namespaces, $module_handler, 'Drupal\login_flow\Plugin\LoginFlowInterface', LoginFlow::class);
    $this->configFactory = $config_factory;
    $this->alterInfo('login_flow_info');
    $this->setCacheBackend($cache_backend, 'login_flow_plugins');
  }

  /**
   * Returns an array of the plugins that are enabled.
   */
  public function getEnabledPlugins() {
    $config = $this->configFactory->get('login_flow.settings');
    $plugin_settings = $config->get('plugins');
    uasort($plugin_settings, 'Drupal\Component\Utility\SortArray::sortByWeightElement');

    $plugins = [];

    foreach ($plugin_settings as $plugin) {
      if ($plugin['enabled']) {
        $plugin_id = $plugin['plugin_id'];
        $config_object = $this->configFactory->get('login_flow.plugin.' . $plugin_id);
        $plugin_config = (empty($config_object->get('configuration'))) ? [] : $config_object->get('configuration');
        $plugins[$plugin_id] = $this->createInstance($plugin_id, $plugin_config);
      }
    }

    return $plugins;
  }

}
