<?php

namespace Drupal\login_flow\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implements Login Flow configuration form.
 */
class LoginFlowSettingsForm extends ConfigFormBase {

  /**
   * Authentication Plugin Manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $authenticationPluginManager;

  /**
   * Challenge Plugin Manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $challengePluginManager;

  /**
   * Constructs a LoginFlowSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $authentication_plugin_manager
   *   The Login Flow Authentication plugin manager.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $challenge_plugin_manager
   *   The Login Flow Challenge plugin manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, PluginManagerInterface $authentication_plugin_manager, PluginManagerInterface $challenge_plugin_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->authenticationPluginManager = $authentication_plugin_manager;
    $this->challengePluginManager = $challenge_plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.login_flow.authentication'),
      $container->get('plugin.manager.login_flow.challenge'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'login_flow_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    $config_names = [
      'login_flow.settings',
    ];
    return $config_names;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $configuration = $this->config('login_flow.settings');
    $authentication_plugin_definitions = $this->authenticationPluginManager->getDefinitions();
    $challenge_plugin_definitions = $this->challengePluginManager->getDefinitions();
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    $form['authentication'] = [
      '#type' => 'container',
      '#tree' => TRUE,
      '#weight' => 1,
      'header' => [
        '#markup' => $this->t('<h2>Authentication</h2>Authentication plug-ins are evaluated in order, only the first matching plug-in will be used to evaluate the login attempt.'),
      ],
    ];

    $form['authentication']['plugins'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Plug-in Name'),
        $this->t('Enabled'),
        $this->t('Summary'),
        $this->t('Operations'),
        $this->t('Weight'),
      ],
      '#empty' => $this->t('Sorry, there are no plug-ins found.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'table-sort-weight',
        ],
      ],
    ];

    $authentication_plugin_settings = [];
    $authentication_plugins = $configuration->get('authentication');
    if (!empty($authentication_plugins)) {
      foreach ($authentication_plugins as $plugin) {
        $authentication_plugin_settings[$plugin['plugin_id']] = $plugin;
      }
    }

    // Add weight and enabled status from settings and then sort by weight.
    foreach ($authentication_plugin_definitions as $plugin_id => $plugin) {
      $authentication_plugin_definitions[$plugin_id]['enabled'] = (empty($authentication_plugin_settings[$plugin_id]['enabled'])) ? 0 : $authentication_plugin_settings[$plugin_id]['enabled'];
      $authentication_plugin_definitions[$plugin_id]['weight'] = (empty($authentication_plugin_settings[$plugin_id]['weight'])) ? 0 : $authentication_plugin_settings[$plugin_id]['weight'];
    }
    uasort($authentication_plugin_definitions, 'Drupal\Component\Utility\SortArray::sortByWeightElement');

    foreach ($authentication_plugin_definitions as $plugin_id => $plugin) {
      $config_object = $this->config('login_flow.authentication.' . $plugin_id);
      $plugin_config = (empty($config_object->get('configuration'))) ? [] : $config_object->get('configuration');
      $plugin_instance = $this->authenticationPluginManager->createInstance($plugin_id, $plugin_config);

      $form['authentication']['plugins'][$plugin_id]['#attributes']['class'][] = 'draggable';
      $form['authentication']['plugins'][$plugin_id]['#weight'] = $plugin['weight'];

      $form['authentication']['plugins'][$plugin_id]['name'] = [
        '#markup' => $this->t('<strong>@label:</strong> @description', [
          '@label' => $plugin['label'],
          '@description' => $plugin['description'],
        ]),
      ];

      $form['authentication']['plugins'][$plugin_id]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enabled'),
        '#default_value' => $plugin['enabled'],
      ];

      // This shouldn't be disabled.
      if ($plugin_id == 'login_flow_password') {
        $form['authentication']['plugins'][$plugin_id]['enabled']['#disabled'] = TRUE;
      }

      $form['authentication']['plugins'][$plugin_id]['summary'] = [
        '#prefix' => '<span class="login-flow-authentication-summary ' . str_replace('_', '-', $plugin_id) . '">',
        '#markup' => $plugin_instance->getConfigurationSummary(),
        '#suffix' => '</span>',
      ];

      $form['authentication']['plugins'][$plugin_id]['settings'] = [
        '#type' => 'link',
        '#title' => $this->t('Edit'),
        '#url' => Url::fromRoute('login_flow.login_flow_authentication_settings_form', [
          'id' => $plugin_id,
        ]),
        '#attributes' => [
          'class' => ['use-ajax', 'button', 'button--small'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => 800,
          ]),
        ],
      ];

      $form['authentication']['plugins'][$plugin_id]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @label', ['@label' => $plugin['label']]),
        '#title_display' => 'invisible',
        '#default_value' => $plugin['weight'],
        '#attributes' => ['class' => ['table-sort-weight']],
      ];
    }

    $form['challenge'] = [
      '#type' => 'container',
      '#tree' => TRUE,
      '#weight' => 2,
      'header' => [
        '#markup' => $this->t('<h2>Challenge</h2>Challenge plug-ins are optional and executed after the Authentication Plug-in has handled the login attempt.'),
      ],
    ];

    $form['challenge']['plugins'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Plug-in Name'),
        $this->t('Enabled'),
        $this->t('Summary'),
        $this->t('Operations'),
        $this->t('Weight'),
      ],
      '#empty' => $this->t('Sorry, there are no plug-ins found.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'table-sort-weight',
        ],
      ],
    ];

    $challenge_plugin_settings = [];
    $challenge_plugins = $configuration->get('challenge');
    if (!empty($challenge_plugins)) {
      foreach ($challenge_plugins as $plugin) {
        $challenge_plugin_settings[$plugin['plugin_id']] = $plugin;
      }
    }

    // Add weight and enabled status from settings and then sort by weight.
    foreach ($challenge_plugin_definitions as $plugin_id => $plugin) {
      $challenge_plugin_definitions[$plugin_id]['enabled'] = (empty($challenge_plugin_settings[$plugin_id]['enabled'])) ? 0 : $challenge_plugin_settings[$plugin_id]['enabled'];
      $challenge_plugin_definitions[$plugin_id]['weight'] = (empty($challenge_plugin_settings[$plugin_id]['weight'])) ? 0 : $challenge_plugin_settings[$plugin_id]['weight'];
    }
    uasort($challenge_plugin_definitions, 'Drupal\Component\Utility\SortArray::sortByWeightElement');

    foreach ($challenge_plugin_definitions as $plugin_id => $plugin) {
      $config_object = $this->config('login_flow.challenge.' . $plugin_id);
      $plugin_config = (empty($config_object->get('configuration'))) ? [] : $config_object->get('configuration');
      $plugin_instance = $this->challengePluginManager->createInstance($plugin_id, $plugin_config);

      $form['challenge']['plugins'][$plugin_id]['#attributes']['class'][] = 'draggable';
      $form['challenge']['plugins'][$plugin_id]['#weight'] = $plugin['weight'];

      $form['challenge']['plugins'][$plugin_id]['name'] = [
        '#markup' => $this->t('<strong>@label:</strong> @description', [
          '@label' => $plugin['label'],
          '@description' => $plugin['description'],
        ]),
      ];

      $form['challenge']['plugins'][$plugin_id]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enabled'),
        '#default_value' => $plugin['enabled'],
      ];

      $form['challenge']['plugins'][$plugin_id]['summary'] = [
        '#prefix' => '<span class="login-flow-challenge-summary ' . str_replace('_', '-', $plugin_id) . '">',
        '#markup' => $plugin_instance->getConfigurationSummary(),
        '#suffix' => '</span>',
      ];

      $form['challenge']['plugins'][$plugin_id]['settings'] = [
        '#type' => 'link',
        '#title' => $this->t('Edit'),
        '#url' => Url::fromRoute('login_flow.login_flow_challenge_settings_form', [
          'id' => $plugin_id,
        ]),
        '#attributes' => [
          'class' => ['use-ajax', 'button', 'button--small'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => 800,
          ]),
        ],
      ];

      $form['challenge']['plugins'][$plugin_id]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @label', ['@label' => $plugin['label']]),
        '#title_display' => 'invisible',
        '#default_value' => $plugin['weight'],
        '#attributes' => ['class' => ['table-sort-weight']],
      ];
    }

    $form['form_settings'] = [
      '#type' => 'container',
      '#weight' => 2,
      'header' => [
        '#markup' => $this->t('<h2>Login Form Settings</h2>'),
      ],
    ];

    $form['form_settings']['username_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username title'),
      '#default_value' => $configuration->get('username_title'),
      '#description' => $this->t('The title for the username field.'),
      '#weight' => 10,
    ];

    $form['form_settings']['username_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username description'),
      '#default_value' => $configuration->get('username_description'),
      '#description' => $this->t('The description for the username field.'),
      '#weight' => 11,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $authentication_plugins = [];
    $authentication_plugins_value = $form_state->getValue(['authentication', 'plugins']);
    if (!empty($authentication_plugins_value)) {
      foreach ($authentication_plugins_value as $plugin_id => $settings) {
        $authentication_plugins[] = ['plugin_id' => $plugin_id] + $settings;
      }
    }

    $challenge_plugins = [];
    $challenge_plugins_value = $form_state->getValue(['challenge', 'plugins']);
    if (!empty($challenge_plugins_value)) {
      foreach ($challenge_plugins_value as $plugin_id => $settings) {
        $challenge_plugins[] = ['plugin_id' => $plugin_id] + $settings;
      }
    }

    $this->config('login_flow.settings')
      ->set('authentication', $authentication_plugins)
      ->set('challenge', $challenge_plugins)
      ->set('username_title', $form_state->getValue('username_title'))
      ->set('username_description', $form_state->getValue('username_description'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
