<?php

namespace Drupal\Tests\login_flow\Functional;

use Drupal\Core\Test\AssertMailTrait;
use Drupal\Tests\BrowserTestBase;

/**
 * Test setup for the Login Flow module.
 *
 * @group login_flow
 */
abstract class LoginFlowTestBase extends BrowserTestBase {

  use AssertMailTrait {
    getMails as drupalGetMails;
  }

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'login_flow',
    'login_flow_email_link',
    'login_flow_email_code',
  ];

  /**
   * An admin user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $adminUser;

  /**
   * Default theme for testing.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * Set to TRUE to strict check all configuration saved.
   *
   * @var bool
   */
  protected $strictConfigSchema = TRUE;

  /**
   * Performs the basic setup tasks.
   */
  public function setUp(): void {
    parent::setUp();
    // Creates role and user.
    $permissions = [
      'administer site configuration',
    ];
    $this->adminUser = $this->drupalCreateUser($permissions);

    $authentication = [
      [
        'plugin_id' => 'login_flow_email_link',
        'enabled' => FALSE,
        'weight' => 0,
      ],
      [
        'plugin_id' => 'login_flow_email_code',
        'enabled' => FALSE,
        'weight' => 5,
      ],
      [
        'plugin_id' => 'login_flow_password',
        'enabled' => TRUE,
        'weight' => 10,
      ],
    ];

    $challenge = [
      [
        'plugin_id' => 'login_flow_email_code',
        'enabled' => FALSE,
        'weight' => 0,
      ],
    ];

    $this->config('login_flow.settings')
      ->set('authentication', $authentication)
      ->set('challenge', $challenge)
      ->set('username_title', 'Username or email address')
      ->set('username_description', 'This is the test description')
      ->save();
  }

  /**
   * Enables and configures the Email Link Authentication plugin.
   */
  protected function enableEmailLinkPlugin() {
    $settings = $this->config('login_flow.settings');
    $plugins = $settings->get('authentication');
    foreach ($plugins as &$plugin) {
      if ($plugin['plugin_id'] == 'login_flow_email_link') {
        $plugin['enabled'] = TRUE;
        break;
      }
    }
    $settings->set('authentication', $plugins);
    $settings->save();

    $this->config('login_flow.authentication.login_flow_email_link')
      ->set('configuration', [
        'roles' => [],
        'domains' => [
          'example.com',
        ],
      ])
      ->save();
  }

  /**
   * Enables and configures the Email Code Authentication plugin.
   */
  protected function enableEmailCodeAuthPlugin() {
    $settings = $this->config('login_flow.settings');
    $plugins = $settings->get('authentication');
    foreach ($plugins as &$plugin) {
      if ($plugin['plugin_id'] == 'login_flow_email_code') {
        $plugin['enabled'] = TRUE;
        break;
      }
    }
    $settings->set('authentication', $plugins);
    $settings->save();

    $this->config('login_flow.authentication.login_flow_email_code')
      ->set('configuration', [
        'roles' => [],
        'domains' => [
          'example.com',
        ],
      ])
      ->save();
  }

  /**
   * Enables and configures the Email Code Challenge plugin.
   */
  protected function enableEmailCodeChallengePlugin() {
    $settings = $this->config('login_flow.settings');
    $plugins = $settings->get('challenge');
    foreach ($plugins as &$plugin) {
      if ($plugin['plugin_id'] == 'login_flow_email_code') {
        $plugin['enabled'] = TRUE;
        break;
      }
    }
    $settings->set('challenge', $plugins);
    $settings->save();

    $this->config('login_flow.challenge.login_flow_email_code')
      ->set('configuration', [
        'roles' => [],
        'domains' => [
          'example.com',
        ],
      ])
      ->save();
  }

  /**
   * Load the last email sent and get code via regex.
   */
  protected function getEmailCode() {
    $emails = $this->drupalGetMails();
    $email = end($emails);
    $body = $email['body'];
    preg_match('/[0-9]{6}/', $body, $output_array);
    $code = end($output_array);
    $this->assertNotEmpty($code);
    return $code;
  }

  /**
   * Generates a random 6 digit number.
   *
   * @return int
   *   A random 6 digit number.
   */
  protected function generateCode() {
    $min = (int) pow(10, 5);
    $max = (int) pow(10, 6) - 1;
    return mt_rand($min, $max);
  }

}
