<?php

namespace Drupal\login_flow\Controller;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Handles the configuration of the plug-ins.
 */
class LoginFlowRedirectController extends ControllerBase {

  /**
   * Injected Plugin Manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $pluginManager;

  /**
   * The Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The ModalFormExampleController constructor.
   *
   * @param \Drupal\Component\Plugin\PluginManagerInterface $plugin_manager
   *   The plugin manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(PluginManagerInterface $plugin_manager, EntityTypeManagerInterface $entity_type_manager) {
    $this->pluginManager = $plugin_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The Drupal service container.
   *
   * @return static
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.login_flow.authentication'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Redirect after logout.
   */
  public function processLogout(Request $request) {
    // Set default URL.
    $language = $this->languageManager()->getCurrentLanguage();
    $default_url = Url::fromRoute('<front>', [], ['language' => $language])->toString(TRUE);
    $response = new RedirectResponse($default_url->getGeneratedUrl());

    $plugins = $this->pluginManager->getEnabledPlugins();
    $account = $this->entityTypeManager()->getStorage('user')->load($this->currentUser()->id());

    foreach ($plugins as $plugin) {
      /** @var \Drupal\login_flow\Plugin\AuthenticationInterface $plugin */
      if ($plugin->isApplicable($account)) {
        $result = $plugin->processLogout($account);
        if ($result instanceof RedirectResponse) {
          // Remove destination otherwise response may be ignored.
          $request->query->remove('destination');
          $response = $result;
        }
        break;
      }
    }
    if ($this->currentUser()->isAuthenticated()) {
      user_logout();
    }
    return $response;
  }

}
