<?php

namespace Drupal\login_flow\Plugin;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\login_flow\Attribute\Authentication;

/**
 * Manages Login Flow Authentication plugins.
 *
 * @see hook_login_flow_authentication_info_alter()
 * @see \Drupal\login_flow\Attribute\Authentication
 * @see \Drupal\login_flow\Plugin\AuthenticationBase
 * @see \Drupal\login_flow\Plugin\AuthenticationInterface
 * @see plugin_api
 */
class AuthenticationManager extends DefaultPluginManager {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new LoginFlowManager.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler, ConfigFactoryInterface $config_factory) {
    parent::__construct('Plugin/login_flow/Authentication', $namespaces, $module_handler, 'Drupal\login_flow\Plugin\AuthenticationInterface', Authentication::class);
    $this->configFactory = $config_factory;
    $this->alterInfo('login_flow_authentication_info');
    $this->setCacheBackend($cache_backend, 'login_flow_authentication_plugins');
  }

  /**
   * Returns an array of the plugins that are enabled.
   */
  public function getEnabledPlugins() {
    $config = $this->configFactory->get('login_flow.settings');
    $plugin_settings = $config->get('authentication');
    uasort($plugin_settings, 'Drupal\Component\Utility\SortArray::sortByWeightElement');

    $plugins = [];

    foreach ($plugin_settings as $plugin) {
      if ($plugin['enabled']) {
        $plugin_id = $plugin['plugin_id'];
        $config_object = $this->configFactory->get('login_flow.authentication.' . str_replace(':', '_', $plugin_id));
        $plugin_config = (empty($config_object->get('configuration'))) ? [] : $config_object->get('configuration');
        $plugins[$plugin_id] = $this->createInstance($plugin_id, $plugin_config);
      }
    }

    return $plugins;
  }

}
