<?php

namespace Drupal\login_flow_email_code\Plugin\login_flow\Challenge;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\login_flow\Attribute\Challenge;
use Drupal\login_flow\Plugin\ChallengeBase;
use Drupal\login_flow\Plugin\ChallengeInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an email link Challenge plug-in for Login Flow.
 *
 * @see \Drupal\login_flow\Attribute\Challenge
 * @see \Drupal\login_flow\ChallengeManager
 * @see \Drupal\login_flow\Plugin\ChallengeBase
 * @see \Drupal\login_flow\Plugin\ChallengeInterface
 * @see plugin_api
 */
#[Challenge(
  id: 'login_flow_email_code',
  label: new TranslatableMarkup('Email Code'),
  description: new TranslatableMarkup('Allow someone to login using a code sent via email.'),
)]
final class EmailCode extends ChallengeBase implements ChallengeInterface {

  /**
   * Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Mail Manager Service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * The tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * Creates a new Email Code Challenge instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Mail\MailManagerInterface $mail_manager
   *   The mail manager service.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The factory for the temp store object.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    MailManagerInterface $mail_manager,
    PrivateTempStoreFactory $temp_store_factory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->mailManager = $mail_manager;
    $this->tempStoreFactory = $temp_store_factory->get('login_flow_email_code');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.mail'),
      $container->get('tempstore.private'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigurationSummary() {
    $summary = [];
    if (!empty($this->configuration['roles'])) {
      $allowed_roles = $this->configuration['roles'];
      $roles_entities = $this->entityTypeManager->getStorage('user_role')->loadMultiple($allowed_roles);
      $role_labels = [];

      foreach ($roles_entities as $role) {
        $role_labels[] = $role->label();
      }
      $summary[] = $this->t('Roles: @roles', ['@roles' => implode(', ', $role_labels)]);
    }

    if (!empty($this->configuration['domains'])) {
      $domains = $this->configuration['domains'];
      $summary[] = $this->t('Domains: @domains', ['@domains' => implode(', ', $domains)]);
    }

    if (empty($summary)) {
      $summary[] = $this->t('No restrictions');
    }

    return implode('<br>', $summary);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    // The default configuration is no roles selected.
    return [
      'roles' => [],
      'domains' => [],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm($form, FormStateInterface $form_state) {
    parent::buildConfigurationForm($form, $form_state);

    $roles = [];
    $roles_entities = $this->entityTypeManager->getStorage('user_role')->loadMultiple();

    foreach ($roles_entities as $role_id => $role) {
      $roles[$role_id] = $role->label();
    }

    $form['roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles that can login via email code'),
      '#options' => $roles,
      '#description' => $this->t('Leave empty to allow all roles.'),
      '#default_value' => empty($this->configuration['roles']) ? [] : $this->configuration['roles'],
    ];

    $form['domains'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed email domains'),
      '#description' => $this->t('One domain per line. Leave empty to allow all domains.'),
      '#default_value' => empty($this->configuration['domains']) ? '' : implode("\n", $this->configuration['domains']),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $allowed_roles = array_filter($form_state->getValue('roles'));
    $this->configuration['roles'] = array_keys($allowed_roles);
    $this->configuration['domains'] = array_filter(explode("\n", $form_state->getValue('domains')));
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(UserInterface $account, ?FormStateInterface $form_state = NULL) {
    if (!empty($this->configuration['roles'])) {
      $account_roles = $account->getRoles();
      $allowed_roles = $this->configuration['roles'];
      $intersection = array_intersect($account_roles, $allowed_roles);

      if (!empty($intersection)) {
        return TRUE;
      }
    }

    if (!empty($this->configuration['domains'])) {
      $email = ($account->isAnonymous()) ? $form_state->getValue('name') : $account->getEmail();
      $email_parts = explode('@', $email);
      $domain = end($email_parts);

      if (in_array($domain, $this->configuration['domains'])) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormElements(UserInterface $account, FormStateInterface $form_state) {
    if (!$account->isAnonymous()) {
      $code = $this->generateCode();
      $this->tempStoreFactory->set('login_flow_email_code_created', time());
      $this->tempStoreFactory->set('login_flow_email_code_number', $code);

      if ($to = $account->getEmail()) {
        $this->mailManager->mail('login_flow_email_code', 'login', $to, $account->getPreferredLangcode(), [
          'user' => $account,
          'login_flow_email_code' => $code,
        ]);
      }
    }

    $this->messenger()->addStatus($this->t('Please check your email for your code to log in.'));

    return [
      'login_flow_email_code' => [
        '#type' => 'textfield',
        '#required' => TRUE,
        '#title' => $this->t('Email code'),
        '#weight' => 10,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateChallenge(UserInterface $account, FormStateInterface $form_state) {
    $code = $form_state->getValue('login_flow_email_code');

    if (empty($code)) {
      $form_state->setErrorByName('login_flow_email_code', $this->t('Email code is required.'));
    }
    else {
      $timestamp = $this->tempStoreFactory->get('login_flow_email_code_created');
      if (time() - (int) $timestamp < 900
      && $code == $this->tempStoreFactory->get('login_flow_email_code_number')
      && is_numeric($code)) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Generates a random 6 digit number.
   *
   * @return int
   *   A random 6 digit number.
   */
  protected function generateCode() {
    $min = (int) pow(10, 5);
    $max = (int) pow(10, 6) - 1;
    return mt_rand($min, $max);
  }

}
