<?php

namespace Drupal\login_flow\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form used to display plugin settings.
 */
class LoginFlowAuthenticationSettingsForm extends ConfigFormBase {

  /**
   * Injected Plugin Manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $pluginManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The Login Flow plugin being configured.
   *
   * @var \Drupal\login_flow\Plugin\LoginFlowAccessRulePluginInterface
   */
  protected $plugin;

  /**
   * Constructs a LoginFlowSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $plugin_manager
   *   The Login Flow plugin manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, PluginManagerInterface $plugin_manager, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->pluginManager = $plugin_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.login_flow.authentication'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $config_object = $this->config('login_flow.authentication.' . str_replace(':', '_', $id));
    $plugin_config = (empty($config_object->get('configuration'))) ? [] : $config_object->get('configuration');
    $this->plugin = $this->pluginManager->createInstance($id, $plugin_config);

    $form = $this->plugin->buildConfigurationForm([], $form_state);

    $form['#prefix'] = '<div id="login-flow-authentication-plugin-settings-form">';
    $form['#suffix'] = '</div>';

    $form['status_messages'] = [
      '#type' => 'status_messages',
      '#weight' => -10,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['send'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#attributes' => [
        'class' => [
          'use-ajax',
        ],
      ],
      '#ajax' => [
        'callback' => [$this, 'submitModalFormAjax'],
        'event' => 'click',
      ],
    ];
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
    return $form;
  }

  /**
   * AJAX callback handler that displays any errors or a success message.
   */
  public function submitModalFormAjax(array $form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    // If there are any form errors, re-display the form.
    if ($form_state->hasAnyErrors()) {
      $response->addCommand(new ReplaceCommand('#login-flow-authentication-plugin-settings-form', $form));
    }
    else {
      $response->addCommand(new HtmlCommand('.login-flow-authentication-plugin-summary.' . str_replace('_', '-', $this->plugin->getPluginId()), $this->plugin->getConfigurationSummary()));
      $response->addCommand(new CloseModalDialogCommand());
    }

    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $this->plugin->validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->plugin->submitConfigurationForm($form, $form_state);
    $plugin_id = str_replace(':', '_', $this->plugin->getPluginId());

    $this->config('login_flow.authentication.' . $plugin_id)
      ->set('configuration', $this->plugin->getConfiguration())
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'login_flow_authentication_settings';
  }

  /**
   * Gets the configuration names that will be editable.
   *
   * @return array
   *   An array of configuration object names that are editable if called in
   *   conjunction with the trait's config() method.
   */
  protected function getEditableConfigNames() {
    $config_names = [];
    $definitions = $this->pluginManager->getDefinitions();

    foreach ($definitions as $plugin_id => $definition) {
      $config_names[] = 'login_flow.authentication.' . str_replace(':', '_', $plugin_id);
    }
    return $config_names;
  }

}
