<?php

namespace Drupal\login_flow_email_link\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Facilitates authentication by login link.
 */
class LoginFlowEmailLinkController extends ControllerBase {

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs an AuthenticationController object.
   *
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(TimeInterface $time) {
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('datetime.time'),
    );
  }

  /**
   * Attempt to authenticate a user using the specified parameters.
   *
   * This controller must always return a redirect response in order to avoid
   * disclosing a login link in the user agent's referrer headers.
   *
   * @param int $uid
   *   User ID of the user requesting reset.
   * @param int $timestamp
   *   The current timestamp.
   * @param string $hash
   *   Login link hash.
   */
  public function authenticate($uid, $timestamp, $hash) {
    /** @var \Drupal\user\UserInterface|null */
    $target_user = $this->entityTypeManager()->getStorage('user')->load($uid);
    $current_time = $this->time->getRequestTime();

    // Checks that the user agent is not already logged in.
    if ($this->currentUser()->isAuthenticated()) {
      $this->messenger()->addError($this->t('You must log out before you can attempt to log in as another user.'));
      return $this->redirect('<front>');
    }

    // Checks that the target user exists and isn't blocked.
    if (!$target_user || !$target_user->isActive()) {
      $this->messenger()->addError($this->t('The one-time login link you clicked is invalid.'));
      return $this->redirect('<front>');
    }

    // Checks that the link has not expired (not enforced for new users).
    if ($target_user->getLastLoginTime() && $current_time - $timestamp > 86400) {
      $this->messenger()->addError($this->t('You have tried to use a one-time login link that has expired.'));
      return $this->redirect('user.login');
    }

    // Checks that all of the following are true:
    // - The target user is a regular account.
    // - The target user has not logged in after the link was generated.
    // - The link was generated in the past.
    // - The hash matches the expected value.
    if (!$target_user->isAuthenticated() || $timestamp < $target_user->getLastLoginTime() || $timestamp > $current_time || !\hash_equals($hash, \user_pass_rehash($target_user, $timestamp))) {
      $this->messenger()->addError($this->t('You have tried to use a one-time login link that has either been used or is no longer valid.'));
      return $this->redirect('user.login');
    }

    \user_login_finalize($target_user);
    $this->messenger()->addStatus($this->t('You have been logged in successfully.'));
    return $this->redirect('<front>');
  }

}
