<?php

namespace Drupal\Tests\login_flow\Functional;

use Drupal\Core\Url;

/**
 * Test the changes to the login form.
 *
 * @group login_flow
 */
class LoginFlowFormTest extends LoginFlowTestBase {

  /**
   * An example.com user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $exampleUser;

  /**
   * The private temp store.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStore
   */
  protected $tempStore;

  /**
   * Performs the basic setup tasks.
   */
  public function setUp(): void {
    parent::setUp();
    $this->exampleUser = $this->drupalCreateUser([], 'john', FALSE, ['mail' => 'john@example.com', 'pass' => 'P4ssw0rd']);
    $this->tempStore = $this->container->get('tempstore.private')->get('login_flow_email_code');
  }

  /**
   * Test login form in default mode (should be the same).
   */
  public function testDefaultLoginForm() {
    $this->drupalGet(Url::fromRoute('user.login'));
    $this->assertSession()->pageTextContains('Username or email address');
    $this->assertSession()->pageTextContains('Password');
    $this->assertSession()->fieldExists('edit-pass');
    $this->assertSession()->elementAttributeContains('css', '#edit-submit', 'value', 'Log in');
  }

  /**
   * Test login form Email Link.
   */
  public function testEmailLinkLoginForm() {
    $this->enableEmailLinkPlugin();
    $this->resetAll();
    // After enabling the email link plugin, the password field should be gone.
    $this->drupalGet(Url::fromRoute('user.login'));
    $this->assertSession()->fieldNotExists('edit-pass');
    $this->assertSession()->elementAttributeContains('css', '#edit-submit', 'value', 'Next >');

    // Try and login as user with email link.
    $page = $this->getSession()->getPage();
    $page->fillField('name', 'john@example.com');
    $page->pressButton('Next >');

    $this->assertSession()->statusMessageContains("If john@example.com is a valid account, an email was sent with a one-time login link.");
  }

  /**
   * Test login form Email Code.
   */
  public function testEmailCodeAuthLoginForm() {
    $this->enableEmailCodeAuthPlugin();
    $this->resetAll();
    // After enabling the email code plugin, the password field should be gone.
    $this->drupalGet(Url::fromRoute('user.login'));
    $this->assertSession()->fieldNotExists('edit-pass');
    $this->assertSession()->elementAttributeContains('css', '#edit-submit', 'value', 'Next >');

    // Try and login as user with email code.
    $page = $this->getSession()->getPage();
    $page->fillField('name', 'john@example.com');
    $page->pressButton('Next >');
    $this->assertSession()->fieldExists('login_flow_email_code');
    $this->assertSession()->statusMessageContains('Please check your email for your code to log in.');
    $random_code = $this->generateCode();
    $page->fillField('login_flow_email_code', $random_code);
    $page->pressButton('Next >');
    $this->assertSession()->statusMessageContains('Your authentication information is incorrect.', 'error');
    $code = $this->getEmailCode();
    $this->assertIsNumeric($code, 'Code is a number');
    $page->fillField('login_flow_email_code', $code);
    $page->pressButton('Next >');
    // Check if user profile page loads.
    $this->drupalGet(Url::fromRoute('entity.user.canonical', ['user' => $this->exampleUser->id()]));
    $this->assertSession()->pageTextContains($this->exampleUser->getAccountName());
  }

  /**
   * Test login form Email Code Challenge.
   */
  public function testEmailCodeChallengeLoginForm() {
    $this->enableEmailCodeChallengePlugin();
    $this->resetAll();
    // With just the challenge plugin, the password field should remain.
    $this->drupalGet(Url::fromRoute('user.login'));
    $this->assertSession()->fieldExists('edit-pass');
    $this->assertSession()->elementAttributeContains('css', '#edit-submit', 'value', 'Log in');

    // Try and login as user with email code.
    $page = $this->getSession()->getPage();
    $page->fillField('name', 'john@example.com');
    $page->fillField('pass', 'P4ssw0rd');
    $page->pressButton('Log in');
    $this->assertSession()->fieldExists('login_flow_email_code');
    $this->assertSession()->statusMessageContains('Please check your email for your code to log in.');
    // Make sure wrong code doesn't give us access.
    $random_code = $this->generateCode();
    $page->fillField('pass', 'P4ssw0rd');
    $page->fillField('login_flow_email_code', $random_code);
    $page->pressButton('Log in');
    $this->assertSession()->statusMessageContains('Your authentication information is incorrect.', 'error');
    // Get real code and try and login.
    $code = $this->getEmailCode();
    $this->assertIsNumeric($code, 'Code is a number');
    // Password needs to be provided again due to page reload (no JS).
    $page->fillField('pass', 'P4ssw0rd');
    $page->fillField('login_flow_email_code', $code);
    $page->pressButton('Log in');
    // Check if user profile page loads.
    $this->drupalGet(Url::fromRoute('entity.user.canonical', ['user' => $this->exampleUser->id()]));
    $this->assertSession()->pageTextContains($this->exampleUser->getAccountName());
  }

  /**
   * Make sure changes to user edit form are correct.
   */
  public function testUserEditForm() {
    $this->drupalLogin($this->exampleUser);
    $this->enableEmailCodeAuthPlugin();
    $this->resetAll();
    // After enabling the email code plugin, the password field should be gone.
    $this->drupalGet(Url::fromRoute('entity.user.edit_form', ['user' => $this->exampleUser->id()]));
    $this->assertSession()->fieldNotExists('edit-pass-pass1');
    $this->assertSession()->fieldDisabled('edit-mail');
  }

}
