<?php

declare(strict_types=1);

namespace Drupal\user\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\WorkspaceSafeFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Provides a confirmation form for user logout.
 */
class LoginFlowUserLogoutConfirm extends ConfirmFormBase implements WorkspaceSafeFormInterface {

  /**
   * Authentication Plugin Manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected $authenticationPluginManager;

  /**
   * The Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a LoginFlowSettingsForm object.
   *
   * @param \Drupal\Component\Plugin\PluginManagerInterface $authentication_plugin_manager
   *   The Login Flow Authentication plugin manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(PluginManagerInterface $authentication_plugin_manager, EntityTypeManagerInterface $entity_type_manager) {
    $this->authenticationPluginManager = $authentication_plugin_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.login_flow.authentication'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Log out');
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to log out?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('<front>');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'user_logout_confirm';
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    $plugins = $this->authenticationPluginManager->getEnabledPlugins();
    $account = $this->entityTypeManager->getStorage('user')->load($this->currentUser()->id());
    $form_state->setRedirect('<front>');

    foreach ($plugins as $plugin) {
      /** @var \Drupal\login_flow\Plugin\AuthenticationInterface $plugin */
      if ($plugin->isApplicable($account)) {
        $result = $plugin->processLogout($account);
        if ($result instanceof RedirectResponse) {
          $form_state->setResponse($result);
        }
        break;
      }
    }
    if ($this->currentUser()->isAuthenticated()) {
      user_logout();
    }
  }

}
