<?php

namespace Drupal\login_gov\Plugin\OpenIDConnectClient;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\GeneratedUrl;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\key\Entity\Key;
use Drupal\login_gov\Exception\LoginGovConfigException;
use Drupal\openid_connect\Plugin\OpenIDConnectClientBase;
use Firebase\JWT\JWK;
use Firebase\JWT\JWT;

/**
 * Login.gov OpenID Connect client.
 *
 * Implements OpenID Connect Client plugin for Login.gov.
 *
 * @OpenIDConnectClient(
 *   id = "login_gov",
 *   label = @Translation("Login.Gov")
 * )
 */
class OpenIDConnectLoginGovClient extends OpenIDConnectClientBase {

  /**
   * A list of data fields available on login.gov.
   *
   * @var array
   */
  protected static $userinfoFields = [
    'all_emails' => 'All emails',
    'given_name' => 'First name',
    'family_name' => 'Last name',
    'address' => 'Address',
    'phone' => 'Phone',
    'birthdate' => 'Date of birth',
    'social_security_number' => 'Social security number',
    'verified_at' => 'Verification timestamp',
    'x509' => 'x509',
    'x509_subject' => 'x509 Subject',
    'x509_presented' => 'x509 Presented',
  ];

  /**
   * A list of fields we always request from the site.
   *
   * @var array
   */
  protected static $alwaysFetchFields = [
    'sub' => 'UUID',
    'email' => 'Email',
    'ial' => 'Identity Assurance Level',
    'aal' => 'Authenticator Assurance Level',
  ];

  /**
   * A mapping of userinfo fields to the scopes required to receive them.
   *
   * @var array
   */
  protected static $fieldToScopeMap = [
    'sub' => 'openid',
    'email' => 'email',
    'all_emails' => 'all_emails',
    'ial' => 'openid',
    'aal' => 'openid',
    'given_name' => 'profile:name',
    'family_name' => 'profile:name',
    'address' => 'address',
    'phone' => 'phone',
    'birthdate' => 'profile:birthdate',
    'social_security_number' => 'social_security_number',
    'verified_at' => 'profile:verified_at',
    'x509' => 'x509',
    'x509_subject' => 'x509:subject',
    'x509_presented' => 'x509:presented',
    'x509_issuer' => 'x509:issuer',
  ];

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'client_id' => '',
      'ial_level' => 'verified',
      'aal_level' => 'phishing_resistant',
      'sandbox_mode' => TRUE,
      'userinfo_fields' => [],
      'key_private_key' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['client_id'] = [
      '#title' => $this->t('Client ID'),
      '#type' => 'textfield',
      '#default_value' => $this->configuration['client_id'],
      '#required' => TRUE,
      '#description' => $this->t('The client ID is called "Issuer" in login.gov, and looks like urn:gov:gsa:openidconnect.profiles:sp:sso:<em>agency</em>:<em>application</em>'),
    ];

    $form['sandbox_mode'] = [
      '#title' => $this->t('Sandbox Mode'),
      '#type' => 'checkbox',
      '#description' => $this->t('Check here to use the identitysandbox.gov test environment.'),
      '#default_value' => $this->configuration['sandbox_mode'],
    ];

    // These two will determine the acr_values options.
    $form['ial_level'] = [
      '#title' => $this->t('Identity Assurance Level'),
      '#type' => 'radios',
      '#options' => [
        'auth-only' => $this->t('Authentication only'),
        'verified' => $this->t('Verified, no facial matching'),
        'verified-facial-match-preferred' => $this->t('Verified, facial match preferred'),
        'verified-facial-match-required' => $this->t('Verified, facial match required'),
      ],
      '#default_value' => $this->configuration['ial_level'],
      '#description' => $this->t('See the @login_gov_documentation for more details on the options and IAL/AAL compliance.', [
        '@login_gov_documentation' => Link::fromTextAndUrl($this->t('Login.gov documentation'), Url::fromUri('https://developers.login.gov/oidc/authorization/#service_level'))->toString(),
      ]),
    ];

    $form['aal_level'] = [
      '#title' => $this->t('Authentication Assurance Level'),
      '#type' => 'radios',
      '#options' => [
        'duo' => $this->t('Require basic two-factor authentication'),
        'separate' => $this->t('Require separate second factor (i.e., not a remembered device.)'),
        'phishing_resistant' => $this->t('Require a phishing resistant second factor (WebAuthn or PIV/CAC)'),
        'require_hspd12' => $this->t('Require an HSPD12 credential (requires PIV/CAC)'),
      ],
      '#default_value' => $this->configuration['aal_level'],
      '#description' => $this->t('See the @login_gov_documentation for more details on the options and IAL/AAL compliance.', [
        '@login_gov_documentation' => Link::fromTextAndUrl($this->t('Login.gov documentation'), Url::fromUri('https://developers.login.gov/oidc/authorization/#aal_values'))->toString(),
      ]),
    ];

    $form['userinfo_fields'] = [
      '#title' => $this->t('User fields'),
      '#type' => 'select',
      '#multiple' => TRUE,
      '#options' => static::$userinfoFields,
      '#description' => $this->t('List of fields to fetch, which will translate to the required scopes. Some fields require a minimum Authentication Assurance Level. See the @login_gov_documentation for more details. The Email and UUID (sub) fields are always fetched.', [
        '@login_gov_documentation' => Link::fromTextAndUrl($this->t('Login.gov documentation'), Url::fromUri('https://developers.login.gov/attributes/'))->toString(),
      ]),
      '#default_value' => $this->configuration['userinfo_fields'],
    ];

    $form['key_private_key'] = [
      '#title' => $this->t('Key from Key'),
      '#type' => 'key_select',
      '#default_value' => $this->configuration['key_private_key'],
      '#key_filters' => ['type' => ['asymmetric_private']],
      '#description' => ' ' . $this->t('A Private key managed by the @key_module.', ['@key_module' => Link::fromTextAndUrl($this->t('Key module'), Url::fromRoute('entity.key.collection'))->toString()]),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoints(): array {
    return $this->configuration['sandbox_mode'] ? [
      'authorization' => 'https://idp.int.identitysandbox.gov/openid_connect/authorize',
      'token' => 'https://idp.int.identitysandbox.gov/api/openid_connect/token',
      'userinfo' => 'https://idp.int.identitysandbox.gov/api/openid_connect/userinfo',
      'end_session' => 'https://idp.int.identitysandbox.gov/openid_connect/logout',
      'certs' => 'https://idp.int.identitysandbox.gov/api/openid_connect/certs',
    ] :
    [
      'authorization' => 'https://secure.login.gov/openid_connect/authorize',
      'token' => 'https://secure.login.gov/api/openid_connect/token',
      'userinfo' => 'https://secure.login.gov/api/openid_connect/userinfo',
      'end_session' => 'https://secure.login.gov/openid_connect/logout',
      'certs' => 'https://secure.login.gov/api/openid_connect/certs',
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function getRequestOptions(string $authorization_code, string $redirect_uri): array {
    $endpoints = $this->getEndpoints();

    // Build the client assertion.
    // See https://developers.login.gov/oidc/#token
    $client_assertion_payload = [
      'iss' => $this->configuration['client_id'],
      'sub' => $this->configuration['client_id'],
      'aud' => $endpoints['token'],
      'jti' => $this->generateNonce(),
      'exp' => time() + 300,
    ];
    // Add the client assertion to the list of options.
    $options = [
      'client_assertion' => $this->signJwtPayload($client_assertion_payload),
      'client_assertion_type' => 'urn:ietf:params:oauth:client-assertion-type:jwt-bearer',
      'code' => $authorization_code,
      'grant_type' => 'authorization_code',
    ];
    return [
      'form_params' => $options,
      'headers' => [
        'Accept' => 'application/json',
      ],
    ];
  }

  /**
   * Sign the JWT.
   *
   * @param array $payload
   *   An array of key-value pairs.
   *
   * @return string
   *   The signed JWT.
   */
  public function signJwtPayload(array $payload): string {
    return JWT::encode($payload, $this->getPrivateKey(), 'RS256');
  }

  /**
   * Return the private key for signing the JWTs.
   *
   * @return string
   *   The private key in PEM format.
   */
  protected function getPrivateKey(): ?string {
    $key = Key::load($this->configuration['key_private_key']);
    // Return the key's KeyValue, or fall back to the old configuration if there
    // is no Key.
    return $key ? $key->getKeyValue() : $this->configuration['private_key'];
  }

  /**
   * Get login.gov's public signing key.
   *
   * @return array|null
   *   A list of public keys.
   */
  protected function getPeerPublicKeys(): ?array {
    $endpoints = $this->getEndpoints();
    $keys_json = $this->httpClient->get($endpoints['certs'])->getBody()->getContents();
    $keys = Json::decode($keys_json);
    return JWK::parseKeySet($keys);
  }

  /**
   * Generate a one-time use code word, a nonce.
   *
   * @param int $length
   *   The length of the nonce.
   *
   * @return string
   *   The nonce.
   */
  protected function generateNonce(int $length = 26): string {
    return substr(Crypt::randomBytesBase64($length), 0, $length);
  }

  /**
   * Generate the acr_values portion of the URL options.
   *
   * @return string
   *   The Authentication Context Class Reference value.
   */
  protected function generateAcrValue(): string {
    $valid_service_levels = [
      'auth-only',
      'verified',
      'verified-facial-match-preferred',
      'verified-facial-match-required',
    ];

    $aal_map = [
      'duo' => 'urn:gov:gsa:ac:classes:sp:PasswordProtectedTransport:duo',
      'separate' => 'http://idmanagement.gov/ns/assurance/aal/2',
      'phishing_resistant' => 'http://idmanagement.gov/ns/assurance/aal/2?phishing_resistant=true',
      'require_hspd12' => 'http://idmanagement.gov/ns/assurance/aal/2?hspd12=true',
    ];

    // Validate before assembling the acr values.
    if (!in_array($this->configuration['ial_level'], $valid_service_levels) ||
        !in_array($this->configuration['aal_level'], array_keys($aal_map))) {
      throw new LoginGovConfigException('Bad client configuration.');
    }

    $acrs = [];
    $acrs[] = 'urn:acr.login.gov:' . $this->configuration['ial_level'];
    $acrs[] = $aal_map[$this->configuration['aal_level']];

    return implode(' ', $acrs);
  }

  /**
   * {@inheritdoc}
   */
  protected function getUrlOptions(string $scope, GeneratedUrl $redirect_uri): array {
    $options = parent::getUrlOptions($scope, $redirect_uri);

    $nonce = $this->generateNonce();
    $options['query'] += [
      'acr_values' => $this->generateAcrValue(),
      'nonce' => $nonce,
      'prompt' => 'select_account',
    ];
    $this->requestStack->getCurrentRequest()->getSession()->set('login_gov.nonce', $nonce);

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function retrieveTokens(string $authorization_code): ?array {
    $tokens = parent::retrieveTokens($authorization_code);

    // Verify the nonce is the one we sent earlier.
    if (!empty($tokens['id_token'])) {
      $keys = $this->getPeerPublicKeys();
      $decoded_tokens = JWT::decode($tokens['id_token'], $keys);
      $session_nonce = $this->requestStack->getCurrentRequest()->getSession()->get('login_gov.nonce');
      if (!empty($session_nonce) && ($decoded_tokens->nonce !== $session_nonce)) {
        return NULL;
      }
    }

    return $tokens;
  }

  /**
   * {@inheritdoc}
   */
  public function getClientScopes(): ?array {
    $fields = static::$alwaysFetchFields + ($this->configuration['userinfo_fields'] ?? []);
    return array_values(array_unique(array_intersect_key(static::$fieldToScopeMap, $fields)));
  }

  /**
   * {@inheritdoc}
   */
  public function alterLogoutRedirectionQuery(array $query): array {
    unset($query['id_token_hint']);
    $query['client_id'] = $this->configuration['client_id'];

    return $query;
  }

}
