<?php

namespace Drupal\Tests\login_switch\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Test the Login switch settings form and all module's features.
 *
 * @group login_switch
 */
class LoginSwitchTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'login_switch',
  ];

  /**
   * A user with access to the Login switch settings form.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $permissions = [
      'access administration pages',
      'administer site configuration',
    ];
    $this->adminUser = $this->drupalCreateUser($permissions);
  }

  /**
   * Test backend Login switch settings form fields, submission and features.
   */
  public function testLoginSwitch(): void {
    /** @var \Drupal\Tests\WebAssert $assert */
    $assert = $this->assertSession();

    // Define all the user routes to be tested.
    $routes = ['login', 'register', 'password'];

    // Check all the default core user routes are working as expected before
    // they are disabled with the configuration of the module.
    foreach ($routes as $key) {
      // Check the default user page is enabled with the form displayed.
      $this->drupalGet('user/' . $key);
      $assert->statusCodeEquals(200);
      $assert->fieldExists('name');
      $assert->responseHeaderDoesNotExist('X-Robots-Tag');
    }

    // Log in as an admin user to test the Login switch settings page.
    $this->drupalLogin($this->adminUser);

    // Test the 'Login switch' page form submission and fields.
    $this->drupalGet('admin/config/people/login-switch');

    $edit = [];
    // Enable all settings and set all field values.
    foreach ($routes as $key) {
      $edit += [
        $key . '_disabled' => TRUE,
        $key . '_route' => 'test/path/' . $key,
        $key . '_noindex' => TRUE,
      ];
    }
    $this->submitForm($edit, 'Save configuration');
    // Check the form is submitted without errors.
    $assert->pageTextContains('Configuration changes were successfully saved.');

    // Log the user out with the standard core logout method.
    $this->drupalLogout();

    // Check all the default core user routes have been switched to the ones
    // configured on module's settings page.
    foreach ($routes as $key) {
      // Check the default user page is disabled and returns a 404 status code.
      $this->drupalGet('user/' . $key);
      $assert->statusCodeEquals(404);

      // Check the user page is accessible with the configured route.
      $this->drupalGet('test/path/' . $key);
      $assert->statusCodeEquals(200);
      // Check the 'name' field, common to user forms, is displayed as expected.
      $assert->fieldExists('name');

      // Check the response header contains 'noindex'.
      $assert->responseHeaderEquals('X-Robots-Tag', 'noindex');
    }

  }

}
