<?php

namespace Drupal\loginnotification\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\SessionManager;
use Drupal\user\UserStorageInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * LoginNotificationController Controller definition.
 */
class LoginNotificationController extends ControllerBase {

  /**
   * The session manager.
   *
   * @var \Drupal\Core\Session\SessionManager
   */
  protected $serviceSession;

  /**
   * The user storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected $userStorage;

  /**
   * A logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a LoginNotificationController object.
   *
   * @param \Drupal\Core\Session\SessionManager $sessionManager
   *   The session manager.
   * @param \Drupal\user\UserStorageInterface $user_storage
   *   The user storage.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   */
  public function __construct(SessionManager $sessionManager, UserStorageInterface $user_storage, LoggerInterface $logger) {
    $this->serviceSession = $sessionManager;
    $this->userStorage = $user_storage;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('session_manager'),
      $container->get('entity_type.manager')->getStorage('user'),
      $container->get('logger.factory')->get('loginnotification')
    );
  }

  /**
   * Destroy all session for the specific user.
   *
   * @param int $uid
   *   User ID of the user requesting reset.
   * @param int $timestamp
   *   The login timestamp.
   * @param string $hash
   *   Login link hash.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   The redirect response.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   If the $uid is for a blocked user or invalid user ID.
   */
  public function logoutUser($uid, $timestamp, $hash) {
    /** @var \Drupal\user\UserInterface $user */
    $user = $this->userStorage->load($uid);

    // Verify that the user exists and is active.
    if ($user === NULL || !$user->isActive()) {
      // Blocked or invalid user ID, deny access.
      throw new AccessDeniedHttpException();
    }
    if (hash_equals($hash, loginnotification_user_hash($user, $timestamp))) {
      $this->logger->notice('User %name was logged out.', ['%name' => $user->getDisplayName()]);
      $this->messenger()->addStatus($this->t('You have just used login notification link to logout %name from the system.', ['%name' => $user->getDisplayName()]));
      $this->serviceSession->delete($user->id());
    }
    return $this->redirect('<front>');
  }

}
