<?php

namespace Drupal\logo_image_enhanced\Render;

use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\image\Entity\ImageStyle;

/**
 * Provides pre-render callback for the site logo.
 */
class LogoPreRenderCallback implements TrustedCallbackInterface {

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['preRender'];
  }

  /**
   * Pre-render callback to apply image style and attributes to logo.
   *
   * @param array $build
   *   The render array.
   *
   * @return array
   *   The modified render array.
   */
  public static function preRender(array $build): array {
    if (!isset($build['content']['site_logo'])) {
      return $build;
    }

    $config = \Drupal::config('logo_image_enhanced.settings');
    $site_name = \Drupal::config('system.site')->get('name');

    $logo_style = $config->get('logo_style') ?: '_none';
    $logo_alt = $config->get('logo_alt') ?: $site_name;
    $logo_title = $config->get('logo_title') ?: $site_name;

    // Get logo URI from multiple sources.
    $logo_uri = self::getLogoUri($build, $config);

    // Apply image style if configured.
    if ($logo_style !== '_none' && !empty($logo_uri)) {
      $style = ImageStyle::load($logo_style);

      if ($style && file_exists($logo_uri)) {
        $styled_url = $style->buildUrl($logo_uri);

        // Try different render array structures.
        if (isset($build['content']['site_logo']['#uri'])) {
          $build['content']['site_logo']['#uri'] = $styled_url;
        }
        elseif (isset($build['content']['site_logo']['#markup'])) {
          // Replace src in markup.
          $build['content']['site_logo']['#markup'] = preg_replace(
            '/src="[^"]*"/',
            'src="' . $styled_url . '"',
            $build['content']['site_logo']['#markup']
          );
        }

        // Add dimensions.
        $dimensions = self::getDimensionsFromStyle($style, $logo_uri);
        if ($dimensions) {
          $build['content']['site_logo']['#width'] = $dimensions['width'];
          $build['content']['site_logo']['#height'] = $dimensions['height'];
          
          if (isset($build['content']['site_logo']['#attributes'])) {
            $build['content']['site_logo']['#attributes']['width'] = $dimensions['width'];
            $build['content']['site_logo']['#attributes']['height'] = $dimensions['height'];
          }
        }
      }
    }

    // Apply accessibility attributes.
    if (!isset($build['content']['site_logo']['#attributes'])) {
      $build['content']['site_logo']['#attributes'] = [];
    }
    $build['content']['site_logo']['#attributes']['alt'] = $logo_alt;
    $build['content']['site_logo']['#attributes']['title'] = $logo_title;

    // Support for #theme => 'image'.
    if (isset($build['content']['site_logo']['#theme']) &&
        $build['content']['site_logo']['#theme'] === 'image') {
      $build['content']['site_logo']['#alt'] = $logo_alt;
      $build['content']['site_logo']['#title'] = $logo_title;
    }

    // Add cache metadata.
    $build['#cache']['tags'][] = 'config:logo_image_enhanced.settings';

    return $build;
  }

  /**
   * Get logo URI from various sources.
   *
   * @param array $build
   *   The render array.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   *
   * @return string
   *   The logo URI or empty string.
   */
  protected static function getLogoUri(array $build, $config): string {
    // 1. Try from render array #uri.
    if (isset($build['content']['site_logo']['#uri'])) {
      $uri = $build['content']['site_logo']['#uri'];
      // Convert URL to URI if needed.
      if (strpos($uri, '://') !== FALSE && strpos($uri, 'public://') === FALSE) {
        // It's a URL, try to convert to URI.
        $uri = self::urlToUri($uri);
      }
      if (!empty($uri) && file_exists($uri)) {
        return $uri;
      }
    }

    // 2. Try from module config.
    $logo_path = $config->get('logo_path');
    if (!empty($logo_path)) {
      $uri = self::normalizeUri($logo_path);
      if (file_exists($uri)) {
        return $uri;
      }
    }

    // 3. Try from theme settings.
    $theme_logo = theme_get_setting('logo.path');
    if (!empty($theme_logo)) {
      $uri = self::normalizeUri($theme_logo);
      if (file_exists($uri)) {
        return $uri;
      }
    }

    // 4. Try from global theme settings.
    $global_config = \Drupal::config('system.theme.global');
    $global_logo = $global_config->get('logo.path');
    if (!empty($global_logo)) {
      $uri = self::normalizeUri($global_logo);
      if (file_exists($uri)) {
        return $uri;
      }
    }

    // 5. Try from active theme config.
    $active_theme = \Drupal::theme()->getActiveTheme()->getName();
    $theme_config = \Drupal::config($active_theme . '.settings');
    $active_logo = $theme_config->get('logo.path');
    if (!empty($active_logo)) {
      $uri = self::normalizeUri($active_logo);
      if (file_exists($uri)) {
        return $uri;
      }
    }

    return '';
  }

  /**
   * Convert a URL to a stream URI.
   *
   * @param string $url
   *   The URL.
   *
   * @return string
   *   The URI or empty string.
   */
  protected static function urlToUri(string $url): string {
    // Extract path from URL.
    $parsed = parse_url($url);
    $path = $parsed['path'] ?? '';

    // Check if it's in the files directory.
    $public_path = \Drupal::service('stream_wrapper_manager')->getViaScheme('public')->getDirectoryPath();
    
    if (strpos($path, '/' . $public_path . '/') !== FALSE) {
      $relative = substr($path, strpos($path, '/' . $public_path . '/') + strlen('/' . $public_path . '/'));
      return 'public://' . $relative;
    }

    return '';
  }

  /**
   * Normalize a path to a stream wrapper URI.
   *
   * @param string $path
   *   The file path.
   *
   * @return string
   *   The normalized URI.
   */
  protected static function normalizeUri(string $path): string {
    // Already a URI.
    if (strpos($path, '://') !== FALSE) {
      return $path;
    }

    // Absolute path starting with /.
    if (strpos($path, '/') === 0) {
      // Check if it's in the public files directory.
      $public_path = \Drupal::service('stream_wrapper_manager')->getViaScheme('public')->getDirectoryPath();
      if (strpos($path, '/' . $public_path . '/') === 0) {
        return 'public://' . substr($path, strlen('/' . $public_path . '/'));
      }
      // Check if it's relative to DRUPAL_ROOT.
      $full_path = DRUPAL_ROOT . $path;
      if (file_exists($full_path)) {
        // Try to convert to public://.
        $real_public = \Drupal::service('file_system')->realpath('public://');
        $real_full = realpath($full_path);
        if ($real_full && strpos($real_full, $real_public) === 0) {
          return 'public://' . substr($real_full, strlen($real_public) + 1);
        }
      }
      return $path;
    }

    // Relative path - assume public://.
    return 'public://' . ltrim($path, '/');
  }

  /**
   * Get dimensions from an image style.
   *
   * @param \Drupal\image\Entity\ImageStyle $style
   *   The image style.
   * @param string $uri
   *   The image URI.
   *
   * @return array|null
   *   Array with 'width' and 'height' or NULL.
   */
  protected static function getDimensionsFromStyle(ImageStyle $style, string $uri): ?array {
    $width = NULL;
    $height = NULL;

    // Try to get dimensions from style effects.
    foreach ($style->getEffects() as $effect) {
      $effect_config = $effect->getConfiguration();
      $resize_effects = ['image_resize', 'image_scale', 'image_scale_and_crop'];

      if (in_array($effect_config['id'], $resize_effects, TRUE)) {
        $width = $effect_config['data']['width'] ?? $width;
        $height = $effect_config['data']['height'] ?? $height;
      }
    }

    // Fallback to original image dimensions.
    if ($width === NULL || $height === NULL) {
      $image = \Drupal::service('image.factory')->get($uri);
      if ($image->isValid()) {
        $width = $width ?? $image->getWidth();
        $height = $height ?? $image->getHeight();
      }
    }

    if ($width && $height) {
      return ['width' => (int) $width, 'height' => (int) $height];
    }

    return NULL;
  }

}