<?php

namespace Drupal\lupus_decoupled_views\Plugin\views\display;

use Drupal\views\Element\View;
use Drupal\views\Plugin\Block\ViewsBlock;
use Drupal\views\Plugin\views\display\Block;

/**
 * The plugin that handles a block.
 *
 * @ingroup views_display_plugins
 *
 * @ViewsDisplay(
 *   id = "custom_elements_block",
 *   title = @Translation("Custom Elements Block"),
 *   help = @Translation("Display the view as a custom elements block."),
 *   theme = "views_view",
 *   register_theme = FALSE,
 *   uses_hook_block = TRUE,
 *   contextual_links_locations = {"block"},
 *   admin = @Translation("Block")
 * )
 *
 * @see \Drupal\views\Plugin\Block\ViewsBlock
 * @see \Drupal\views\Plugin\Derivative\ViewsBlock
 */
class CustomElementsBlock extends Block {

  use CustomElementsViewsDisplayTrait;

  /**
   * {@inheritdoc}
   */
  public function preBlockBuild(ViewsBlock $block) {
    parent::preBlockBuild($block);

    $config = $block->getConfiguration();
    // Make sure the block title-related configuration is respected.
    if (empty($config['label_display'])) {
      $this->view->setTitle('');
    }
    elseif (!empty($config['views_label'])) {
      // Inject the overridden block title into the view. This is normally done
      // by ViewsBlock::build() but it only sets it after we generated the CE.
      // @see static::buildRenderable()
      $this->view->setTitle($config['views_label']);
    }
    // Else the default title of the view is used.
  }

  /**
   * {@inheritdoc}
   *
   * @see \Drupal\views\Plugin\Block\ViewsBlock::build()
   */
  public function buildRenderable(array $args = [], $cache = TRUE) {
    // Called from ViewsBlock::build() when calling $view->buildRenderable().
    // We already take care of pre-rendering the View here, instead of having
    // ViewsBlock::build() do it, such that we can control the process and
    // make sure we provide the custom-element as the top-level element that is
    // provided by the block.
    $render = parent::buildRenderable($args, $cache);

    // Restore the (possibly overridden) title from the view. This needs to be
    // done before the view is rendered, as during view-rendering a title
    // override would be lost.
    // @see static::preBlockBuild()
    $title = isset($this->view->build_info['title']) && $this->view->build_info['title'] === '' ? '' : $this->view->getTitle();

    // Pre-render view like ViewsBlock::build does it. It sets #pre_rendered
    // such that it will be executed only once.
    $render = View::preRenderViewElement($render);

    // Now, we can process the render-array as needed.
    // The render() method has pre-rendered into the custom element.
    if (!empty($render['view_build']['#custom_element'])) {
      $render['#custom_element'] = $render['view_build']['#custom_element'];
      $render['#custom_element']->setAttribute('title', $title);
      $render['#theme'] = 'custom_element';
    }
    return $render;
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $render = parent::render();
    $rows = $render['#rows']['rows'] ?? [];
    // Manually pre-render the element, such that we can convert results into
    // the custom element.
    $render = $this->elementPreRender($render);
    $render = $this->buildCustomElement($this->view, $render, $rows)
      ->toRenderArray();
    return $render;
  }

  /**
   * {@inheritdoc}
   */
  public function preview() {
    $element = parent::preview();
    $element['#prefix'] = '<div class="preview-section"><pre>';
    $element['#suffix'] = '</pre></div>';
    return $element;
  }

}
