<?php

namespace Drupal\lupus_decoupled_views\Plugin\views\display;

use Drupal\Core\Render\BubbleableMetadata;
use Drupal\custom_elements\CustomElement;
use Drupal\views\Plugin\views\pager\None;
use Drupal\views\Plugin\views\pager\Some;
use Drupal\views\ViewExecutable;

/**
 * Provides helpers for the custom-element views display plugins.
 *
 * @internal
 */
trait CustomElementsViewsDisplayTrait {

  /**
   * Build a custom element from a pre-rendered view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view.
   * @param array $render
   *   Pre-rendered render array of the view.
   * @param array $rows
   *   The rendered rows of the view.
   *
   * @return \Drupal\custom_elements\CustomElement
   *   Custom element.
   */
  public function buildCustomElement(ViewExecutable $view, array $render, array $rows) {
    // Remove "custom_elements_" prefix from display type.
    $display_type = strpos($view->getDisplay()
      ->getPluginId(), 'custom_elements_') === 0 ?
      substr($view->getDisplay()->getPluginId(), 16) :
      $view->getDisplay()->getPluginId();
    $custom_element = CustomElement::create('drupal-view-' . str_replace('_', '-', $view->id() . '-' . $display_type));

    $custom_element->setAttribute('title', $view->getTitle());
    $custom_element->setAttribute('view_id', $view->id());
    $custom_element->setAttribute('display_id', $view->getDisplay()->display['id']);
    if (!empty($view->args)) {
      $custom_element->setAttribute('args', $view->args);
    }

    // @todo Add header, footer areas.
    $custom_element->setAttribute('pager', !empty($view->pager) ? $this->getPaginationData($view->pager, $rows) : NULL);

    // Add the child rows, i.e. the actual views content.
    $custom_element->setSlotFromNestedElements('rows', array_filter(array_map(
      fn($row) => $row['#custom_element'] ?? NULL,
      $rows
    )));
    $custom_element->addCacheableDependency(BubbleableMetadata::createFromRenderArray($render));
    return $custom_element;
  }

  /**
   * Gets relevant data from the pager plugin that already 'built' it.
   *
   * @return array|null
   *   Returns data for the "pager" attribute.
   */
  private function getPaginationData($pager, $rows) {
    $pagination = [];
    $class = get_class($pager);
    if ($class === NULL) {
      return NULL;
    }
    if (method_exists($pager, 'getPagerTotal')) {
      $pagination['total_pages'] = $pager->getPagerTotal();
    }
    if (method_exists($pager, 'getCurrentPage')) {
      $pagination['current'] = $pager->getCurrentPage() ?? 0;
    }
    if ($pager instanceof None) {
      $pagination['items_per_page'] = $pager->getTotalItems();
    }
    elseif ($pager instanceof Some) {
      $pagination['total_items'] = count($rows);
    }
    return $pagination;
  }

}
