<?php

namespace Drupal\maestro_activepieces\Plugin\EngineTasks;


use Drupal\Core\Plugin\PluginBase;
use Drupal\maestro\MaestroEngineTaskInterface;
use Drupal\maestro\Engine\MaestroEngine;
use Drupal\maestro\MaestroTaskTrait;
use Drupal\Core\Form\FormStateInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

use Drupal\maestro\Form\MaestroExecuteInteractive;


/**
 * Maestro Activepieces Task Plugin.
 *
 * The plugin annotations below should include:
 * id: The task type ID for this task.  For Maestro tasks, this is Maestro[TaskType].
 *     So for example, the start task shipped by Maestro is MaestroStart.
 *     The Maestro End task has an id of MaestroEnd
 *     Those task IDs are what's used in the engine when a task is injected into the queue.
 *
 * @Plugin(
 *   id = "MaestroActivepiecesTask",
 *   task_description = @Translation("The Maestro Engine's Activepieces task."),
 * )
 */
class MaestroActivepiecesTask extends PluginBase implements MaestroEngineTaskInterface {

  use MaestroTaskTrait;

  /**
   * Constructor.
   */
  public function __construct($configuration = NULL) {
    if (is_array($configuration)) {
      $this->processID = $configuration[0];
      $this->queueID = $configuration[1];
    }
  }

  /**
   * {@inheritDoc}
   */
  public function isInteractive() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function shortDescription() {
    return $this->t('Activepieces Task');
  }

  /**
   * {@inheritDoc}
   */
  public function description() {
    return $this->t('Activepieces Task.');
  }

  /**
   * {@inheritDoc}
   *
   * @see \Drupal\Component\Plugin\PluginBase::getPluginId()
   */
  public function getPluginId() {
    return 'MaestroActivepiecesTask';
  }

  /**
   * {@inheritDoc}
   */
  public function getTaskColours() {
    // AI tasks will be this Dodger Blue colour.
    return '#04BDFF';
  }

  /**
   * Part of the ExecutableInterface.
   *
   * Execution of the Batch Function task will use the handler for this task as the executable function.
   * The handler must return TRUE in order for this function to be completed by the engine.
   * We simply pass the return boolean value back from the called handler to the engine for processing.
   * {@inheritdoc}.
   */
  public function execute() {
    $responseValue = NULL;
    $returnValue = TRUE;
    $returnStatus = FALSE;
    $templateMachineName = MaestroEngine::getTemplateIdFromProcessId($this->processID);
    $taskMachineName = MaestroEngine::getTaskIdFromQueueId($this->queueID);
    $task = MaestroEngine::getTemplateTaskByID($templateMachineName, $taskMachineName);
    $taskData = $task['data']['activepieces'] ?? [];
    $queueRecord = MaestroEngine::getQueueEntryById($this->queueID);
    if ($queueRecord) {
      // Perform a Guzzle call to the configured Activepieces webhook.
      $client = new Client();
      $url = $taskData['activepieces_url'] ?? NULL;

      // Generate the data output array based on the configuration.
      $data_items = $taskData['activepieces_items'] ?? [];
      $data = [];

      foreach($data_items as $data_item) {
        $data_value = '';
        if($data_item['type'] == 'process_variable') {
          // Get the process variable value
          $data_value = MaestroEngine::getProcessVariable($data_item['value'] ?? '', $this->processID);
          if($data_value === FALSE) {
            $data_value = '';
          }
        }
        else {
          $data_value = $data_item['value'] ?? '';
          // Now run this through the token service.
          $tokenService = \Drupal::token();
          $data_value = $tokenService->replace(
            $data_value,
            [
              'maestro' => 
                [
                  'task' => $task, 
                  'queueID' => $this->queueID,
                  'processID' => $this->processID,
                ]
            ]
          );
        }

        // Now, construct this data element
        $data[$data_item['key']] = $data_value;
      }

      if($url) {
        try {
          $options = [
            'headers' => [
              'Content-Type' => 'application/json',
            ],
            'body' => json_encode($data, JSON_THROW_ON_ERROR),
          ];
          // Trap the response value in the event we need it for further Maestro processing.
          // We will be able to configure the AP task to set task status based on return processing codes
          // and also dump the value of any errors into a process variable so that we can notify people in 
          // their task consoles.
          $responseValue = $client->request('POST', $url, $options);
        }
        catch (GuzzleException | \JsonException $e) {
          // @todo:  Log this exception via configuration options in the task edit.
          // Log an exception.  
          // Set a PV
          // Set task processing status
          $returnValue = FALSE; // Keep the engine from continuing for now
        }
      }
    }
    
    return $returnValue;
  }

  /**
   * {@inheritdoc}
   */
  public function getExecutableForm($modal, MaestroExecuteInteractive $parent) {

  }

  /**
   * {@inheritdoc}
   */
  public function handleExecuteSubmit(array &$form, FormStateInterface $form_state) {

  }

  /**
   * {@inheritdoc}
   */
  public function getTaskEditForm(array $task, $templateMachineName) {
    /** @var FormStateInterface $form_state */
    $form_state = $task['form_state'];
    $taskData = $task['data']['activepieces'] ?? [];
    $form_state->disableCache();
    $form = [];

    $form['activepieces_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Activepieces Webhook URL'),
      '#required' => TRUE,
      '#default_value' => $taskData['activepieces_url'] ?? '',
      '#description' => $this->t('The URL which will call your Activepieces webhook.'),
    ];

    // Dynamic listing of key/value pairs that we will send along to Activepieces.
    // This listing will be in the form of a textbox for the key, followed by a drop-down to specify what TYPE 
    // of value: process variable, token or manually_entered
    // Then a final box where you will either choose the PV, provide the token or manually_entered value

    $form['activepieces_data'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Data to send to Activepieces'),
      '#description' => $this->t('Provide the data structure to send. Provide a key and define the data type you are sending'),
      '#prefix' => '<div id="maestro-ap-data-ajax-refresh-wrapper">',
      '#suffix' => '</div>',
    ];

    $form_input = $form_state->getUserInput();
    $unique_id = uniqid();
    // Check if this is an ajax operation
    $triggering_element = $form_input['_triggering_element_name'] ?? NULL;
    if($triggering_element && $triggering_element == 'add_another_ap_data') {
      $items = $form_input['ap_data_items'];
      $items[$unique_id] = [
        'key' . $unique_id => '',
        'type' . $unique_id => 'manually_entered',
        'value' . $unique_id => '',
      ];
    }
    elseif($triggering_element && strpos($triggering_element, 'remove_') === 0) {
      $index = str_replace('remove_', '', $triggering_element);
      if ($index !== NULL) {
        $items = $form_input['ap_data_items'];
        unset($items[$index]);
        if(count($items) === 0) {
          $items[$unique_id] = [
            'key' . $unique_id => '',
            'type' . $unique_id => 'manually_entered',
            'value' . $unique_id => '',
          ];
        }
      }
    }
    elseif(isset($form_input['ap_data_items']) && is_array($form_input['ap_data_items'])) {
      $items = $form_input['ap_data_items'];
    }
    else {
      $items = $taskData['activepieces_items'] ?? [];
    }
    
    $form['activepieces_data']['ap_data_items'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'maestro-ap-data-items'],
      '#tree' => TRUE,
    ];

    $type_options = [
      'process_variable' => $this->t('Process Variable'),
      'manually_entered' => $this->t('Manual Entry and/or Tokens'),
    ];

    $variables = MaestroEngine::getTemplateVariables($templateMachineName);
    $variable_options = [
      '' => $this->t('Choose Process Variable'),
    ];
    foreach ($variables as $variableName => $arr) {
      $variable_options[$variableName] = $variableName;
    }

    // Provide the token browser top and bottom of data entry area
    if (\Drupal::moduleHandler()->moduleExists('token')) {
      $form['activepieces_data']['ap_data_items']['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['maestro'],
      ];
    }
    else {
      $form['activepieces_data']['ap_data_items']['token_tree'] = [
        '#plain_text' => $this->t('Enabling the Token module will reveal the replacable tokens.'),
      ];
    }

    foreach ($items as $i => $item) {
      $form['activepieces_data']['ap_data_items'][$i] = [
        '#type' => 'fieldset',
        '#attributes' => [
          'class' => ['maestro-ap-row',],
          'data-index' => $i,
        ],
      ];

      $form['activepieces_data']['ap_data_items'][$i]['key' . $i] = [
        '#type' => 'textfield',
        '#title' => $this->t('Key'),
        '#default_value' => $item['key'] ?? '',
      ];

      $type = $item['type'] ?? 'manually_entered';
      $form['activepieces_data']['ap_data_items'][$i]['type' . $i] = [
        '#type' => 'select',
        '#title' => $this->t('Type'),
        '#options' => $type_options,
        '#default_value' => $type,
      ];

      // Based on the Type selected above, we will show one of the following:  A textarea for tokens or manually entered values or a process variable select
      $form['activepieces_data']['ap_data_items'][$i]['manual_value' . $i] = [
        '#type' => 'textarea',
        '#title' => $this->t('Value'),
        '#default_value' => $type == 'process_variable' ? '' : ($item['value'] ?? ''),
        '#states' => [
          'visible' => [
            ':input[name="ap_data_items[' . $i . '][type' . $i . ']"]' => ['value' => 'manually_entered'],
          ],
        ],
      ];

      $form['activepieces_data']['ap_data_items'][$i]['pv_value' . $i] = [
        '#type' => 'select',
        '#options' => $variable_options,
        '#title' => $this->t('Value'),
        '#default_value' => $type == 'process_variable' ? ($item['value'] ?? '') : '',
        // use state to show hide this based on type
        '#states' => [
          'visible' => [
            ':input[name="ap_data_items[' . $i . '][type' . $i . ']"]' => ['value' => 'process_variable'],
          ],
        ],
      ];

      $form['activepieces_data']['ap_data_items'][$i]['remove' . $i] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#name' => 'remove_' . $i,
        '#submit' => [[$this, 'apDataRemoveSubmit']],
        '#ajax' => [
          'callback' => [$this, 'apDataRemoveSubmit'],
          'wrapper' => 'maestro-ap-data-ajax-refresh-wrapper',
          'extra_data' => $i,
        ],
        // Prevent full form validation when adding/removing rows.
        '#limit_validation_errors' => [],
      ];
    }

    $form['activepieces_data']['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another'),
      '#name' => 'add_another_ap_data',
      '#submit' => [[$this, 'apDataAddSubmit']],
      '#ajax' => [
        'callback' => [$this, 'apDataAjaxCallback'],
        'wrapper' => 'maestro-ap-data-ajax-refresh-wrapper',
      ],
      '#limit_validation_errors' => [],
    ];
    
    // Provide the token browser top and bottom of data entry area
    if (\Drupal::moduleHandler()->moduleExists('token')) {
      $form['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['maestro'],
      ];
    }
    else {
      $form['token_tree'] = [
        '#plain_text' => $this->t('Enabling the Token module will reveal the replacable tokens.'),
      ];
    }

    return $form;
  }

  /**
   * AJAX callback to rebuild the activepieces_data fieldset.
   */
  public function apDataAjaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['activepieces_data'];
  }

  /**
   * Submit handler for the Add button. Adds an empty row and rebuilds form.
   */
  public function apDataAddSubmit(array &$form, FormStateInterface $form_state) {
    return $form['activepieces_data'];
  }

  /**
   * Submit handler for the Remove button. Removes the targeted row and rebuilds.
   */
  public function apDataRemoveSubmit(array &$form, FormStateInterface $form_state) {
    return $form['activepieces_data'];
  }

 

  /**
   * {@inheritDoc}
   */
  public function validateTaskEditForm(array &$form, FormStateInterface $form_state) {
    $taskID = $form['task_id']['#default_value'] ?? NULL;
    $templateMachineName = $form['template_machine_name']['#default_value'] ?? NULL;
    $task = MaestroEngine::getTemplateTaskByID($templateMachineName, $taskID);
  }

  /**
   * {@inheritDoc}  
   */
  public function prepareTaskForSave(array &$form, FormStateInterface $form_state, array &$task) {
    $task['data']['activepieces'] = [];

    $task['data']['activepieces']['activepieces_url'] = $form_state->getValue('activepieces_url');
    // Persist activepieces items submitted from the form.
    $activepieces_data = $form_state->getValue('ap_data_items');
    if (is_array($activepieces_data)) {
      $items = [];
      foreach ($activepieces_data as $key => $row) {
        if (is_array($row) && (!empty($row['key' . $key]) || !empty($row['value' . $key]))) {
          $type = $row['type' . $key] ?? 'manually_entered'; // We fall back on the Type to manually entered.
          $value = $row['manual_value' . $key] ?? '';
          if($type == 'process_variable') {
            $value = $row['pv_value' . $key] ?? '';
          }
          $items[] = [
            'key' => $row['key' . $key] ?? '',
            'type' => $type,
            'value' => $value,
          ];
        }
      }
      $task['data']['activepieces']['activepieces_items'] = $items;
    }
  }

  /**
   * {@inheritDoc}
   */
  public function performValidityCheck(array &$validation_failure_tasks, array &$validation_information_tasks, array $task) {
    // So we know that we need a few keys in this $task array to allow the LLM to work.
    if ((array_key_exists('activepieces_url', $task['data']['activepieces'] ?? []) && $task['data']['activepieces']['activepieces_url'] == '')  || !array_key_exists('activepieces_url', $task['data']['activepieces'] ?? [])) {
      $validation_failure_tasks[] = [
        'taskID' => $task['id'],
        'taskLabel' => $task['label'],
        'reason' => $this->t('The Activepieces URI for the task has not been set.'),
      ];
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getTemplateBuilderCapabilities() {
    return ['edit', 'drawlineto', 'removelines', 'remove'];
  }

}
