<?php

namespace Drupal\maestro_activepieces\Plugin\EngineTasks;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\Entity\Key;
use Drupal\maestro\MaestroEngineTaskOrchestrationInterface;
use Drupal\maestro\MaestroTaskTrait;
use Drupal\maestro\Engine\MaestroEngine;
use Drupal\maestro\Form\MaestroExecuteInteractive;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Maestro Activepieces Task Plugin.
 *
 * The plugin annotations below should include:
 * id: The task type ID for this task.  For Maestro tasks, this is Maestro[TaskType].
 *     So for example, the start task shipped by Maestro is MaestroStart.
 *     The Maestro End task has an id of MaestroEnd
 *     Those task IDs are what's used in the engine when a task is injected into the queue.
 *
 * @Plugin(
 *   id = "MaestroActivepiecesTask",
 *   task_description = @Translation("The Maestro Engine's Activepieces task."),
 * )
 */
class MaestroActivepiecesTask extends PluginBase implements MaestroEngineTaskOrchestrationInterface {

  use MaestroTaskTrait;

  /**
   * Constructor.
   */
  public function __construct($configuration = NULL) {
    if (is_array($configuration)) {
      $this->processID = $configuration[0];
      $this->queueID = $configuration[1];
    }
  }

  /**
   * {@inheritDoc}
   */
  public function isInteractive() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function shortDescription() {
    return $this->t('Activepieces Task');
  }

  /**
   * {@inheritDoc}
   */
  public function description() {
    return $this->t('Activepieces Task.');
  }

  /**
   * {@inheritDoc}
   *
   * @see \Drupal\Component\Plugin\PluginBase::getPluginId()
   */
  public function getPluginId() {
    return 'MaestroActivepiecesTask';
  }

  /**
   * {@inheritDoc}
   */
  public function getTaskColours() {
    // AI tasks will be this Dodger Blue colour.
    return '#04BDFF';
  }

  /**
   * Part of the ExecutableInterface.
   *
   * Execution of the Batch Function task will use the handler for this task as the executable function.
   * The handler must return TRUE in order for this function to be completed by the engine.
   * We simply pass the return boolean value back from the called handler to the engine for processing.
   * {@inheritdoc}.
   */
  public function execute() {
    // Always keep from completing based on our $suspension_flag settings in the queue data.
    $suspension_flag = MaestroEngine::getSuspensionFlag($this->queueID);
    $returnValue = FALSE;
    // If we are suspending this task (not an error condition), then we check if our task data contains
    // instructions that we are to continually suspend operation.
    // Only execute on unsuspended, otherwise we return FALSE and the task is pseudo-suspended.
    if ($suspension_flag === MAESTRO_UNSUSPENDED) {
      $templateMachineName = MaestroEngine::getTemplateIdFromProcessId($this->processID);
      $taskMachineName = MaestroEngine::getTaskIdFromQueueId($this->queueID);
      $task = MaestroEngine::getTemplateTaskByID($templateMachineName, $taskMachineName);
      // This is the task's configured data.
      $taskData = $task['data']['activepieces'] ?? [];
      // Perform a Guzzle call to the configured Activepieces webhook.
      $client = new Client();
      $url = $taskData['activepieces_url'] ?? NULL;

      // Generate the data output array based on the configuration.
      $data_items = $taskData['activepieces_items'] ?? [];
      $data = [];
      // Let's seed our POST data with the process and queue IDs.
      // This way, we can target our return data to the appropriate process or queue item,
      // dependent upon what we are trying to achieve.
      $data['maestro_queue_id'] = $this->queueID;
      $data['maestro_process_id'] = $this->processID;

      foreach ($data_items as $data_item) {
        $data_value = '';
        if ($data_item['type'] == 'process_variable') {
          // Get the process variable value.
          $data_value = MaestroEngine::getProcessVariable($data_item['value'] ?? '', $this->processID);
          if ($data_value === FALSE) {
            $data_value = '';
          }
        }
        else {
          $data_value = $data_item['value'] ?? '';
          // Now run this through the token service.
          $tokenService = \Drupal::token();
          $data_value = $tokenService->replace(
            $data_value,
            [
              'maestro' =>
                [
                  'task' => $task,
                  'queueID' => $this->queueID,
                  'processID' => $this->processID,
                ],
            ]
          );
        }

        // Now, construct this data element.
        $data[$data_item['key']] = $data_value;
      }

      if ($url) {
        try {
          $options = [
            'headers' => [
              'Content-Type' => 'application/json',
            ],
            'body' => json_encode($data, JSON_THROW_ON_ERROR),
          ];

          // If we have any auth turned on the webhook, we'll need to also provide that information here.
          $auth_type = $taskData['activepieces_auth_type'] ?? NULL;
          if ($auth_type && $auth_type == 'basic') {
            // Basic auth we store the username/password in the Key module.
            $key = $taskData['activepieces_basic_auth_key'] ?? NULL;
            if ($key) {
              $key_value = \Drupal::service('key.repository')
                ->getKey($key)
                ->getKeyValue();
              $auth = json_decode($key_value, TRUE);
              $auth_user = $auth['username'] ?? '';
              $auth_pass = $auth['password'] ?? '';
              $options['auth'] = [$auth_user, $auth_pass, 'basic'];
            }
          }
          elseif ($auth_type && $auth_type == 'header') {
            // Header auth we store the name/value in the Key module.
            $key = $taskData['activepieces_header_auth_key'] ?? NULL;
            if ($key) {
              $key_value = \Drupal::service('key.repository')
                ->getKey($key)
                ->getKeyValue();
              $auth = json_decode($key_value, TRUE);
              $auth_header_name = $auth['headername'] ?? '';
              $auth_header_value = $auth['headervalue'] ?? '';
              $options['headers'][$auth_header_name] = $auth_header_value;
            }
          }

          // @todo Trap the response value in the event we need it for further Maestro processing.
          $client->request('POST', $url, $options);

          // Successful execution. Now set the suspension flag if we've configured the task to do so.
          $run_once = $taskData['activepieces_run_once'] ?? NULL;
          if ($run_once) {
            MaestroEngine::setSuspensionFlag($this->queueID);
            // Keep it resident in the Queue.
            $returnValue = FALSE;
          }
          else {
            // This is the only situation that the task will auto-complete.
            $returnValue = TRUE;
          }
        }
        catch (GuzzleException | \JsonException $e) {
          // Get the first 254 characters of the error.
          $error_message = substr($e->getMessage(), 0, 254);

          $log_return = $taskData['activepieces_error_log'] ?? NULL;
          if ($log_return) {
            \Drupal::logger('maestro_activepieces')->error($error_message);
          }

          $stall_task = $taskData['activepieces_stall'] ?? NULL;
          if ($stall_task == 1) {
            // Keep the engine from continuing.
            $returnValue = FALSE;
          }

          $task_status = $taskData['activepieces_task_status'] ?? TASK_STATUS_ACTIVE;
          $this->executionStatus = $task_status;

          $pv_error = $taskData['activepieces_failure_pv'] ?? '';
          if (!empty($pv_error)) {
            MaestroEngine::setProcessVariable($pv_error, $error_message, $this->processID);
          }
        }
      }
    }
    elseif ($suspension_flag == MAESTRO_ALLOW_SUSPENDED_COMPLETION) {
      // We can now safely complete this suspended task.
      $returnValue = TRUE;
    }

    return $returnValue;
  }

  /**
   * {@inheritdoc}
   */
  public function getExecutableForm($modal, MaestroExecuteInteractive $parent) {

  }

  /**
   * {@inheritdoc}
   */
  public function handleExecuteSubmit(array &$form, FormStateInterface $form_state) {

  }

  /**
   * {@inheritdoc}
   */
  public function getTaskEditForm(array $task, $templateMachineName) {
    /** @var \Drupal\Core\Form\FormStateInterface $form_state */
    $form_state = $task['form_state'];
    $taskData = $task['data']['activepieces'] ?? [];
    $form_state->disableCache();
    $form = [];
    $variables = MaestroEngine::getTemplateVariables($templateMachineName);

    $form['activepieces_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Activepieces Webhook URL'),
      '#required' => TRUE,
      '#default_value' => $taskData['activepieces_url'] ?? '',
      '#description' => $this->t('The URL which will call your Activepieces webhook.'),
    ];

    // On success-run-once, but only if Orchestration is enabled
    // We check for Orchestration module presence first.
    if (\Drupal::moduleHandler()->moduleExists('orchestration')) {
      $form['activepieces_run_once'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Keep this task resident in the Maestro Queue?'),
        '#required' => FALSE,
        '#default_value' => $taskData['activepieces_run_once'] ?? '',
        '#description' =>
        $this->t('When checked, this task will execute only once, communicating via a suspension flag.') . ' ' .
        $this->t('This allows you to communicate FROM Activepieces TO this open task and process via the Orchestrator module.'),
      ];
    }
    else {
      $form['activepieces_run_once'] = [
        '#markup' =>
        '<div class="messages">' .
        $this->t('The Orchestration module will allow you to suspend this task after execution') . ' ' .
        $this->t('and allow for bi-directional communication with Activepieces.') .
        '</div>',
      ];
    }

    // Authentication.
    $form['activepieces_auth'] = [
      '#type' => 'fieldset',
    ];

    $auth_options = ['' => $this->t('none')];
    // If we have the Key module installed, we can provide basic and header
    // auth and store those values in the keys.
    $keys = NULL;
    if (\Drupal::moduleHandler()->moduleExists('key')) {
      $auth_options['basic'] = $this->t('Basic Auth');
      $auth_options['header'] = $this->t('Header Auth');
      $keys = Key::loadMultiple();
      // Build options array: machine_name => label.
      $key_options = [];
      foreach ($keys as $key) {
        $key_options[$key->id()] = $key->label();
      }
    }
    else {
      $form['activepieces_auth']['key_information'] = [
        '#markup' =>
        '<div class="messages">' .
        $this->t('The Key module will allow you to store Basic Auth and Header Auth') . ' ' .
        $this->t('values. Basic and Header auth as options only become available when') . ' ' .
        $this->t('The Key module is enabled. See documentation for instructions.') .
        '</div>',
      ];
    }

    $form['activepieces_auth']['activepieces_auth_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication Type'),
      '#options' => $auth_options,
      '#required' => FALSE,
      '#default_value' => $taskData['activepieces_auth_type'] ?? '',
      '#description' => $this->t('The authentication required to make the webhook call. Configure this to match what is configured in the Activepieces webhook.'),
    ];

    // Based on the auth selection, we'll show/hide the appropriate fields.
    if (\Drupal::moduleHandler()->moduleExists('key')) {
      // Basic Auth.
      $form['activepieces_auth']['activepieces_basic_auth_key'] = [
        '#type' => 'select',
        '#title' => $this->t('Basic Auth Key'),
        '#options' => $key_options,
        '#required' => FALSE,
        '#default_value' => $taskData['activepieces_basic_auth_key'] ?? '',
        '#description' => $this->t('Select the configured Key for Basic Auth.'),
        '#states' => [
          'visible' => [
            ':input[name="activepieces_auth_type"]' => ['value' => 'basic'],
          ],
        ],
      ];
    }

    // Header Auth.
    $form['activepieces_auth']['activepieces_header_auth_key'] = [
      '#type' => 'select',
      '#title' => $this->t('Header Auth Key'),
      '#options' => $key_options,
      '#required' => FALSE,
      '#default_value' => $taskData['activepieces_header_auth_key'] ?? '',
      '#description' => $this->t('Select the configured Key for Header Auth.'),
      '#states' => [
        'visible' => [
          ':input[name="activepieces_auth_type"]' => ['value' => 'header'],
        ],
      ],
    ];

    $form['activepieces_error_handling'] = [
      '#type' => 'details',
      '#title' => $this->t('Error handling'),
    ];

    $form['activepieces_error_handling']['activepieces_error_log'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log failure to call the Activepieces webhook?'),
      '#required' => FALSE,
      '#default_value' => $taskData['activepieces_error_log'] ?? '',
      '#description' => $this->t('When checked, a log of the failure will be created in the system log.'),
    ];

    $form['activepieces_error_handling']['activepieces_stall'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Keep this task in the queue on failure to call the Activepieces webhook?'),
      '#required' => FALSE,
      '#default_value' => $taskData['activepieces_stall'] ?? '',
      '#description' => $this->t('When checked, this task will remain resident in the Maestro queue when a failure is detected during the webhook call.'),
    ];

    $task_status_array = MaestroEngine::getTaskStatusArray();
    // Remove the "active" option.
    unset($task_status_array[TASK_STATUS_ACTIVE]);
    $form['activepieces_error_handling']['activepieces_task_status'] = [
      '#type' => 'select',
      '#title' => $this->t('What to set the task status to upon webhook call failure?'),
      '#options' => $task_status_array,
      '#required' => FALSE,
      '#default_value' => $taskData['activepieces_task_status'] ?? 'TASK_STATUS_SUCCESS',
      '#description' => $this->t('If a webhook call fails, the status will be set to this value. Default is "Successfully Completed" (TASK_STATUS_SUCCESS)'),
      '#states' => [
        'visible' => [
          ':input[name="activepieces_stall"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $pv_failure_variables = [
      '' => $this->t('Do not set a Process Variable'),
    ];
    foreach ($variables as $variableName => $arr) {
      $pv_failure_variables[$variableName] = $variableName;
    }

    $form['activepieces_error_handling']['activepieces_failure_pv'] = [
      '#type' => 'select',
      '#title' => $this->t('Set the webhook error to a process variable on webhook call failure?'),
      '#options' => $pv_failure_variables,
      '#required' => FALSE,
      '#default_value' => $taskData['activepieces_failure_pv'] ?? '',
      '#description' => $this->t('If a webhook call fails, the process variable chosen above will be set to the error.'),
    ];

    // Dynamic listing of key/value pairs that we will send along to Activepieces.
    // This listing will be in the form of a textbox for the key, followed by a drop-down to specify what TYPE
    // of value: process variable, token or manually_entered
    // Then a final box where you will either choose the PV, provide the token or manually_entered value.
    $form['activepieces_data'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Data to send to Activepieces'),
      '#description' => $this->t('Provide the data structure to send. Provide a key and define the data type you are sending'),
      '#prefix' => '<div id="maestro-ap-data-ajax-refresh-wrapper">',
      '#suffix' => '</div>',
    ];

    $form_input = $form_state->getUserInput();
    $unique_id = uniqid();
    // Check if this is an ajax operation.
    $triggering_element = $form_input['_triggering_element_name'] ?? NULL;
    if ($triggering_element && $triggering_element == 'add_another_ap_data') {
      $items = $form_input['ap_data_items'];
      $items[$unique_id] = [
        'key' . $unique_id => '',
        'type' . $unique_id => 'manually_entered',
        'value' . $unique_id => '',
      ];
    }
    elseif ($triggering_element && strpos($triggering_element, 'remove_') === 0) {
      $index = str_replace('remove_', '', $triggering_element);
      if ($index !== NULL) {
        $items = $form_input['ap_data_items'];
        unset($items[$index]);
        if (count($items) === 0) {
          $items[$unique_id] = [
            'key' . $unique_id => '',
            'type' . $unique_id => 'manually_entered',
            'value' . $unique_id => '',
          ];
        }
      }
    }
    elseif (isset($form_input['ap_data_items']) && is_array($form_input['ap_data_items'])) {
      $items = $form_input['ap_data_items'];
    }
    else {
      $items = $taskData['activepieces_items'] ?? [];
    }

    $form['activepieces_data']['ap_data_items'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'maestro-ap-data-items'],
      '#tree' => TRUE,
    ];

    $type_options = [
      'process_variable' => $this->t('Process Variable'),
      'manually_entered' => $this->t('Manual Entry and/or Tokens'),
    ];

    $variable_options = [
      '' => $this->t('Choose Process Variable'),
    ];
    foreach ($variables as $variableName => $arr) {
      $variable_options[$variableName] = $variableName;
    }

    // Provide the token browser top and bottom of data entry area.
    if (\Drupal::moduleHandler()->moduleExists('token')) {
      $form['activepieces_data']['ap_data_items']['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['maestro'],
      ];
    }
    else {
      $form['activepieces_data']['ap_data_items']['token_tree'] = [
        '#plain_text' => $this->t('Enabling the Token module will reveal the replacable tokens.'),
      ];
    }

    foreach ($items as $i => $item) {
      $form['activepieces_data']['ap_data_items'][$i] = [
        '#type' => 'fieldset',
        '#attributes' => [
          'class' => ['maestro-ap-row'],
          'data-index' => $i,
        ],
      ];

      $form['activepieces_data']['ap_data_items'][$i]['key' . $i] = [
        '#type' => 'textfield',
        '#title' => $this->t('Key'),
        '#default_value' => $item['key'] ?? '',
        '#description' => $this->t('The key you specify here becomes the key used in the Activepieces parsed JSON input.'),
      ];

      $type = $item['type'] ?? 'manually_entered';
      $form['activepieces_data']['ap_data_items'][$i]['type' . $i] = [
        '#type' => 'select',
        '#title' => $this->t('Type'),
        '#options' => $type_options,
        '#default_value' => $type,
        '#description' => $this->t('The type of data being passed.'),
      ];

      // Based on the Type selected above, we will show one of the following:  A textarea for tokens or manually entered values or a process variable select.
      $form['activepieces_data']['ap_data_items'][$i]['manual_value' . $i] = [
        '#type' => 'textarea',
        '#title' => $this->t('Value'),
        '#default_value' => $type == 'process_variable' ? '' : ($item['value'] ?? ''),
        '#description' => $this->t("The manually entered data you're passing to Activepieces. You can use tokens."),
        '#states' => [
          'visible' => [
            ':input[name="ap_data_items[' . $i . '][type' . $i . ']"]' => ['value' => 'manually_entered'],
          ],
        ],
      ];

      $form['activepieces_data']['ap_data_items'][$i]['pv_value' . $i] = [
        '#type' => 'select',
        '#options' => $variable_options,
        '#title' => $this->t('Value'),
        '#default_value' => $type == 'process_variable' ? ($item['value'] ?? '') : '',
        '#description' => $this->t("The process variable whose value you're sending to Activepieces."),
        '#states' => [
          'visible' => [
            ':input[name="ap_data_items[' . $i . '][type' . $i . ']"]' => ['value' => 'process_variable'],
          ],
        ],
      ];

      $form['activepieces_data']['ap_data_items'][$i]['remove' . $i] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#name' => 'remove_' . $i,
        '#submit' => [[$this, 'apDataAjaxCallback']],
        '#ajax' => [
          'callback' => [$this, 'apDataAjaxCallback'],
          'wrapper' => 'maestro-ap-data-ajax-refresh-wrapper',
          'extra_data' => $i,
        ],
        // Prevent full form validation when adding/removing rows.
        '#limit_validation_errors' => [],
      ];
    }

    $form['activepieces_data']['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another'),
      '#name' => 'add_another_ap_data',
      '#submit' => [[$this, 'apDataAjaxCallback']],
      '#ajax' => [
        'callback' => [$this, 'apDataAjaxCallback'],
        'wrapper' => 'maestro-ap-data-ajax-refresh-wrapper',
      ],
      '#limit_validation_errors' => [],
    ];

    // Provide the token browser top and bottom of data entry area.
    if (\Drupal::moduleHandler()->moduleExists('token')) {
      $form['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['maestro'],
      ];
    }
    else {
      $form['token_tree'] = [
        '#plain_text' => $this->t('Enabling the Token module will reveal the replacable tokens.'),
      ];
    }

    return $form;
  }

  /**
   * AJAX callback to rebuild the activepieces_data fieldset.
   */
  public function apDataAjaxCallback(array &$form, FormStateInterface $form_state) {
    return $form['activepieces_data'];
  }

  /**
   * {@inheritDoc}
   */
  public function validateTaskEditForm(array &$form, FormStateInterface $form_state) {
    // If we have header or basic auth, we will validate that the key chosen actually conforms.
    $activepieces_auth_type = $form_state->getValue('activepieces_auth_type');
    $activepieces_header_auth_key = $form_state->getValue('activepieces_header_auth_key');
    $activepieces_basic_auth_key = $form_state->getValue('activepieces_basic_auth_key');
    if ($activepieces_auth_type == 'header') {
      if (empty($activepieces_header_auth_key)) {
        $form_state->setErrorByName(
            'activepieces_header_auth_key',
            $this->t('Choose the appropriate Header Auth key.')
          );
      }
      else {
        $key_value = \Drupal::service('key.repository')
          ->getKey($activepieces_header_auth_key)
          ->getKeyValue();
        $auth = json_decode($key_value, TRUE);
        if (empty($auth['headername']) || empty($auth['headervalue'])) {
          $form_state->setErrorByName(
            'activepieces_header_auth_key',
            $this->t('Header Auth key requires JSON format and the properties of headername and headervalue.')
          );
        }
      }
    }

    if ($activepieces_auth_type == 'basic') {
      if (empty($activepieces_basic_auth_key)) {
        $form_state->setErrorByName(
            'activepieces_basic_auth_key',
            $this->t('Choose the appropriate Header Auth key.')
          );
      }
      else {
        $key_value = \Drupal::service('key.repository')
          ->getKey($activepieces_basic_auth_key)
          ->getKeyValue();
        $auth = json_decode($key_value, TRUE);
        if (empty($auth['username']) || empty($auth['password'])) {
          $form_state->setErrorByName(
            'activepieces_basic_auth_key',
            $this->t('Basic Auth key requires JSON format and the properties of username and password.')
          );
        }
      }
    }

    // Make sure that we have a URL that at least matches a URL pattern and data config that has keys and potentially a PV chosen.
    $webhook = $form_state->getValue('activepieces_url');
    if (!UrlHelper::isExternal($webhook, TRUE)) {
      $form_state->setErrorByName('activepieces_url', $this->t('It appears your URL is invalid.'));
    }

    $activepieces_data = $form_state->getValue('ap_data_items');
    $existing_keys = [];
    if (is_array($activepieces_data)) {
      foreach ($activepieces_data as $key => $row) {
        $supplied_key = $row['key' . $key] ?? '';
        // Does the key already exist? If so, show an error.
        if (array_key_exists($supplied_key, $existing_keys)) {
          $form_state->setErrorByName(
            'ap_data_items][' . $key . '][key' . $key,
            $this->t('The :key key already exists and must be unique.', [':key' => $supplied_key])
          );
        }
        else {
          // Add the key to the list.
          $existing_keys[$supplied_key] = $supplied_key;
        }
        // Does the key match a reserved key name such as maestro_queue_id and maestro_process_id?
        if ($supplied_key == 'maestro_queue_id' || $supplied_key == 'maestro_process_id') {
          $form_state->setErrorByName('ap_data_items][' . $key . '][key' . $key, $this->t('This key is reserved for Maestro Activepieces integration.'));
        }
        // Is the key blank?
        if (is_array($row) && empty($supplied_key)) {
          $form_state->setErrorByName('ap_data_items][' . $key . '][key' . $key, $this->t('Every data bundle must have a key.'));
        }
        // If the type is process variable, then the PV should be chosen.
        $type = $row['type' . $key] ?? NULL;
        $value = $row['pv_value' . $key] ?? NULL;
        if ($type == 'process_variable' && !$value) {
          $form_state->setErrorByName('ap_data_items][' . $key . '][pv_value' . $key, $this->t('Must choose a process variable.'));
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public function prepareTaskForSave(array &$form, FormStateInterface $form_state, array &$task) {
    $task['data']['activepieces'] = [];

    $task['data']['activepieces']['activepieces_run_once'] = $form_state->getValue('activepieces_run_once');
    $task['data']['activepieces']['activepieces_error_log'] = $form_state->getValue('activepieces_error_log');
    $task['data']['activepieces']['activepieces_stall'] = $form_state->getValue('activepieces_stall');
    $task['data']['activepieces']['activepieces_task_status'] = $form_state->getValue('activepieces_task_status');
    $task['data']['activepieces']['activepieces_failure_pv'] = $form_state->getValue('activepieces_failure_pv');
    $task['data']['activepieces']['activepieces_url'] = $form_state->getValue('activepieces_url');
    $task['data']['activepieces']['activepieces_auth_type'] = $form_state->getValue('activepieces_auth_type');
    $task['data']['activepieces']['activepieces_header_auth_key'] = $form_state->getValue('activepieces_header_auth_key');
    $task['data']['activepieces']['activepieces_basic_auth_key'] = $form_state->getValue('activepieces_basic_auth_key');

    // Persist activepieces items submitted from the form.
    $activepieces_data = $form_state->getValue('ap_data_items');
    if (is_array($activepieces_data)) {
      $items = [];
      foreach ($activepieces_data as $key => $row) {
        if (is_array($row) && (!empty($row['key' . $key]) || !empty($row['value' . $key]))) {
          // We fall back on the Type to manually entered.
          $type = $row['type' . $key] ?? 'manually_entered';
          $value = $row['manual_value' . $key] ?? '';
          if ($type == 'process_variable') {
            $value = $row['pv_value' . $key] ?? '';
          }
          $items[] = [
            'key' => $row['key' . $key] ?? '',
            'type' => $type,
            'value' => $value,
          ];
        }
      }
      $task['data']['activepieces']['activepieces_items'] = $items;
    }
  }

  /**
   * {@inheritDoc}
   */
  public function performValidityCheck(array &$validation_failure_tasks, array &$validation_information_tasks, array $task) {
    // So we know that we need a few keys in this $task array to allow the LLM to work.
    if ((array_key_exists('activepieces_url', $task['data']['activepieces'] ?? []) && $task['data']['activepieces']['activepieces_url'] == '')  || !array_key_exists('activepieces_url', $task['data']['activepieces'] ?? [])) {
      $validation_failure_tasks[] = [
        'taskID' => $task['id'],
        'taskLabel' => $task['label'],
        'reason' => $this->t('The Activepieces URI for the task has not been set.'),
      ];
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getTemplateBuilderCapabilities() {
    return ['edit', 'drawlineto', 'removelines', 'remove'];
  }

  /**
   * {@inheritDoc}
   */
  public function orchestrationAllowTaskCompletion($queueID) {
    MaestroEngine::setSuspensionFlag($queueID, MAESTRO_ALLOW_SUSPENDED_COMPLETION);
  }

}
