(function (Drupal) {
  Drupal.behaviors.magicCodeVerifyInput = {
    attach(context) {
      once("magicCodeVerifyInput", ".verify-code-input", context).forEach(
        /**
         * @param {HTMLElement} element
         */
        (element) => {
          const inputs = element.querySelectorAll("input");

          /**
           * Submit the form.
           */
          function submitForm() {
            const form = element.closest("form");
            if (form) {
              form.dispatchEvent(new Event("submit"));
              form.submit();
            }
          }

          /**
           * Map the code to each individual input.
           */
          function setCode(code, autosubmit = false) {
            code
              .replace("-", "")
              .split("")
              .forEach((char, index) => {
                inputs[index].value = char;
              });

            if (autosubmit) {
              submitForm();
            }
          }

          inputs.forEach((input, index) => {
            // Handle input.
            input.addEventListener("input", () => {
              if (input.value === "") {
                return;
              }

              // If input, focus next one.
              if (index < 5) {
                inputs[index + 1]?.focus();
              } else {
                const form = element.closest("form");
                const data = new FormData(form);

                const parts = [
                  data.get("code_input_1"),
                  data.get("code_input_2"),
                  data.get("code_input_3"),
                  "-",
                  data.get("code_input_4"),
                  data.get("code_input_5"),
                  data.get("code_input_6"),
                ];

                const code = parts.join("");

                // Autosubmit if the code is valid.
                const regex = /^[a-zA-Z0-9]{3}-[a-zA-Z0-9]{3}$/;
                if (regex.test(code)) {
                  submitForm();
                }
              }
            });

            // Handle keydown (backspace).
            input.addEventListener("keydown", (event) => {
              if (input.value !== "" || event.key !== "Backspace") {
                return;
              }

              // Focus previous one.
              if (index > 0) {
                inputs[index - 1]?.focus();
              }
            });

            // Handle focus.
            input.addEventListener("focus", () => {
              if (input.value === "") {
                return;
              }

              input.select();
            });

            // Handle paste.
            input.addEventListener("paste", (event) => {
              event.preventDefault();

              const text = event.clipboardData.getData("text").trim();

              // Validate input.
              if (text.length !== 7 || text[3] !== "-") {
                return;
              }

              setCode(text, true);
            });
          });

          /**
           * Prefill from query params.
           */
          const code = new URLSearchParams(location.search).get("code");
          if (code) {
            setCode(code);
          }
        },
      );
    },
  };

  Drupal.behaviors.magicCodeResendTimeout = {
    attach(context) {
      once("magicCodeResendTimeout", "#resend", context).forEach(
        /**
         * @param {HTMLButtonElement} button
         */
        (button) => {
          const TIMEOUT_DURATION = 20000; // 20 seconds in milliseconds
          let timeoutId = null;
          let countdownInterval = null;
          const originalText = button.value || button.textContent;

          // Find the icon element (sibling or within parent container)
          const container = button.closest(".verify-action-resend__container");
          const icon = container
            ? container.querySelector(".verify-action-resend__icon")
            : null;

          /**
           * Disable the button and start countdown.
           */
          function disableButton() {
            button.disabled = true;
            button.setAttribute("aria-disabled", "true");

            // Apply opacity and cursor styles to button and icon
            button.style.opacity = "0.5";
            button.style.cursor = "not-allowed";
            if (icon) {
              icon.style.opacity = "0.5";
              icon.style.pointerEvents = "none";
              icon.style.cursor = "not-allowed";
            }
            if (container) {
              container.style.cursor = "not-allowed";
            }

            let remainingSeconds = TIMEOUT_DURATION / 1000;

            // Update button text with countdown
            function updateCountdown() {
              if (remainingSeconds > 0) {
                button.value = originalText + ` (${remainingSeconds}s)`;
                remainingSeconds--;
              } else {
                enableButton();
              }
            }

            // Initial update
            updateCountdown();

            // Update every second
            countdownInterval = setInterval(updateCountdown, 1000);

            // Re-enable after timeout
            timeoutId = setTimeout(() => {
              enableButton();
            }, TIMEOUT_DURATION);
          }

          /**
           * Enable the button and restore original text.
           */
          function enableButton() {
            if (timeoutId) {
              clearTimeout(timeoutId);
              timeoutId = null;
            }
            if (countdownInterval) {
              clearInterval(countdownInterval);
              countdownInterval = null;
            }
            button.disabled = false;
            button.removeAttribute("aria-disabled");
            button.value = originalText;

            // Restore opacity and cursor styles for button and icon
            button.style.opacity = "";
            button.style.cursor = "pointer";
            if (icon) {
              icon.style.opacity = "";
              icon.style.pointerEvents = "";
              icon.style.cursor = "";
            }
            if (container) {
              container.style.cursor = "pointer";
            }
          }

          // Disable button on initial page load
          disableButton();

          // Handle button click - disable again after click
          // Use setTimeout to ensure form submission happens first
          button.addEventListener("click", () => {
            // Only disable if button is currently enabled
            if (!button.disabled) {
              // Use setTimeout to let the form submit first
              setTimeout(() => {
                // Check if button still exists (page might have reloaded)
                if (button && button.parentNode) {
                  disableButton();
                }
              }, 0);
            }
          });
        },
      );
    },
  };
})(Drupal);
