<?php

namespace Drupal\mail_action\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Component\Render\PlainTextOutput;
use Drupal\Component\Utility\DeprecationHelper;
use Drupal\Core\Render\RenderContext;

/**
 * Sends an email message with formatted text.
 */
#[Action(
  id: 'mail_action:formatted',
  label: new TranslatableMarkup('Send email with formatted text'),
  type: 'system'
)]
class FormattedMailAction extends MailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'message' => ['value' => '', 'format' => 'plain_text'],
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    if ($this->moduleHandler->moduleExists('filter')) {
      $configuration = $this->getConfiguration();
      $form['message']['#type'] = 'text_format';
      $form['message']['#format'] = $configuration['message']['format'] ?? 'plain_text';
      $form['message']['#allowed_formats'] = array_keys(filter_formats(\Drupal::currentUser()));
    }

    // @Todo Once ECA standard modeller can handle native Drupal forms, this
    // workaround block can be dropped.
    // @see https://www.drupal.org/project/bpmn_io/issues/3336030
    if (self::isBpmnIoModeller()) {
      $form['message']['#type'] = 'textarea';
      $form['message_format'] = [
        '#type' => 'select',
        '#title' => $this->t('Format'),
        '#options' => array_combine($form['message']['#allowed_formats'], $form['message']['#allowed_formats']),
        '#default_value' => $form['message']['#format'],
        '#required' => TRUE,
        '#weight' => ($form['message']['#weight'] + 10),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    // @Todo Once ECA standard modeller can handle native Drupal forms, this
    // workaround block can be dropped.
    // @see https://www.drupal.org/project/bpmn_io/issues/3336030
    if (self::isBpmnIoModeller()) {
      $configuration = [];

      $message = $form_state->getValue('message');
      if (is_string($message)) {
        $message = ['value' => $message] + ($this->defaultConfiguration()['message'] ?? []);
      }
      $message['format'] = $form_state->getValue('message_format');
      $configuration['message'] = $message;
      unset($configuration['message_format']);

      $this->setConfiguration($configuration + $this->getConfiguration());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepareMessage(&$message, $subject, $body, array $data): void {
    $format = isset($this->configuration['message']['format']) ? $this->configuration['message']['format'] : NULL;
    if (!isset($format) || !$this->moduleHandler->moduleExists('filter')) {
      $body = PlainTextOutput::renderFromHtml($body);
    }
    else {
      $body = $this->renderer->executeInRenderContext(new RenderContext(), function () use ($body, $format) {
        $build = [
          '#type' => 'processed_text',
          '#text' => $body,
          '#format' => $format,
        ];
        return DeprecationHelper::backwardsCompatibleCall(
          currentVersion: \Drupal::VERSION,
          deprecatedVersion: '10.3',
          currentCallable: fn() => $this->renderer->renderInIsolation($build),
          deprecatedCallable: fn() => $this->renderer->renderPlain($build),
        );
      });
    }

    parent::prepareMessage($message, $subject, $body, $data);
  }

  /**
   * Helper method to determine whether we are in the ECA BPMN io modeller.
   *
   * Used to implement a fallback so that the form is compatible with it.
   *
   * @return bool
   *   Returns TRUE if BPMN io modeller, FALSE otherwise.
   */
  private static function isBpmnIoModeller(): bool {
    // @Todo Once ECA standard modeller can handle native Drupal forms, this
    // workaround block can be dropped.
    // @see https://www.drupal.org/project/bpmn_io/issues/3336030
    $is_bpmn_modeller = FALSE;
    if (\Drupal::routeMatch()->getRouteName() === 'bpmn_io.add') {
      $is_bpmn_modeller = TRUE;
    }
    elseif (\Drupal::routeMatch()->getRouteObject() && (\Drupal::routeMatch()->getParameter('modeller_id') === 'bpmn_io')) {
      $is_bpmn_modeller = TRUE;
    }
    elseif (str_starts_with((string) \Drupal::routeMatch()->getRouteName(), 'entity.eca.') && !is_null(\Drupal::routeMatch()->getParameter('eca'))) {
      if ($eca = \Drupal::entityTypeManager()->getStorage('eca')->load(\Drupal::routeMatch()->getParameter('eca'))) {
        /** @var \Drupal\eca\Entity\Eca $eca */
        if ($eca->get('modeller') === 'bpmn_io') {
          $is_bpmn_modeller = TRUE;
        }
      }
    }
    return $is_bpmn_modeller;
  }

}
