<?php

namespace Drupal\mail_action\Plugin\Action;

use Drupal\Core\Action\ConfigurableActionBase;
use Drupal\Component\Render\PlainTextOutput;
use Drupal\Component\Utility\EmailValidatorInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Template\TwigEnvironment;
use Drupal\Core\Utility\Token;
use Drupal\user\UserInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for mail_action plugins.
 */
abstract class MailActionBase extends ConfigurableActionBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * A logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected MailManagerInterface $mailManager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The email validator.
   *
   * @var \Drupal\Component\Utility\EmailValidatorInterface
   */
  protected EmailValidatorInterface $emailValidator;

  /**
   * The token replacement service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected Token $token;

  /**
   * The Twig environment service.
   *
   * @var \Drupal\Core\Template\TwigEnvironment
   */
  protected TwigEnvironment $twig;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->logger = $container->get('logger.factory')->get('mail_action');
    $instance->mailManager = $container->get('plugin.manager.mail');
    $instance->languageManager = $container->get('language_manager');
    $instance->moduleHandler = $container->get('module_handler');
    $instance->renderer = $container->get('renderer');
    $instance->emailValidator = $container->get('email.validator');
    $instance->token = $container->get('token');
    $instance->twig = $container->get('twig');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'recipient' => '',
      'subject' => '',
      'message' => ['value' => ''],
      'processing' => '_none',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    return $return_as_object ? AccessResult::allowed() : TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    $user_storage = $this->entityTypeManager->getStorage('user');

    $data = [];
    foreach (func_get_args() as $k => $v) {
      if ($v instanceof EntityInterface) {
        $k = $v->getEntityType()->get('token_type') ?: $v->getEntityTypeId();
      }
      $data[$k] = $v;
    }

    $recipient = PlainTextOutput::renderFromHtml($this->token->replacePlain($this->configuration['recipient'], $data));
    $langcode = $this->languageManager->getDefaultLanguage()->getId();
    if (!str_contains($recipient, ',')) {
      // If the recipient is a registered user with a language preference, use
      // the recipient's preferred language. Otherwise, use the system default
      // language.
      $recipient_accounts = $user_storage->loadByProperties(['mail' => $recipient]);
      $recipient_account = reset($recipient_accounts);
      if ($recipient_account instanceof UserInterface) {
        $langcode = $recipient_account->getPreferredLangcode();
      }
    }

    $subject = PlainTextOutput::renderFromHtml($this->token->replacePlain($this->configuration['subject'], $data));

    $body = isset($this->configuration['message']['value']) ? $this->configuration['message']['value'] : '';

    switch ($this->configuration['processing']) {

      case 'token':
        $body = $this->token->replacePlain($body, $data);
        break;

      case 'twig':
        $build = [
          '#type' => 'inline_template',
          '#template' => $body,
          '#context' => $data,
        ];
        $body = $this->renderer->executeInRenderContext(new RenderContext(), function () use (&$build) {
          return $this->renderer->render($build);
        });
        break;

    }

    $params = [
      'plugin' => $this,
      'data' => $data,
      'subject' => $subject,
      'body' => $body,
      '_error_message' => FALSE,
    ];
    $reply = NULL;
    $send = TRUE;

    $message = $this->mailManager->mail('mail_action', substr(strstr($this->getPluginId(), ':'), 1), $recipient, $langcode, $params, $reply, $send);
    // Error logging is handled by \Drupal\Core\Mail\MailManager::mail().
    if ($message['result']) {
      $this->logger->info('Sent email to %recipient', ['%recipient' => $recipient]);
    }
  }

  /**
   * Prepares the message of this mail instance.
   *
   * @see mail_action_mail()
   *
   * @param &message
   *   The message array to be filled in. See hook_mail() for details.
   * @param mixed $subject
   *   The message subject.
   * @param mixed $body
   *   The message body.
   * @param array $data
   *   Data arguments that are passed to the execute method of this plugin.
   */
  public function prepareMessage(&$message, $subject, $body, array $data): void {
    $message['subject'] .= str_replace(["\r", "\n"], '', $subject);
    $message['body'][] = $body;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $configuration = $this->getConfiguration();

    $weight = 1000;

    $form['recipient'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Recipient email address'),
      '#default_value' => $configuration['recipient'],
      '#maxlength' => '254',
      '#description' => $this->t('Separate multiple recipients with a comma. <a href=":token_url" target="_blank" rel="noopener noreferrer">Tokens</a> are supported.', [
        ':token_url' => 'https://www.drupal.org/project/token',
      ]),
      '#required' => TRUE,
      '#weight' => ($weight += 10),
    ];
    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $configuration['subject'],
      '#maxlength' => '254',
      '#required' => TRUE,
      '#weight' => ($weight += 10),
    ];

    $form['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message'),
      '#default_value' => $configuration['message']['value'] ?? '',
      '#required' => TRUE,
      '#weight' => ($weight += 10),
    ];

    $form['processing'] = [
      '#type' => 'select',
      '#title' => $this->t('Further processing of the message contents'),
      '#description' => $this->t('You may choose either no further processing, enable <a href=":token_url" target="_blank" rel="noopener noreferrer">Token replacement</a> or use the <a href=":twig_url" target="_blank" rel="noopener noreferrer">Twig templating language</a>.', [
        ':token_url' => 'https://www.drupal.org/project/token',
        ':twig_url' => 'https://twig.symfony.com/doc/3.x/',
      ]),
      '#options' => $this->getProcessingOptions(),
      '#default_value' => $configuration['processing'],
      '#required' => FALSE,
      '#empty_value' => '_none',
      '#weight' => ($weight += 10),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $recipient_values = $form_state->getValue('recipient');

    if ($form_state->getValue('processing') === 'twig') {
      try {
        $message_body = $form_state->getValue('message');
        if (is_array($message_body)) {
          $message_body = $message_body['value'];
        }
        $this->twig->renderInline($message_body);
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('message', $this->t('The provided template is not valid Twig.'));
      }
    }

    foreach (explode(',', $recipient_values) as $recipient_value) {
      $recipient_value = trim($recipient_value);
      $is_token = str_starts_with($recipient_value, '[') && str_ends_with($recipient_value, ']');
      if (!$is_token && !$this->emailValidator->isValid($recipient_value)) {
        $form_state->setErrorByName('recipient', $this->t('Enter a valid email address.'));
        break;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $configuration = [];

    $configuration['recipient'] = $form_state->getValue('recipient');
    $configuration['subject'] = $form_state->getValue('subject');

    $message = $form_state->getValue('message');
    if (is_string($message)) {
      $message = ['value' => $message] + ($this->defaultConfiguration()['message'] ?? []);
    }
    $configuration['message'] = $message;

    $configuration['processing'] = $form_state->getValue('processing', '_none');

    $this->setConfiguration($configuration + $this->getConfiguration());
  }

  /**
   * Returns a list of available processing options.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup[]
   *   Available processing options as array.
   */
  protected function getProcessingOptions(): array {
    return [
      '_none' => $this->t('No further processing'),
      'token' => $this->t('Apply Token replacement'),
      'twig' => $this->t('Use Twig templating language'),
    ];
  }

}
