<?php

namespace Drupal\mail_entity_queue_symfony_mailer\Plugin\MailEntityQueue;

use Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface;
use Drupal\mail_entity_queue\Event\MailEntityQueueItemEvents;
use Drupal\mail_entity_queue\Plugin\MailEntityQueue\MailEntityQueueProcessorBase;
use Drupal\mail_entity_queue_symfony_mailer\SymfonyMailEntityQueueProcessorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Symfony Mailer processor for Mail Entity Queue.
 *
 * @MailEntityQueueProcessor(
 *   id = "symfony_mail_entity_queue",
 *   label = @Translation("Symfony Mailer queue processor"),
 *   description = @Translation("Processes mail_entity_queue items using Symfony Mailer and a custom EmailBuilder.")
 * )
 */
class SymfonyMailEntityQueueProcessor extends MailEntityQueueProcessorBase implements SymfonyMailEntityQueueProcessorInterface {

  /**
   * Email factory.
   *
   * @var \Drupal\symfony_mailer\EmailFactoryInterface
   */
  protected $emailFactory;

  /**
   * Symfony mailer.
   *
   * @var \Drupal\symfony_mailer\MailerInterface
   */
  protected $mailer;

  /**
   * The email builder manager.
   *
   * @var \Drupal\symfony_mailer\Processor\EmailBuilderManagerInterface
   */
  protected $builderManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('language_manager'),
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher'),
      $container->get('logger.factory')->get('mail_entity_queue')
    );

    $instance->emailFactory = $container->get('email_factory');
    $instance->mailer = $container->get('symfony_mailer');
    $instance->builderManager = $container->get('plugin.manager.email_builder');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function processItem(MailEntityQueueItemInterface $item, int $delay = 0) {
    if ($item->getStatus() === MailEntityQueueItemInterface::SENT) {
      $this->logger->error($this->t('Mail @name already has been sent. Skipping.', ['@name' => $item->label()]));
      return FALSE;
    }

    $params = $item->getData();
    $params['mail_entity_queue'] = $item->queue();
    $params['mail_entity_queue_item'] = $item->id();

    $message = [
      'module' => 'mail_entity_queue',
      'key' => 'symfony_mail_entity_queue',
      'to' => $params['to'],
      'langcode' => $params['langcode'] ?? $this->languageManager->getDefaultLanguage()->getId(),
      'params' => $params,
    ];

    /** @var \Drupal\symfony_mailer\Processor\EmailBuilderInterface $builder */
    $builder = $this->builderManager->createInstance('symfony_mail_entity_queue');
    $email = $builder->fromArray($this->emailFactory, $message);

    // Send the email.
    $sent = $this->mailer->send($email);

    // Update the item status.
    $max_attempts = $item->getMaxAttempts() ?? 5;
    $item->setAttempts($item->getAttempts() + 1);
    // If max attempts is -1, we retry indefinitely.
    if (!$sent && ($max_attempts === -1 || $item->getAttempts() < $max_attempts)) {
      $item->setStatus(MailEntityQueueItemInterface::RETRYING);
      $this->dispatchEvent($item, MailEntityQueueItemEvents::MAIL_ENTITY_QUEUE_ITEM_PROCESSED_WRONGLY);
    }
    elseif (!$sent) {
      $item->setStatus(MailEntityQueueItemInterface::DISCARDED);
      $this->dispatchEvent($item, MailEntityQueueItemEvents::MAIL_ENTITY_QUEUE_ITEM_DISCARDED);
    }
    else {
      $item->setStatus(MailEntityQueueItemInterface::SENT);
      $this->dispatchEvent($item, MailEntityQueueItemEvents::MAIL_ENTITY_QUEUE_ITEM_PROCESSED_SUCCESSFULLY);
    }

    try {
      $item->save();
    }
    catch (\Exception $e) {
      $this->logger->error($this->t('Failed to save mail_entity_queue_item @id: @message', [
        '@id' => $item->id(),
        '@message' => $e->getMessage(),
      ]));

      throw $e;
    }

    if ($delay > 0) {
      usleep($delay);
    }

    return TRUE;
  }

}
