<?php

namespace Drupal\mail_entity_queue\Plugin\MailEntityQueue;

use Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface;
use Drupal\mail_entity_queue\Event\MailEntityQueueItemEvents;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a default mail queue processor.
 *
 * @MailEntityQueueProcessor(
 *   id = "mail_entity_queue_default",
 *   label = @Translation("Default mail queue processor"),
 *   description = @Translation("Provides a default mail queue processor."),
 * )
 */
class DefaultMailEntityQueueProcessor extends MailEntityQueueProcessorBase {

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('language_manager'),
      $container->get('entity_type.manager'),
      $container->get('event_dispatcher'),
      $container->get('logger.factory')->get('mail_entity_queue')
    );

    $instance->mailManager = $container->get('plugin.manager.mail');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function processItem(MailEntityQueueItemInterface $item, int $delay = 0) {
    if ((int) $item->getStatus() === MailEntityQueueItemInterface::SENT) {
      $this->logger->error($this->t('Mail @name already has been sent. Skipping.', ['@name' => $item->label()]));
      return FALSE;
    }

    $params = $item->getData();
    $params['mail_entity_queue'] = $item->queue();

    if (empty($params['langcode'])) {
      $langcode = $this->languageManager->getDefaultLanguage()->getId();
    }
    else {
      $langcode = $params['langcode'];
    }

    $reply = NULL;
    if (!empty($params['reply'])) {
      $reply = $params['reply'];
    }

    // Make sure we can update this item in hook_mail() or hook_mail_alter().
    $params['mail_entity_queue_item'] = $item->id();

    $message = $this->mailManager->mail('mail_entity_queue', 'default_mail_entity_queue_processor', $item->getMail(), $langcode, $params, $reply, TRUE);

    // @todo Make attempts configurable.
    $item->setAttempts($item->getAttempts() + 1);
    if (!$message['result'] && $item->getAttempts() < 5) {
      $item->setStatus(MailEntityQueueItemInterface::RETRYING);
      $this->dispatchEvent($item, MailEntityQueueItemEvents::MAIL_ENTITY_QUEUE_ITEM_PROCESSED_WRONGLY);
    }
    elseif (!$message['result']) {
      $item->setStatus(MailEntityQueueItemInterface::DISCARDED);
      $this->dispatchEvent($item, MailEntityQueueItemEvents::MAIL_ENTITY_QUEUE_ITEM_DISCARDED);
    }
    else {
      $item->setStatus(MailEntityQueueItemInterface::SENT);
      $this->dispatchEvent($item, MailEntityQueueItemEvents::MAIL_ENTITY_QUEUE_ITEM_PROCESSED_SUCCESSFULLY);
    }

    $item->save();

    // Congestion control.
    usleep($delay);

    return $message['result'];
  }

}
