<?php

namespace Drupal\mail_entity_queue_symfony_mailer\Plugin\EmailAdjuster;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\mail_entity_queue\Entity\MailEntityQueue;
use Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface;
use Drupal\mail_entity_queue_symfony_mailer\SymfonyMailEntityQueueProcessorInterface;
use Drupal\symfony_mailer\AddressInterface;
use Drupal\symfony_mailer\EmailInterface;
use Drupal\symfony_mailer\Exception\SkipMailException;
use Drupal\symfony_mailer\Processor\EmailAdjusterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines an email adjuster to queue sending of emails.
 *
 * @EmailAdjuster(
 *   id = "symfony_mail_entity_queue",
 *   label = @Translation("Symfony Mail Entity Queue"),
 *   description = @Translation("Queue all emails to a mail entity queue using Symfony Mailer."),
 *   weight = 0,
 * )
 */
class SymfonyQueueMailAdjuster extends EmailAdjusterBase implements ContainerFactoryPluginInterface {

  /**
   * The default number of attempts.
   */
  const DEFAULT_ATTEMPTS = 5;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->time = $container->get('datetime.time');
    $instance->languageManager = $container->get('language_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function build(EmailInterface $email): void {
    // Load the queue.
    $queue_id = $this->configuration['queue'] ?? NULL;
    if (empty($queue_id)) {
      // No queue configured, nothing to do.
      return;
    }

    if ($email->getType() === 'mail_entity_queue' || $email->getSubType() === 'default_mail_entity_queue_processor') {
      // Do not queue emails that are already being sent from a
      // mail entity queue.
      return;
    }

    $vars = $email->getVariables();
    if (isset($vars['mail_entity_queue']) && $vars['mail_entity_queue'] instanceof MailEntityQueue) {
      // The email is already in a queue, do not enqueue it again.
      return;
    }

    $data = [
      'type' => $email->getType(),
      'subtype' => $email->getSubType(),
      'params' => $this->serializeParams($email),
      'variables' => $email->getVariables(),
      'addresses' => $this->serializeAddresses($email),
      'sender' => $email->getSender(),
      'subject' => $email->getSubject(),
      'body' => $email->getBody(),
      'theme' => $email->getTheme(),
      'transport_dsn' => $email->getTransportDsn(),
      'langcode' => $this->languageManager->getCurrentLanguage()->getId(),
    ];

    $values = [
      'queue' => $queue_id,
      'mail' => $this->formatAddresses($email->getTo(), FALSE),
      'data' => $data,
      'entity_type' => $email->getEntity() ? $email->getEntity()->getEntityTypeId() : NULL,
      'entity_id' => $email->getEntity() ? $email->getEntity()->id() : NULL,
      'status' => MailEntityQueueItemInterface::PENDING,
      'attempts' => 0,
      'max_attempts' => (int) $this->configuration['attempts_count'],
      'created' => $this->time->getRequestTime(),
    ];

    $storage = $this->entityTypeManager->getStorage('mail_entity_queue_item');
    /** @var \Drupal\mail_entity_queue\Entity\MailEntityQueueItem $item **/
    $item = $storage->create($values);
    $item->save();

    throw new SkipMailException('The email has been queued for sending');
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $queues = $this->entityTypeManager->getStorage('mail_entity_queue')->loadMultiple();

    $options = [];
    /** @var \Drupal\mail_entity_queue\Entity\MailEntityQueue $queue **/
    foreach ($queues as $id => $queue) {
      if (!$queue->getQueueProcessor() instanceof SymfonyMailEntityQueueProcessorInterface) {
        // Only show queues that implement the Symfony Mail Entity Queue
        // processor interface.
        continue;
      }

      $options[$id] = $queue->label();
    }

    if (empty($options)) {
      $form['message'] = [
        '#markup' => $this->t('No mail entity queues are available.'),
      ];
      return $form;
    }

    $form['queue'] = [
      '#type' => 'select',
      '#title' => $this->t('Mail Entity Queue'),
      '#options' => $options,
      '#default_value' => $this->configuration['queue'] ?? NULL,
      '#description' => $this->t('Select the mail entity queue to use for queue emails. Note: the queue must have the Symfony Mail Entity Queue processor selected.'),
      '#required' => TRUE,
    ];

    $form['attempts_count'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of Attempts'),
      '#default_value' => ($this->configuration['attempts_count'] ?? self::DEFAULT_ATTEMPTS),
      '#description' => $this->t('Maximum number of attempts to send an email. Set to -1 for unlimited attempts.'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * Format an array of addresses into a string.
   *
   * @param \Drupal\symfony_mailer\AddressInterface[] $addresses
   *   An array of address objects.
   * @param bool $display_name
   *   Whether to include the display name in the formatted string.
   *
   * @return string|null
   *   The formatted address string, or null if no addresses are provided.
   */
  protected function formatAddresses(array $addresses, bool $display_name = TRUE): ?string {
    if (empty($addresses)) {
      return NULL;
    }

    $result = [];
    $name = NULL;
    foreach ($addresses as $address) {
      if ($address instanceof AddressInterface) {
        if ($display_name) {
          $name = $address->getDisplayName();
        }
        $email = $address->getEmail();
        $result[] = $name ? "$name <$email>" : $email;
      }
    }

    return implode(', ', $result);
  }

  /**
   * Serialize email addresses into an array.
   *
   * @param \Drupal\symfony_mailer\EmailInterface $email
   *   The email object.
   *
   * @return array
   *   The serialized addresses.
   */
  protected function serializeAddresses(EmailInterface $email): array {
    $result = [];

    foreach (['to', 'from', 'cc', 'bcc', 'reply-to'] as $type) {
      $result[$type] = array_map(fn(AddressInterface $address) => [
        'email' => $address->getEmail(),
        'display_name' => $address->getDisplayName(),
        'langcode' => $address->getLangcode(),
        'account_id' => $address->getAccount() ? $address->getAccount()->id() : NULL,
      ], $email->getAddress($type) ?? []);
    }
    return $result;
  }

  /**
   * Serialize email parameters into an array.
   *
   * @param \Drupal\symfony_mailer\EmailInterface $email
   *   The email object.
   *
   * @return array
   *   The serialized parameters.
   */
  protected function serializeParams(EmailInterface $email): array {
    $params = $email->getParams();
    $serialized_params = [];
    foreach ($params as $key => $param) {
      if (is_object($param)) {
        // Set type and ID for entity objects.
        if ($param instanceof EntityInterface) {
          $serialized_params[$key] = [
            'entity_type' => $param->getEntityTypeId(),
            'id' => $param->id(),
          ];
        }
      }
      else {
        $serialized_params[$key] = $param;
      }
    }

    return $serialized_params;
  }

}
