<?php

namespace Drupal\mail_entity_queue_symfony_mailer\Plugin\EmailBuilder;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\symfony_mailer\Address;
use Drupal\symfony_mailer\EmailFactoryInterface;
use Drupal\symfony_mailer\MailerHelperTrait;
use Drupal\symfony_mailer\Processor\EmailBuilderBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Email Builder for emails queued via mail_entity_queue and Symfony Mailer.
 *
 * @EmailBuilder(
 *   id = "symfony_mail_entity_queue",
 *   sub_types = { },
 *   common_adjusters = {},
 *   import = @Translation(""),
 *   proxy = FALSE,
 *   override = {},
 * )
 */
class SymfonyMailEntityQueueEmailBuilder extends EmailBuilderBase implements ContainerFactoryPluginInterface {

  use MailerHelperTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function fromArray(EmailFactoryInterface $factory, array $message) {
    $data = $message['data'];
    $params = isset($data['params']) ? $this->unserializeParams($data['params']) : [];

    // Create email based on entity or type/subtype.
    if (isset($data['entity_id']) && isset($data['entity_type'])) {
      $entity = $this->entityTypeManager
        ->getStorage($data['entity_type'])
        ->load($data['entity_id']);

      if ($entity instanceof ConfigEntityInterface) {
        $email = $factory->newEntityEmail(
          $entity,
          $data['subtype'] ?? '',
          ...$params,
        );
      }
    }
    else {
      $email = $factory->newTypedEmail(
        $data['type'] ?? '',
        $data['subtype'] ?? '',
        ...$params,
      );
    }

    if (!isset($email)) {
      throw new \InvalidArgumentException('Unable to create email from provided data.');
    }

    isset($data['variables']) && $email->setVariables($data['variables']);
    isset($data['sender']) && $email->setSender($data['sender']);
    isset($data['subject']) && $email->setSubject($data['subject']);
    isset($data['body']) && $email->setBody($data['body']);
    isset($data['theme']) && $email->setTheme($data['theme']);
    isset($data['transportDsn']) && $email->setTransportDsn($data['transportDsn']);

    foreach (['to', 'from', 'cc', 'bcc', 'reply-to'] as $type) {
      $addresses = [];
      if (isset($data['addresses'][$type])) {
        foreach ($data['addresses'][$type] ?? [] as $address) {
          if (isset($address['email'])) {
            // Load the user account if an account ID is provided.
            $account = isset($address['account']) ? $this->entityTypeManager
              ->getStorage('user')
              ->load($address['account_id']) : NULL;

            // Create the Address object.
            $addresses[] = new Address(
              $address['email'],
              $address['display_name'],
              $address['langcode'],
              $account
            );
          }
        }
      }
      elseif (!empty($data[$type])) {
        $address = $data[$type];

        // Ensure display names are quoted.
        if (preg_match('/^(.*?)\s*<([^>]+)>$/', $address, $matches)) {
          $display_name = trim($matches[1], '" ');
          $email_address = trim($matches[2]);
          if ($display_name !== '' && $display_name[0] !== '"' && substr($display_name, -1) !== '"') {
            $address = sprintf('"%s" <%s>', $display_name, $email_address);
          }
        }

        // Retro compatibility with older serialized formats.
        $addresses = $this->helper()->parseAddress($address);
      }

      if ($addresses) {
        $email->setAddress($type, $addresses);
      }
    }

    foreach (['mail_entity_queue', 'mail_entity_queue_item'] as $variable) {
      if (!empty($data[$variable])) {
        $email->setVariable($variable, $data[$variable]);
      }
    }

    return $email;
  }

  /**
   * {@inheritdoc}
   */
  public function import() {
  }

  /**
   * Unserialize parameters, loading entities as needed.
   *
   * @param array $params
   *   The serialized parameters.
   *
   * @return array
   *   The unserialized parameters.
   */
  protected function unserializeParams(array $params): array {
    $unserialize_params = [];
    foreach ($params as $key => $value) {
      if (isset($value['entity_type'], $value['id'])) {
        $unserialize_params[$key] = $this->entityTypeManager
          ->getStorage($value['entity_type'])
          ->load($value['id']);
      }
      else {
        $unserialize_params[$key] = $value;
      }
    }

    return $unserialize_params;
  }

}
