<?php

namespace Drupal\mail_entity_queue_webform\Plugin\WebformHandler;

use Drupal\Component\Serialization\Yaml;
use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\Utility\WebformElementHelper;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform\WebformThemeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'mail_entity_queue_webform' handler.
 *
 * @WebformHandler(
 *   id = "mail_entity_queue_webform",
 *   label = @Translation("Mail Entity Queue"),
 *   description = @Translation("Enqueue an email using Mail Entity Queue module."),
 *   category = @Translation("Mail"),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 *   submission = \Drupal\webform\Plugin\WebformHandlerInterface::SUBMISSION_OPTIONAL,
 *   tokens = TRUE,
 * )
 */
class MailEntityQueueWebformHandler extends WebformHandlerBase {

  /**
   * The webform theme manager.
   *
   * @var \Drupal\Webform\WebformThemeManagerInterface
   */
  protected WebformThemeManagerInterface $themeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->themeManager = $container->get('webform.theme_manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'mail_entity_queue' => NULL,
      'entity_type' => NULL,
      'entity_id' => NULL,
      'other_fields' => '',
      'to_mail' => [],
      'cc_mail' => [],
      'bcc_mail' => [],
      'from_mail' => NULL,
      'reply_mail' => NULL,
      'subject' => NULL,
      'body' => NULL,
      'theme_name' => NULL,
      'langcode' => NULL,
      'states' => [WebformSubmissionInterface::STATE_COMPLETED],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary(): array {
    $settings = $this->getSettings();
    // Get state labels.
    $settings['states'] = implode(
      ', ',
      array_intersect_key(
        $this->getStates(),
        array_combine($settings['states'], $settings['states'])
      )
    );
    $settings['settings']['mail_entity_queue'] = [
      'title' => $this->t('Queue'),
      'value' => $settings['mail_entity_queue'],
    ];
    $settings['settings']['entity_type'] = [
      'title' => $this->t('Entity type'),
      'value' => $settings['entity_type'],
    ];
    $settings['settings']['entity_id'] = [
      'title' => $this->t('Entity ID'),
      'value' => $settings['entity_id'],
    ];

    return [
      '#settings' => $settings,
    ] + parent::getSummary();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // Mail Entity Queue Item.
    $form['mail_entity_queue_item'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Mail Entity Queue Item settings'),
    ];

    $queues = [];
    foreach ($this->entityTypeManager->getStorage('mail_entity_queue')->loadMultiple() as $queue) {
      $queues[$queue->id()] = $queue->label();
    }
    $form['mail_entity_queue_item']['mail_entity_queue'] = [
      '#type' => 'select',
      '#title' => $this->t('Mail Entity Queue'),
      '#description' => $this->t('Select the queue that will be used to queue this email.'),
      '#required' => TRUE,
      '#options' => $queues,
      '#empty_value' => '',
      '#empty_option' => $this->t('- Select -'),
      '#default_value' => $this->configuration['mail_entity_queue'],
    ];
    $form['mail_entity_queue_item']['entity_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity type'),
      '#description' => $this->t('The type of the entity to which this queue item is related.'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['entity_type'] ?? NULL,
    ];
    $form['mail_entity_queue_item']['entity_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity ID'),
      '#description' => $this->t('The ID of the entity to which this queue item is related.'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['entity_id'] ?? NULL,
    ];
    $form['mail_entity_queue_item']['other_fields'] = [
      '#type' => 'webform_codemirror',
      '#mode' => 'yaml',
      '#title' => $this->t('Custom item data'),
      '#description' => $this->t('Enter custom data that will be included in the item data. In YAML format.'),
      '#default_value' => $this->configuration['other_fields'] ?? '',
    ];
    // To.
    $form['mail_entity_queue_item']['to'] = [
      '#type' => 'details',
      '#title' => $this->t('Send to'),
    ];
    $form['mail_entity_queue_item']['to']['to_mail'] = [
      '#type' => 'textarea',
      '#title' => $this->t('To'),
      '#default_value' => $this->configuration['to_mail'],
      '#description' => $this->t('The email address or addresses where the message will be sent to, one value per line. The formatting of this string will be validated with the <a href=":link">PHP email validation filter</a>. Some examples are: "user@example.com" or "Another User &lt;another-user@example.com&gt;".', [':link' => 'http://php.net/manual/filter.filters.validate.php']),
      '#required' => TRUE,
    ];
    $form['mail_entity_queue_item']['to']['cc_mail'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Cc'),
      '#default_value' => $this->configuration['cc_mail'],
      '#description' => $this->t('The email address or addresses where the message will be sent carbon copy to, one value per line. The formatting of this string will be validated with the <a href=":link">PHP email validation filter</a>. Some examples are: "user@example.com" or "Another User &lt;another-user@example.com&gt;".', [':link' => 'http://php.net/manual/filter.filters.validate.php']),
    ];
    $form['mail_entity_queue_item']['to']['bcc_mail'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Bcc'),
      '#default_value' => $this->configuration['bcc_mail'],
      '#description' => $this->t('The email address or addresses where the message will be sent blind carbon copy to, one value per line. The formatting of this string will be validated with the <a href=":link">PHP email validation filter</a>. Some examples are: "user@example.com" or "Another User &lt;another-user@example.com&gt;".', [':link' => 'http://php.net/manual/filter.filters.validate.php']),
    ];
    // Form.
    $form['mail_entity_queue_item']['from'] = [
      '#type' => 'details',
      '#title' => $this->t('Send from'),
    ];
    $form['mail_entity_queue_item']['from']['from_mail'] = [
      '#type' => 'textfield',
      '#title' => $this->t('From'),
      '#default_value' => $this->configuration['from_mail'] ?? '[site:name] <[site:mail]>',
      '#description' => $this->t('The email address or addresses where the message will be sent from. The formatting of this string will be validated with the <a href=":link">PHP email validation filter</a>. Some examples are: "user@example.com" or "User &lt;user@example.com&gt;".', [':link' => 'http://php.net/manual/filter.filters.validate.php']),
      '#required' => TRUE,
    ];
    $form['mail_entity_queue_item']['from']['reply_mail'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reply to'),
      '#default_value' => $this->configuration['reply_mail'],
      '#description' => $this->t('The email address to be used to answer. The formatting of this string will be validated with the <a href=":link">PHP email validation filter</a>. Some examples are: "user@example.com" or "User &lt;user@example.com&gt;".', [':link' => 'http://php.net/manual/filter.filters.validate.php']),
    ];
    // Message.
    $form['mail_entity_queue_item']['message'] = [
      '#type' => 'details',
      '#title' => $this->t('Message'),
    ];
    $form['mail_entity_queue_item']['message']['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $this->configuration['subject'],
      '#required' => TRUE,
    ];
    $form['mail_entity_queue_item']['message']['body'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Body'),
      '#default_value' => $this->configuration['body']['value'] ?? NULL,
      '#format' => $this->configuration['body']['format'] ?? filter_default_format(),
      '#required' => TRUE,
    ];
    // Additional.
    $form['mail_entity_queue_item']['additional'] = [
      '#type' => 'details',
      '#title' => $this->t('Additional'),
    ];
    $form['mail_entity_queue_item']['additional']['theme_name'] = [
      '#type' => 'select',
      '#title' => $this->t('Theme to render this email'),
      '#description' => $this->t('Select the theme that will be used to render this email.'),
      '#options' => $this->themeManager->getThemeNames(),
      '#empty_value' => '',
      '#empty_option' => $this->t('- Site default -'),
      '#default_value' => $this->configuration['theme_name'],
    ];
    $form['mail_entity_queue_item']['additional']['langcode'] = [
      '#type' => 'language_select',
      '#title' => $this->t('Language to render this email'),
      '#languages' => LanguageInterface::STATE_CONFIGURABLE,
      '#description' => $this->t('Select the language that will be used to render this email.'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['langcode'],
    ];

    // Additional.
    $results_disabled = $this->getWebform()->getSetting('results_disabled');
    $form['additional'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Additional settings'),
    ];
    // Settings: States.
    $form['additional']['states'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Send email'),
      '#options' => $this->getStates(),
      '#access' => !$results_disabled,
      '#default_value' => $results_disabled ? [WebformSubmissionInterface::STATE_COMPLETED] : $this->configuration['states'],
    ];

    // ISSUE: TranslatableMarkup is breaking the #ajax.
    // WORKAROUND: Convert all Render/Markup to strings.
    WebformElementHelper::convertRenderMarkupToStrings($form);

    $this->elementTokenValidate($form);

    return $this->setSettingsParents($form);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $this->applyFormStateToConfiguration($form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE): void {
    $state = $webform_submission->getWebform()->getSetting('results_disabled') ? WebformSubmissionInterface::STATE_COMPLETED : $webform_submission->getState();
    if (in_array($state, $this->configuration['states'])) {
      $this->enqueueMail($webform_submission);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postDelete(WebformSubmissionInterface $webform_submission) {
    $state = $webform_submission->getWebform()->getSetting('results_disabled') ? WebformSubmissionInterface::STATE_COMPLETED : $webform_submission->getState();
    if (in_array($state, $this->configuration['states'])) {
      $this->enqueueMail($webform_submission);
    }
  }

  /**
   * Enqueue the mail.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $webform_submission
   *   The webform submission.
   *
   * @throws \Exception
   */
  protected function enqueueMail(WebformSubmissionInterface $webform_submission): void {
    $config = $this->configuration;
    unset($config['states']);

    $data = $this->tokenManager->replace($config, $webform_submission);
    // Prepare the message, add the body and subject and queue it.
    $to = $cc = $bcc = $from = $reply_to = NULL;
    if (!empty($data['to_mail'])) {
      $to = array_filter(array_map('trim', explode(PHP_EOL, $data['to_mail'])));
    }
    if (!empty($data['cc_mail'])) {
      $cc = array_filter(array_map('trim', explode(PHP_EOL, $data['cc_mail'])));
    }
    if (!empty($data['bcc_mail'])) {
      $bcc = array_filter(array_map('trim', explode(PHP_EOL, $data['bcc_mail'])));
    }
    if (!empty($data['from_mail'])) {
      $from = $data['from_mail'];
    }
    if (!empty($data['reply_mail'])) {
      $reply_to = $data['reply_mail'];
    }

    foreach (['to', 'cc', 'bcc', 'from', 'reply_to'] as $item) {
      if (!is_array($$item)) {
        $$item = [$$item];
      }
      $$item = array_filter($$item);
      $$item = array_map(Html::class . '::decodeEntities', $$item);

      foreach ($$item as $key => $mail) {
        if (empty($mail) || (!filter_var($mail, FILTER_VALIDATE_EMAIL) && !preg_match('/^\s*(.+?)\s*<\s*([^>]+)\s*>$/', $mail))) {
          unset(${$item}[$key]);
        }
      }

      $$item = implode(', ', $$item);
    }

    // Don't send the message if To, CC, and BCC are empty.
    if (empty($to) && empty($cc) && empty($bcc)) {
      $this->getLogger()->warning('Unable to sent email for %submission webform submission because a <em>To</em>, <em>CC</em>, or <em>BCC</em> email was not provided.', ['%submission' => $webform_submission->label()]);

      return;
    }

    // Prepare the message.
    $message['to'] = $to;
    $message['cc'] = $cc;
    $message['bcc'] = $bcc;
    $message['reply-to'] = $reply_to;
    $message['from'] = $from;
    $message['theme_name'] = $this->configuration['theme_name'];
    $message['langcode'] = $data['langcode'];
    // Remove empty recipients.
    $message = array_filter($message);

    $message['subject'] = Html::decodeEntities($data['subject']);
    $data['body']['value'] = Html::decodeEntities($data['body']['value']);
    $build = [
      '#type' => 'processed_text',
      '#text' => $data['body']['value'],
      '#format' => $data['body']['format'],
    ];
    $message['body'] = trim((string) $this->themeManager->renderPlain($build, $message['theme_name']));
    $message['to'] = str_replace(',', ' ', $message['to']);
    if (isset($message['reply-to'])) {
      if (str_contains($message['reply-to'], ',')) {
        $pos_email = strpos($message['reply-to'], ' <');
        $name_rfc2822 = substr($message['reply-to'], 0, $pos_email);
        preg_match('/<(.*?)>/', $message['reply-to'], $match);
        $message['reply-to'] = '"' . $name_rfc2822 . '" ' . $match[0];
      }
    }

    $item_data = Yaml::decode($data['other_fields'] ?? '') ?: [];
    $values = [
      'queue' => $data['mail_entity_queue'],
      'mail' => $message['to'],
      'data' => $message,
      'entity_type' => $data['entity_type'],
      'entity_id' => $data['entity_id'],
      'status' => MailEntityQueueItemInterface::PENDING,
    ] + $item_data;

    $storage = $this->entityTypeManager->getStorage('mail_entity_queue_item');
    $item = $storage->create($values);
    $item->save();
  }

  /**
   * Gets submission states.
   *
   * @return array
   *   The states.
   */
  protected function getStates(): array {
    return [
      WebformSubmissionInterface::STATE_DRAFT_CREATED => $this->t('…when draft is created.'),
      WebformSubmissionInterface::STATE_DRAFT_UPDATED => $this->t('…when draft is updated.'),
      WebformSubmissionInterface::STATE_CONVERTED => $this->t('…when anonymous submission is converted to authenticated.'),
      WebformSubmissionInterface::STATE_COMPLETED => $this->t('…when submission is completed.'),
      WebformSubmissionInterface::STATE_UPDATED => $this->t('…when submission is updated.'),
      WebformSubmissionInterface::STATE_DELETED => $this->t('…when submission is deleted.'),
      WebformSubmissionInterface::STATE_LOCKED => $this->t('…when submission is locked.'),
    ];
  }

}
