<?php

namespace Drupal\mail_entity_queue_webform_node\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface;
use Drupal\node\NodeInterface;

/**
 * Defines a custom access control for the Mail Entity Queue Webform Node.
 */
class MailEntityQueueWebformNodeAccess {

  /**
   * Check whether the user can access a node's mail queue item list.
   *
   * @param string $operation
   *   Operation being performed.
   * @param string $entity_access
   *   Entity access rule that needs to be checked.
   * @param \Drupal\node\NodeInterface $node
   *   A node.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Run access checks for this account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public static function checkMailEntityQueueListAccess(string $operation, string $entity_access, NodeInterface $node, AccountInterface $account): AccessResultInterface {
    $access_result = static::checkAccess($operation, $entity_access, $node, NULL, $account);

    if ($access_result->isAllowed()) {
      // Check if there is any mail queue item associated with any webform
      // submission of the node's webform.
      $mail_q_items = \Drupal::entityTypeManager()
        ->getStorage('mail_entity_queue_item')
        ->getQuery()
        ->accessCheck(FALSE)
        ->condition('entity_type', 'node')
        ->condition('entity_id', $node->id())
        ->count()
        ->execute();

      /** @var \Drupal\webform\WebformEntityReferenceManagerInterface $entity_reference_manager */
      $entity_reference_manager = \Drupal::service('webform.entity_reference_manager');
      $webform = $entity_reference_manager->getWebform($node);

      return AccessResult::allowedIf($mail_q_items > 0)
        ->addCacheableDependency($webform)
        ->addCacheableDependency($node)
        ->addCacheTags(['mail_entity_queue_item_list']);
    }

    return $access_result;
  }

  /**
   * Check whether the user can access a node's webform submission.
   *
   * @param string $operation
   *   Operation being performed.
   * @param string $entity_access
   *   Entity access rule that needs to be checked.
   * @param \Drupal\node\NodeInterface $node
   *   A node.
   * @param \Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface $mail_entity_queue_item
   *   A mail queue item.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Run access checks for this account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public static function checkMailEntityQueueAccess(string $operation, string $entity_access, NodeInterface $node, MailEntityQueueItemInterface $mail_entity_queue_item, AccountInterface $account): AccessResultInterface {
    return static::checkAccess($operation, $entity_access, $node, $mail_entity_queue_item, $account);
  }

  /**
   * Check whether the user can access a node's webform and/or submission.
   *
   * @param string $operation
   *   Operation being performed.
   * @param string $entity_access
   *   Entity access rule that needs to be checked.
   * @param \Drupal\node\NodeInterface $node
   *   A node.
   * @param \Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface|null $mail_entity_queue_item
   *   A mail queue item.
   * @param \Drupal\Core\Session\AccountInterface|null $account
   *   Run access checks for this account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public static function checkAccess(string $operation, string $entity_access, NodeInterface $node, ?MailEntityQueueItemInterface $mail_entity_queue_item = NULL, ?AccountInterface $account = NULL): AccessResultInterface {
    /** @var \Drupal\webform\WebformEntityReferenceManagerInterface $entity_reference_manager */
    $entity_reference_manager = \Drupal::service('webform.entity_reference_manager');

    $webform = $entity_reference_manager->getWebform($node);
    // Check that the node has a valid webform reference.
    if (!$webform) {
      return AccessResult::forbidden();
    }

    // Determine if this is a group node.
    $is_group_node = FALSE;
    if (\Drupal::moduleHandler()->moduleExists('webform_group')) {
      if (\Drupal::entityTypeManager()->hasDefinition('group_content')) {
        $is_group_node = \Drupal::entityTypeManager()->getStorage('group_content')->loadByEntity($node);
      }
      elseif (\Drupal::entityTypeManager()->hasDefinition('group_relation')) {
        $is_group_node = \Drupal::entityTypeManager()->getStorage('group_relation')->loadByEntity($node);
      }
    }

    // Check the node operation.
    if (!$operation) {
      $result = AccessResult::neutral();
    }
    elseif ($is_group_node && str_starts_with($operation, 'mail_entity_queue_item_')) {
      // For group nodes, we need to bypass node access checking for
      // 'mail_entity_queue_item_*' operations which trigger access forbidden.
      // @see group_entity_access()
      // @see https://www.drupal.org/project/webform/issues/3132204
      $result = mail_entity_queue_webform_node_node_access($node, $operation, $account);
    }
    else {
      $result = $node->access($operation, $account, TRUE);
    }

    // Check entity access.
    if ($entity_access) {
      // Check entity access for the mail queue item.
      if ($operation === 'mail_entity_queue_item.view_any_node') {
        // Allow users with 'view mail_entity_queue_item any node' to view all
        // mail items or users with 'view mail_entity_queue_item own node' to
        // view their nodes.
        $any_access = $account->hasPermission('view mail_entity_queue_item any node');
        $own_access = $account->hasPermission('view mail_entity_queue_item own node')
          && $node->getOwnerId() === $account->id();

        $result = AccessResult::allowedIf($any_access || $own_access);
      }
      // Anonymous users can access their own mail items, so we need to check
      // if the mail queue item own|any node permission is set.
      // We need to ensure that the mail queue item is associated with the node.
      elseif ($mail_entity_queue_item && str_starts_with($entity_access, 'mail_entity_queue_item.')) {
        $entity_operation = str_replace('mail_entity_queue_item.', '', $entity_access);

        // If the mail queue item is not associated with the node, deny access.
        $mail_q_item_from_node = in_array($node->id(), static::getNodesFromMailEntityQueueItem($mail_entity_queue_item));
        if (!$mail_q_item_from_node) {
          return AccessResult::forbidden();
        }

        $mail_q_item_access = $mail_entity_queue_item->access($entity_operation, $account);
        $any_access = $account->hasPermission($entity_operation . ' mail_entity_queue_item any node');
        $own_access = $account->hasPermission($entity_operation . ' mail_entity_queue_item own node') &&
          $node->getOwnerId() === $account->id();

        $result = AccessResult::allowedIf(
          ($mail_q_item_access && ($any_access || $own_access)) ||
          ($any_access || $own_access)
        );
      }
    }

    return $result;
  }

  /**
   * Get the node IDs from the mail item items of a mail queue item.
   *
   * @param \Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface $mail_entity_queue_item
   *   The mail queue item.
   *
   * @return array
   *   An array of node IDs.
   */
  protected static function getNodesFromMailEntityQueueItem(MailEntityQueueItemInterface $mail_entity_queue_item): array {
    return $mail_entity_queue_item->getSourceEntityType() === 'node' ? [$mail_entity_queue_item->getSourceEntityId()] : [];
  }

}
