<?php

namespace Drupal\mail_entity_queue\Plugin\MailEntityQueue;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface;
use Drupal\mail_entity_queue\Event\MailEntityQueueItemEvent;
use Drupal\mail_entity_queue\Plugin\MailEntityQueueProcessorInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Base class for Mail Entity Queue processors.
 *
 * Provides common queue logic (processQueue, dispatchEvent),
 * leaving the actual send logic to subclasses (processItem).
 */
abstract class MailEntityQueueProcessorBase extends PluginBase implements MailEntityQueueProcessorInterface {

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The mail queue storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $queueStorage;

  /**
   * The mail queue item storage.
   *
   * @var \Drupal\mail_entity_queue\MailEntityQueueItemEntityStorage
   */
  protected $queueItemStorage;

  /**
   * A logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a MailEntityQueueProcessorBase object.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, LanguageManagerInterface $language_manager, EntityTypeManagerInterface $entity_type_manager, EventDispatcherInterface $event_dispatcher, LoggerInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->languageManager = $language_manager;
    $this->queueStorage = $entity_type_manager->getStorage('mail_entity_queue');
    $this->queueItemStorage = $entity_type_manager->getStorage('mail_entity_queue_item');
    $this->eventDispatcher = $event_dispatcher;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   *
   * Subclasses must implement the sending logic.
   */
  abstract public function processItem(MailEntityQueueItemInterface $item, int $delay = 0);

  /**
   * {@inheritdoc}
   */
  public function processQueue(string $mail_entity_queue) {
    /** @var \Drupal\mail_entity_queue\Entity\MailEntityQueueInterface $queue */
    $queue = $this->queueStorage->load($mail_entity_queue);

    $ids = $this->queueItemStorage->getQuery()
      ->condition('queue', $mail_entity_queue)
      ->condition('status', [MailEntityQueueItemInterface::SENT, MailEntityQueueItemInterface::DISCARDED], 'NOT IN')
      ->sort('changed', 'ASC')
      ->range(0, $queue->getCronItems())
      ->accessCheck(FALSE)
      ->execute();

    if ($ids) {
      $items = $this->queueItemStorage->loadMultiple($ids);
      foreach ($items as $item) {
        /** @var \Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface $item */
        $this->processItem($item, $queue->getCronDelay());
      }
    }
  }

  /**
   * Notify other modules the result of the process.
   *
   * @param \Drupal\mail_entity_queue\Entity\MailEntityQueueItemInterface $item
   *   The element that generates this event.
   * @param string $event_name
   *   The name of the event to dispatch.
   */
  protected function dispatchEvent(MailEntityQueueItemInterface $item, string $event_name) {
    $event = new MailEntityQueueItemEvent($item);
    $this->eventDispatcher->dispatch($event, $event_name);
  }

}
