<?php

namespace Drupal\mailchimp_campaign\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\mailchimp_campaign\MailchimpCampaignInterface;

/**
 * Defines the MailchimpCampaign entity.
 *
 * @ingroup mailchimp_campaign
 *
 * @ContentEntityType(
 *   id = "mailchimp_campaign",
 *   label = @Translation("Mailchimp Campaign"),
 *   handlers = {
 *     "access" = "Drupal\mailchimp_campaign\MailchimpCampaignAccessControlHandler",
 *     "view_builder" = "Drupal\mailchimp_campaign\Entity\MailchimpCampaignViewBuilder",
 *     "form" = {
 *       "send" = "Drupal\mailchimp_campaign\Form\MailchimpCampaignSendForm",
 *       "sendtestmail" = "Drupal\mailchimp_campaign\Form\MailchimpCampaignSendTestMailForm",
 *       "add" = "Drupal\mailchimp_campaign\Form\MailchimpCampaignForm",
 *       "edit" = "Drupal\mailchimp_campaign\Form\MailchimpCampaignForm",
 *       "delete" = "Drupal\mailchimp_campaign\Form\MailchimpCampaignDeleteForm"
 *     }
 *   },
 *   base_table = "mailchimp_campaign",
 *   admin_permission = "administer mailchimp",
 *   fieldable = TRUE,
 *   entity_keys = {
 *     "id" = "mc_campaign_id",
 *     "uuid" = "uuid"
 *   },
 *   links = {
 *     "edit-form" = "/admin/config/services/mailchimp/campaign/{mailchimp_campaign}",
 *     "delete-form" = "/admin/config/services/mailchimp/campaign/{mailchimp_campaign}/delete",
 *     "canonical" = "/admin/config/services/mailchimp/campaign/{mailchimp_campaign}",
 *   }
 * )
 */
class MailchimpCampaign extends ContentEntityBase implements MailchimpCampaignInterface {

  /**
   * {@inheritdoc}
   */
  public function getMcCampaignId() {
    return $this->get('mc_campaign_id')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getTemplate() {
    return unserialize((string) $this->get('template')->value);
  }

  /**
   * {@inheritdoc}
   */
  public function getCreated() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setMcCampaignId($mc_campaign_id) {
    $this->set('mc_campaign_id', $mc_campaign_id);
  }

  /**
   * {@inheritdoc}
   */
  public function setTemplate($template) {
    $this->set('template', serialize($template));
  }

  /**
   * {@inheritdoc}
   *
   * Define the field properties here.
   *
   * Field name, type and size determine the table structure.
   *
   * In addition, we can define how the field and its content can be manipulated
   * in the GUI. The behaviour of the widgets used can be determined here.
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = [];

    // Standard field, used as unique if primary index.
    $fields['mc_campaign_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Mailchimp Campaign ID'))
      ->setDescription(t('Mailchimp campaign ID.'))
      ->setSetting('max_length', 16);

    // Standard field, unique outside of the scope of the current project.
    $fields['uuid'] = BaseFieldDefinition::create('uuid')
      ->setLabel(t('UUID'))
      ->setDescription(t('The UUID of the campaign.'))
      ->setReadOnly(TRUE);

    $fields['template'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Template'))
      ->setDescription(t('Campaign body template.'))
      ->setSetting('case_sensitive', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string_long',
        'weight' => 0,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The Unix timestamp when the campaign was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The Unix timestamp when the campaign was most recently saved.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function access($operation, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    if ($operation == 'create') {
      return $this->entityTypeManager()
        ->getAccessControlHandler($this->entityTypeId)
        ->createAccess($this->bundle(), $account, [], $return_as_object);
    }
    return $this->entityTypeManager()
      ->getAccessControlHandler($this->entityTypeId)
      ->access($this, $operation, $account, $return_as_object);
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    // Ensure the data still exists.
    if (!isset($this->mc_data) || !$this->mc_data || !isset($this->mc_data->settings) || !$this->mc_data->settings) {
      return "";
    }
    return $this->mc_data->settings->title;
  }

  /**
   * Returns whether or not the campaign data is initialized.
   *
   * @return bool
   *   Whether or not the campaign data is initialized.
   */
  public function isInitialized() {
    return isset($this->mc_data);
  }

}
