<?php

declare(strict_types=1);

namespace Drupal\mailchimp;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\Language\ContextProvider\CurrentLanguageContext;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Access point for interacting with the Mailchimp API.
 */
final readonly class ApiService {

  public function __construct(
    private ClientFactory $mailchimpClientFactory,
    private MessengerInterface $messenger,
    private KeyValueFactoryInterface $keyvalue,
    private CacheBackendInterface $cacheMailchimp,
    private LoggerChannelInterface $loggerChannel,
    private CurrentLanguageContext $languageCurrentLanguageContext,
    private ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * Wrapper around MailchimpLists->getMergeFields().
   *
   * @param array $list_ids
   *   Array of Mailchimp audience IDs.
   * @param bool $reset
   *   Set to TRUE if mergevars should not be loaded from cache.
   *
   * @return array
   *   Struct describing mergevars for the specified audiences.
   */
  public function getMergevars(array $list_ids, bool $reset = FALSE) : array {
    $mergevars = [];
    $collection = $this->keyvalue->get('mailchimp_lists');

    if (!$reset) {
      foreach ($list_ids as $key => $list_id) {

        $state_data = $collection->get("list_{$list_id}_mergevars");
        // Get cached data and unset from our remaining audiences to query.
        if ($state_data) {
          $mergevars[$list_id] = $state_data;
          unset($list_ids[$key]);
        }
      }
    }

    // Get the uncached merge vars from Mailchimp.
    if (count($list_ids)) {
      /** @var \Mailchimp\MailchimpLists $mc_lists */
      $mc_lists = mailchimp_get_api_object('MailchimpLists');
      $list_id = NULL;

      try {
        if (!$mc_lists) {
          throw new Exception('Cannot get merge vars without Mailchimp API. Check API key has been entered.');
        }

        foreach ($list_ids as $list_id) {
          // Add default EMAIL merge var for all lists.
          $mergevars[$list_id] = [
            (object) [
              'tag' => 'EMAIL',
              'name' => t('Email Address'),
              'type' => 'email',
              'required' => TRUE,
              'default_value' => '',
              'public' => TRUE,
              'display_order' => 1,
              'options' => (object) [
                'size' => 25,
              ],
            ],
          ];

          $result = $mc_lists->getMergeFields($list_id, ['count' => 500]);

          if ($result->total_items > 0) {
            $mergevars[$list_id] = array_merge($mergevars[$list_id], $result->merge_fields);
          }

          $collection->set("list_{$list_id}_mergevars", $mergevars[$list_id]);
        }
      }

      catch (\Exception $e) {
        $this->loggerChannel->error('An error occurred requesting mergevars for list {list}. "{message}"', [
          'list' => $list_id,
          'message' => $e->getMessage(),
        ]);
      }
    }

    return $mergevars;
  }

}
