# Map Module for Drupal 11

[![Drupal 11](https://img.shields.io/badge/Drupal-11-%23007ACC?logo=drupal)](https://www.drupal.org/project/map)
[![PHP 8.0+](https://img.shields.io/badge/PHP-8.0+-%23777BB4?logo=php)](https://php.net)
[![License: GPL-2.0-or-later](https://img.shields.io/badge/License-GPL--2.0--or--later-blue.svg)](https://www.gnu.org/licenses/gpl-2.0.html)

## Overview

The **Map** module provides a custom page with an interactive Leaflet map at the `/map` URL. It features a main marker with site coordinates and supports custom markers through an administrative interface, following Drupal 11 best practices and PHP 8+ standards.

## Features

- 🗺️ Custom Leaflet map page at `/map`
- 📍 Primary marker with site coordinates
- ⚙️ Flexible map configuration
- 🎨 Customizable markers via JSON
- 🔧 Integration with Site Settings module (optional)
- 📱 Responsive design
- 🔄 Support for multiple map providers
- 🛡️ JSON input validation
- 🏗️ Modular and well-structured architecture
- ♿ Accessibility compliant
- 📊 Proper cache management

## Requirements

- Drupal 11.x
- PHP 8.0 or higher
- Leaflet module (dependency)
- Site Settings module (optional, for site coordinates)

## Installation

### Via Composer (Recommended)

```bash
composer require drupal/leaflet drupal/map
```

This will automatically install:
- The Leaflet module (required dependency)
- The Map module

### Manual Installation

1. Download the module from [drupal.org/project/map](https://www.drupal.org/project/map)
2. Extract to `/modules/contrib/map/`
3. Enable the module at `/admin/modules`

### Enable the Module

```bash
# Using Drush
drush pm:enable map

# Using Drupal Console
drupal module:install map

# Using Drupal UI
# Navigate to /admin/modules, search for "Map", and enable it
```

## Configuration

### 1. General Map Settings
Navigate to `/admin/config/system/map` to configure:
- **Map Type**: OpenStreetMap, Google Maps, Mapbox, CartoDB
- **Center Coordinates**: Latitude and longitude
- **Zoom Levels**: Default, minimum, and maximum
- **Controls**: Enable/disable zoom and attribution controls
- **Site Settings Integration**: Use coordinates from Site Settings module

### 2. Marker Configuration
Navigate to `/admin/config/system/map/markers` to add custom markers in JSON format.

Example JSON format:
```json
{
  "markers": [
    {
      "coords": [42.406246, 12.686570],
      "zoom": 18,
      "title": "Your B&B",
      "description": "Your starting point<br><br>📍 Via del B&B, Cottanello<br>☎️ +39 123 456 789<br>📧 info@bbcottanello.it",
      "icon": "home",
      "color": "#4CAF50",
      "emoji": "🏠",
      "image": "https://images.unsplash.com/photo-1566073771259-6a8506099945"
    }
  ]
}
```

### 3. Site Coordinates (Optional)
If using the Site Settings module, add to your `settings.php`:
```php
$settings['site_coordinates'] = [
  'lat' => 42.406246,
  'lng' => 12.686570,
];
```

## Permissions

The module defines two permissions:

1. **Access map page**: Allows viewing the map page
2. **Administer map configuration**: Allows configuring module settings

Assign these permissions at `/admin/people/permissions`

## Module Structure

Following Drupal best practices, the module is organized as:

```
map/
├── composer.json          # Composer configuration
├── README.md             # Project documentation
├── LICENSE.txt           # GPL-2.0-or-later license
├── CHANGELOG.txt         # Version history
├── map.info.yml          # Module metadata
├── map.libraries.yml     # Library definitions
├── map.routing.yml       # Route definitions
├── map.permissions.yml   # Permission definitions
├── map.module            # Main module file
├── config/               # Configuration files
│   ├── install/         # Default configuration
│   └── schema/          # Configuration schema
├── src/                 # PHP source code
│   ├── Form/           # Configuration forms
│   └── Controller/     # Page controller
├── js/                  # Modular JavaScript
│   ├── map-utils.js    # Utility functions
│   ├── map-markers.js  # Marker management
│   ├── map-controls.js # Map controls
│   └── map-main.js     # Main initialization
├── templates/           # Twig templates
└── tests/              # Test files
    └── src/
        └── Functional/
            └── MapSettingsTest.php
```

## JavaScript Architecture

The JavaScript is organized into reusable modules following Drupal best practices:

### `map-utils.js`
Utility functions for:
- Tile layer creation
- SVG icon generation
- Coordinate validation

### `map-markers.js`
Complete marker management:
- Site marker creation
- JSON marker parsing
- Popup and event handling

### `map-controls.js`
Custom controls:
- Home button
- Layer switcher
- Leaflet control customization

### `map-main.js`
Main initialization and event management

## JavaScript API

The module exposes the following global APIs:

```javascript
// Utilities
Drupal.MapUtils.getTileLayer(mapType);
Drupal.MapUtils.createIcon(icon, color, emoji);
Drupal.MapUtils.isValidCoordinates(lat, lng);

// Markers
Drupal.MapMarkers.createSiteMarker(markerData, map);
Drupal.MapMarkers.createCustomMarkers(markersData, map);

// Controls
Drupal.MapControls.initialize(map, config);

// Map instance access
Drupal.MapMain.getInstance();
```

## Customization

### CSS Styling
CSS styles are defined inline in the module. To customize:

1. Override CSS selectors in your theme
2. Implement `hook_page_attachments_alter()` in your theme
3. Create a sub-module that overrides the CSS

### Template Overrides
The `map-page.html.twig` template can be overridden in your theme:
- Create `templates/map-page.html.twig` in your theme
- Use `hook_theme_registry_alter()` to modify theme registry

### Adding New Map Types
1. Add the option in `MapSettingsForm.php`
2. Extend `Drupal.MapUtils.getTileLayer()` in `map-utils.js`
3. Add logic for the new provider

## Available Hooks

### `hook_map_marker_data_alter(array &$marker_data, array $context)`
Allows altering marker data before rendering.

```php
/**
 * Implements hook_map_marker_data_alter().
 */
function mymodule_map_marker_data_alter(array &$marker_data, array $context) {
  if ($context['marker_type'] === 'site') {
    $marker_data['title'] = t('Our Location');
  }
}
```

### `hook_map_config_alter(array &$map_config, array $context)`
Allows altering map configuration.

```php
/**
 * Implements hook_map_config_alter().
 */
function mymodule_map_config_alter(array &$map_config, array $context) {
  // Add custom layer
  $map_config['additionalLayers'][] = [
    'name' => t('Satellite'),
    'layer' => L.tileLayer('https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}')
  ];
}
```

## Development

### Extending the Module
To add functionality:

1. **Add new controls**:
   ```javascript
   // In your sub-module JavaScript file
   (function(Drupal) {
     Drupal.behaviors.myCustomControl = {
       attach: function(context, settings) {
         const map = Drupal.MapMain.getInstance();
         if (map) {
           // Add custom control
         }
       }
     };
   })(Drupal);
   ```

2. **Add new marker types**:
   ```php
   // In your module's hook_form_alter()
   function mymodule_form_alter(&$form, FormStateInterface $form_state, $form_id) {
     if ($form_id === 'map_marker_settings_form') {
       // Add options for new marker types
     }
   }
   ```

### Debugging
The module includes logging for:
- Zoom level changes
- Map center movements
- Coordinate validation errors

Check the JavaScript console for debug logs.

## Troubleshooting

### Map Not Displaying
1. Verify Leaflet module is installed
2. Check permissions are correctly set
3. Check JavaScript console for errors

### Markers Not Showing
1. Verify JSON format at `/admin/config/system/map/markers`
2. Ensure coordinates are valid
3. Validate JSON using an online validator

### Map Not Responsive
1. Ensure theme CSS doesn't interfere
2. Verify `#map-container` dimensions
3. Check for CSS overflow issues

## Coding Standards

This module follows:
- [Drupal Coding Standards](https://www.drupal.org/docs/develop/standards)
- [PHP 8+ Best Practices](https://www.php.net/manual/en/language.types.declarations.php)
- [JavaScript Drupal Behaviors](https://www.drupal.org/docs/drupal-apis/javascript-api/javascript-api-overview)
- [Accessibility Guidelines](https://www.drupal.org/docs/accessibility)

## Performance Considerations

1. **Marker Optimization**: For large numbers of markers (>100), consider implementing clustering
2. **Lazy Loading**: Markers are loaded eagerly; consider lazy loading for many markers
3. **Cache Strategy**: Proper cache tags and contexts are implemented for optimal performance

## Security Considerations

1. **JSON Validation**: All JSON input is validated and sanitized
2. **XSS Protection**: Output is properly escaped in templates
3. **CSRF Protection**: Forms include CSRF tokens
4. **Permission Checks**: All routes have proper permission checks

## Compatibility

- ✅ Drupal 11.x
- ✅ PHP 8.0+
- ✅ Modern browsers (Chrome, Firefox, Safari, Edge)
- ✅ Mobile responsive
- ✅ Accessibility compliant

## Known Limitations

1. **Google Maps**: Requires API key for production use
2. **Mapbox**: Requires access token
3. **Performance**: May experience issues with very large numbers of markers (>100)

## Dependencies

### Required
- [Leaflet](https://www.drupal.org/project/leaflet): Provides Leaflet library integration

### Optional
- Site Settings module for coordinate integration

## Contributing

We welcome contributions! Please:

1. Follow [Drupal coding standards](https://www.drupal.org/docs/develop/standards)
2. Include unit tests for new functionality
3. Update documentation for new features
4. Ensure backward compatibility

### Issue Queue
Report bugs and request features at:  
[drupal.org/project/issues/map](https://www.drupal.org/project/issues/map)

### Development Workflow
1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Submit a pull request

## Testing

The module includes:
- Form validation tests
- JSON parsing validation
- Coordinate validation
- Permission checks

Run tests with:
```bash
./vendor/bin/phpunit modules/contrib/map
```

## Version History

See [CHANGELOG.txt](CHANGELOG.txt) for detailed version history.

## Credits

### Maintainers
- [Your Name](https://www.drupal.org/u/yourusername)

### Sponsors
- Your Organization

### Contributors
See the [list of contributors](https://www.drupal.org/node/NID/committers) on Drupal.org.

## License

This project is licensed under the [GPL-2.0-or-later](LICENSE.txt).

## Support

For bug reports, feature requests, or support questions:

1. **Issue Queue**: [drupal.org/project/issues/map](https://www.drupal.org/project/issues/map)
2. **Documentation**: [drupal.org/project/map](https://www.drupal.org/project/map)
3. **IRC**: #drupal-contribute on irc.freenode.net
4. **Slack**: [Drupal Slack](https://www.drupal.org/slack)

---

**Note**: This module is under active development. Please check the [issue queue](https://www.drupal.org/project/issues/map) for known issues and upcoming features.
