/**
 * @file
 * Configuration Loader - Loads and processes data from Drupal configuration
 */

(function() {
  'use strict';

  const ConfigLoader = {
    config: null,
    defaults: {
      // Default values if config is not set
      business: {
        name: '[EMB] Your Business',
        location: [41.8349, 12.4700],
        address: 'Business Street, 123 - 00100 Rome, Italy',
        phone: '+39 06 1234 5678',
        email: 'info@yourbusiness.com',
        hours: {
          weekdays: '9:00 AM - 6:00 PM',
          saturday: '9:00 AM - 1:00 PM',
          sunday: 'Closed'
        }
      },
      map: {
        center: [41.9028, 12.4964],
        zoom: 12,
        type: 'openstreetmap'
      },
      transportation: {
        highwayExit: {
          enabled: true,
          name: '[EMB] Roma Nord Toll Booth',
          location: [42.0044, 12.5128],
          description: 'Highway A1 - Exit 5'
        },
        trainStation: {
          enabled: true,
          name: '[EMB] Roma Termini Station',
          location: [41.8518, 12.4940],
          description: 'Main train station'
        }
      },
      parking: {
        enabled: true,
        name: '[EMB] Visitor Parking',
        location: [41.8345, 12.4695],
        capacity: 40
      },
      attractions: [
        {
          name: '[EMB] Trattoria Romano',
          type: 'restaurant',
          location: [41.8355, 12.4710],
          description: 'Traditional Italian cuisine'
        },
        {
          name: '[EMB] EUR Shopping Center',
          type: 'shopping',
          location: [41.8320, 12.4750],
          description: 'Department stores'
        }
      ]
    },

    /**
     * Initialize Config Loader
     */
    initialize: function(drupalSettings) {
      console.log('⚙️ Initializing Config Loader...');


      console.log(drupalSettings);


      // Load configuration from Drupal settings or use defaults
      this.loadConfig(drupalSettings);

      // Validate and process config
      this.processConfig();

      console.log('✅ Config Loader initialized');
      console.log('Loaded config:', this.config);

      return this.config;
    },

    /**
     * Load configuration from Drupal settings
     */
    loadConfig: function(drupalSettings) {
      console.log('📋 Loading configuration...');

      // Check if Drupal settings have map configuration
      if (drupalSettings && drupalSettings.map) {
        console.log('Found map configuration in drupalSettings:', drupalSettings.map);
        this.config = this.mergeConfig(drupalSettings.map);
      } else {
        console.warn('No map configuration found in drupalSettings, using defaults');
        this.config = this.defaults;
      }

      // Also check for configuration passed via data attributes
      this.loadFromDataAttributes();
    },

    /**
     * Merge Drupal config with defaults
     */
    mergeConfig: function(drupalConfig) {
      const merged = JSON.parse(JSON.stringify(this.defaults));

      // Merge business info
      if (drupalConfig.business_name || drupalConfig.address) {
        merged.business = {
          name: drupalConfig.business_name || merged.business.name,
          location: this.parseCoordinates(drupalConfig.center_lat, drupalConfig.center_lng) || merged.business.location,
          address: drupalConfig.address || merged.business.address,
          phone: drupalConfig.phone || merged.business.phone,
          email: drupalConfig.email || merged.business.email,
          hours: merged.business.hours // Keep default hours
        };
      }

      // Merge street address details
      if (drupalConfig.street_address || drupalConfig.city) {
        merged.business.fullAddress = this.formatAddress(drupalConfig);
      }

      // Merge map settings
      if (drupalConfig.center_lat && drupalConfig.center_lng) {
        merged.map.center = this.parseCoordinates(drupalConfig.center_lat, drupalConfig.center_lng);
      }

      if (drupalConfig.default_zoom) {
        merged.map.zoom = parseInt(drupalConfig.default_zoom);
      }

      if (drupalConfig.map_type) {
        merged.map.type = drupalConfig.map_type;
      }

      // Merge transportation settings
      if (drupalConfig.highway_exit_enabled && drupalConfig.highway_exit_lat && drupalConfig.highway_exit_lng) {
        merged.transportation.highwayExit = {
          enabled: drupalConfig.highway_exit_enabled,
          name: drupalConfig.highway_exit_name || 'Highway Exit',
          location: this.parseCoordinates(drupalConfig.highway_exit_lat, drupalConfig.highway_exit_lng),
          description: drupalConfig.highway_exit_description || ''
        };
      }

      if (drupalConfig.train_station_enabled && drupalConfig.train_station_lat && drupalConfig.train_station_lng) {
        merged.transportation.trainStation = {
          enabled: drupalConfig.train_station_enabled,
          name: drupalConfig.train_station_name || 'Train Station',
          location: this.parseCoordinates(drupalConfig.train_station_lat, drupalConfig.train_station_lng),
          description: drupalConfig.train_station_description || ''
        };
      }

      if (drupalConfig.airport_enabled && drupalConfig.airport_lat && drupalConfig.airport_lng) {
        merged.transportation.airport = {
          enabled: drupalConfig.airport_enabled,
          name: drupalConfig.airport_name || 'Airport',
          location: this.parseCoordinates(drupalConfig.airport_lat, drupalConfig.airport_lng),
          description: drupalConfig.airport_description || ''
        };
      }

      // Merge parking settings
      if (drupalConfig.main_parking_enabled && drupalConfig.main_parking_lat && drupalConfig.main_parking_lng) {
        merged.parking = {
          enabled: drupalConfig.main_parking_enabled,
          name: drupalConfig.main_parking_name || 'Parking',
          location: this.parseCoordinates(drupalConfig.main_parking_lat, drupalConfig.main_parking_lng),
          description: drupalConfig.main_parking_description || '',
          capacity: drupalConfig.parking_capacity_total || 40,
          hours: drupalConfig.parking_hours || '24/7',
          feeType: drupalConfig.parking_fee_type || 'free'
        };
      }

      // Parse custom markers from JSON
      if (drupalConfig.custom_markers_json) {
        try {
          const customMarkers = JSON.parse(drupalConfig.custom_markers_json);
          if (Array.isArray(customMarkers)) {
            merged.customMarkers = customMarkers;
            console.log('Loaded custom markers:', customMarkers.length);
          }
        } catch (e) {
          console.error('Error parsing custom_markers_json:', e);
        }
      }

      // Parse custom parking zones from JSON
      if (drupalConfig.parking_zones_json) {
        try {
          const parkingZones = JSON.parse(drupalConfig.parking_zones_json);
          if (Array.isArray(parkingZones)) {
            merged.parkingZones = parkingZones;
            console.log('Loaded parking zones:', parkingZones.length);
          }
        } catch (e) {
          console.error('Error parsing parking_zones_json:', e);
        }
      }

      return merged;
    },

    /**
     * Load configuration from data attributes
     */
    loadFromDataAttributes: function() {
      const mapContainer = document.getElementById('map');
      if (!mapContainer) return;

      // Check for data-config attribute
      if (mapContainer.dataset.config) {
        try {
          const inlineConfig = JSON.parse(mapContainer.dataset.config);
          this.config = this.deepMerge(this.config, inlineConfig);
          console.log('Loaded config from data attribute:', inlineConfig);
        } catch (e) {
          console.error('Error parsing data-config attribute:', e);
        }
      }

      // Check for individual data attributes
      if (mapContainer.dataset.centerLat && mapContainer.dataset.centerLng) {
        this.config.map.center = this.parseCoordinates(
          mapContainer.dataset.centerLat,
          mapContainer.dataset.centerLng
        );
      }

      if (mapContainer.dataset.zoom) {
        this.config.map.zoom = parseInt(mapContainer.dataset.zoom);
      }
    },

    /**
     * Process and validate configuration
     */
    processConfig: function() {
      console.log('🔧 Processing configuration...');

      // Ensure all coordinates are valid numbers
      this.validateCoordinates();

      // Process business info for display
      this.processBusinessInfo();

      // Process attractions from custom markers
      this.processAttractions();

      // Generate formatted strings for display
      this.generateDisplayStrings();
    },

    /**
     * Validate all coordinates
     */
    validateCoordinates: function() {
      // Validate business location
      if (this.config.business.location) {
        this.config.business.location = this.ensureValidCoordinates(this.config.business.location);
      }

      // Validate map center
      if (this.config.map.center) {
        this.config.map.center = this.ensureValidCoordinates(this.config.map.center);
      }

      // Validate transportation coordinates
      if (this.config.transportation.highwayExit.location) {
        this.config.transportation.highwayExit.location =
          this.ensureValidCoordinates(this.config.transportation.highwayExit.location);
      }

      if (this.config.transportation.trainStation.location) {
        this.config.transportation.trainStation.location =
          this.ensureValidCoordinates(this.config.transportation.trainStation.location);
      }

      // Validate parking coordinates
      if (this.config.parking.location) {
        this.config.parking.location = this.ensureValidCoordinates(this.config.parking.location);
      }
    },

    /**
     * Ensure coordinates are valid [lat, lng] array
     */
    ensureValidCoordinates: function(coords) {
      if (!Array.isArray(coords) || coords.length !== 2) {
        console.warn('Invalid coordinates:', coords);
        return [41.8349, 12.4700]; // Default fallback
      }

      const [lat, lng] = coords.map(coord => parseFloat(coord));

      if (isNaN(lat) || isNaN(lng) || lat < -90 || lat > 90 || lng < -180 || lng > 180) {
        console.warn('Coordinates out of range:', coords);
        return [41.8349, 12.4700]; // Default fallback
      }

      return [lat, lng];
    },

    /**
     * Parse coordinates from strings
     */
    parseCoordinates: function(latStr, lngStr) {
      if (!latStr || !lngStr) return null;

      const lat = parseFloat(latStr);
      const lng = parseFloat(lngStr);

      if (isNaN(lat) || isNaN(lng)) {
        console.warn('Invalid coordinate strings:', latStr, lngStr);
        return null;
      }

      return [lat, lng];
    },

    /**
     * Process business information
     */
    processBusinessInfo: function() {
      // Create formatted address string
      if (!this.config.business.fullAddress && this.config.business.address) {
        this.config.business.fullAddress = this.config.business.address;
      }

      // Create HTML for popup
      this.config.business.popupHTML = this.generateBusinessPopupHTML();

      // Create HTML for info panel
      this.config.business.infoHTML = this.generateBusinessInfoHTML();
    },

    /**
     * Process attractions from custom markers
     */
    processAttractions: function() {
      if (!this.config.attractions) {
        this.config.attractions = [];
      }

      // Add custom markers as attractions
      if (this.config.customMarkers && Array.isArray(this.config.customMarkers)) {
        this.config.customMarkers.forEach(marker => {
          if (marker.name && marker.lat && marker.lng) {
            this.config.attractions.push({
              name: marker.name,
              type: marker.icon || 'attraction',
              location: [parseFloat(marker.lat), parseFloat(marker.lng)],
              description: marker.description || '',
              icon: marker.icon,
              color: marker.color
            });
          }
        });
      }

      // Group attractions by type
      this.config.attractionsByType = this.groupAttractionsByType();
    },

    /**
     * Group attractions by type/category
     */
    groupAttractionsByType: function() {
      const groups = {
        restaurant: [],
        shopping: [],
        cultural: [],
        hotel: [],
        other: []
      };

      this.config.attractions.forEach(attraction => {
        const type = attraction.type || 'other';

        if (groups[type]) {
          groups[type].push(attraction);
        } else {
          groups.other.push(attraction);
        }
      });

      return groups;
    },

    /**
     * Generate formatted address string
     */
    formatAddress: function(addressConfig) {
      const parts = [];

      if (addressConfig.street_address) {
        parts.push(addressConfig.street_address);
      }

      if (addressConfig.street_address2) {
        parts.push(addressConfig.street_address2);
      }

      const cityParts = [];
      if (addressConfig.city) cityParts.push(addressConfig.city);
      if (addressConfig.state_province) cityParts.push(addressConfig.state_province);
      if (addressConfig.postal_code) cityParts.push(addressConfig.postal_code);

      if (cityParts.length > 0) {
        parts.push(cityParts.join(', '));
      }

      if (addressConfig.country && addressConfig.country !== 'IT') {
        // Add country if not Italy (default)
        parts.push(this.getCountryName(addressConfig.country));
      }

      return parts.join('<br>');
    },

    /**
     * Get country name from code
     */
    getCountryName: function(countryCode) {
      const countries = {
        'IT': 'Italy',
        'US': 'United States',
        'GB': 'United Kingdom',
        'DE': 'Germany',
        'FR': 'France',
        'ES': 'Spain',
        'CH': 'Switzerland'
      };

      return countries[countryCode] || countryCode;
    },

    /**
     * Generate business popup HTML
     */
    generateBusinessPopupHTML: function() {
      const business = this.config.business;

      return `
        <div class="business-popup">
          <h3 style="margin: 0 0 10px 0; color: #2c3e50;">
            <i class="fas fa-building"></i> ${business.name}
          </h3>
          ${business.fullAddress ? `<p style="margin: 5px 0;"><strong>📍 Address:</strong><br>${business.fullAddress}</p>` : ''}
          ${business.phone ? `<p style="margin: 5px 0;"><strong>📞 Phone:</strong> ${business.phone}</p>` : ''}
          ${business.email ? `<p style="margin: 5px 0;"><strong>✉️ Email:</strong> ${business.email}</p>` : ''}
          ${business.hours ? `
            <p style="margin: 5px 0;">
              <strong>🕒 Hours:</strong><br>
              [EMB] Mon-Fri: ${business.hours.weekdays}<br>
              Sat: ${business.hours.saturday}<br>
              Sun: ${business.hours.sunday}
            </p>
          ` : ''}
        </div>
      `;
    },

    /**
     * Generate business info HTML for panel
     */
    generateBusinessInfoHTML: function() {
      const business = this.config.business;

      return `
        <div class="location-info">
          <p>${business.name} is centrally located with easy access and modern facilities.</p>
          <div class="contact-details">
            ${business.fullAddress ? `<p><strong>📍 Address:</strong><br>${business.fullAddress}</p>` : ''}
            ${business.phone ? `<p><strong>📞 Phone:</strong> ${business.phone}</p>` : ''}
            ${business.email ? `<p><strong>✉️ Email:</strong> ${business.email}</p>` : ''}
            ${business.hours ? `
              <p><strong>🕒 Hours:</strong><br>
             [EMB] Mon-Fri: ${business.hours.weekdays}<br>
              Sat: ${business.hours.saturday}<br>
              Sun: ${business.hours.sunday}</p>
            ` : ''}
          </div>
        </div>
      `;
    },

    /**
     * Generate attractions HTML for panel
     */
    generateAttractionsHTML: function() {
      const groups = this.config.attractionsByType;
      let html = '<div class="attractions-info">';

      html += '<p><strong>Discover what\'s around our location:</strong></p>';

      // Restaurants
      if (groups.restaurant.length > 0) {
        html += this.generateAttractionCategoryHTML('restaurant', 'fa-utensils', 'Restaurants & Cafés', groups.restaurant);
      }

      // Shopping
      if (groups.shopping.length > 0) {
        html += this.generateAttractionCategoryHTML('shopping', 'fa-shopping-bag', 'Shopping', groups.shopping);
      }

      // Cultural
      if (groups.cultural.length > 0) {
        html += this.generateAttractionCategoryHTML('cultural', 'fa-landmark', 'Cultural Sites', groups.cultural);
      }

      // Hotels
      if (groups.hotel.length > 0) {
        html += this.generateAttractionCategoryHTML('hotel', 'fa-hotel', 'Accommodation', groups.hotel);
      }

      // Other
      if (groups.other.length > 0) {
        html += this.generateAttractionCategoryHTML('other', 'fa-star', 'Nearby Points of Interest', groups.other);
      }

      html += '</div>';
      return html;
    },

    /**
     * Generate HTML for an attraction category
     */
    generateAttractionCategoryHTML: function(type, icon, title, attractions) {
      let html = `
        <div class="attraction-category attraction-${type}">
          <h4><i class="fas ${icon}"></i> ${title}</h4>
          <ul>
      `;

      attractions.forEach(attraction => {
        html += `<li><strong>${attraction.name}</strong> - ${attraction.description || 'Nearby location'}</li>`;
      });

      html += `
          </ul>
        </div>
      `;

      return html;
    },

    /**
     * Generate display strings
     */
    generateDisplayStrings: function() {
      // Business title for header
      this.config.display = {
        businessTitle: this.config.business.name,
        businessSubtitle: 'Discover our location and services',

        // Category titles
        categories: {
          where: 'Our Location',
          car: 'By Car',
          publicTransport: 'Public Transport',
          parking: 'Parking',
          attractions: 'Nearby Attractions'
        },

        // Transportation descriptions
        transportation: {
          car: this.generateCarDescription(),
          publicTransport: this.generatePublicTransportDescription()
        }
      };
    },

    /**
     * Generate car route description
     */
    generateCarDescription: function() {
      const exit = this.config.transportation.highwayExit;

      if (exit.enabled) {
        return `
          <div class="transport-info">
            <p><strong>Driving Directions:</strong></p>
            <ol>
              <li>[EMB] Take Highway A1</li>
              <li>Exit at '${exit.name}'</li>
              <li>Follow signs to ${this.config.business.name}</li>
            </ol>
            ${exit.description ? `<p>${exit.description}</p>` : ''}
            <p><strong>Parking:</strong> Available on site</p>
          </div>
        `;
      }

      return `
        <div class="transport-info">
          <p><strong>Driving Directions:</strong></p>
          <p>Easy access via major highways. Ample parking available.</p>
        </div>
      `;
    },

    /**
     * Generate public transport description
     */
    generatePublicTransportDescription: function() {
      const station = this.config.transportation.trainStation;

      if (station.enabled) {
        return `
          <div class="transport-info">
            <p><strong>Public Transport:</strong></p>
            <ul>
              <li>Train to ${station.name}</li>
              <li>Local bus connections available</li>
            </ul>
            ${station.description ? `<p>${station.description}</p>` : ''}
            <p><strong>Convenient access</strong> from city center</p>
          </div>
        `;
      }

      return `
        <div class="transport-info">
          <p><strong>Public Transport:</strong></p>
          <p>[EMB] Well connected via public transportation networks.</p>
        </div>
      `;
    },

    /**
     * Generate parking info HTML
     */
    generateParkingHTML: function() {
      const parking = this.config.parking;

      if (parking.enabled) {
        return `
          <div class="parking-info">
            <p><strong>Parking Facilities:</strong></p>
            <ul>
              <li><strong>${parking.name}:</strong> ${parking.capacity} spaces available</li>
              <li><strong>Hours:</strong> ${parking.hours}</li>
              <li><strong>Fee:</strong> ${parking.feeType === 'free' ? 'Free parking' : 'Paid parking available'}</li>
            </ul>
            ${parking.description ? `<p>${parking.description}</p>` : ''}
          </div>
        `;
      }

      return `
        <div class="parking-info">
          <p><strong>Parking:</strong> [EMB] Limited street parking available nearby.</p>
        </div>
      `;
    },

    /**
     * Get configuration value by path
     */
    get: function(path, defaultValue = null) {
      const keys = path.split('.');
      let value = this.config;

      for (const key of keys) {
        if (value && typeof value === 'object' && key in value) {
          value = value[key];
        } else {
          return defaultValue;
        }
      }

      return value;
    },

    /**
     * Get business configuration
     */
    getBusinessConfig: function() {
      return this.config.business;
    },

    /**
     * Get map configuration
     */
    getMapConfig: function() {
      return this.config.map;
    },

    /**
     * Get attractions configuration
     */
    getAttractionsConfig: function() {
      return this.config.attractions;
    },

    /**
     * Get attractions grouped by type
     */
    getAttractionsByType: function() {
      return this.config.attractionsByType || {};
    },

    /**
     * Get transportation configuration
     */
    getTransportationConfig: function() {
      return this.config.transportation;
    },

    /**
     * Get parking configuration
     */
    getParkingConfig: function() {
      return this.config.parking;
    },

    /**
     * Get display strings
     */
    getDisplayConfig: function() {
      return this.config.display || {};
    },

    /**
     * Get custom markers
     */
    getCustomMarkers: function() {
      return this.config.customMarkers || [];
    },

    /**
     * Get parking zones
     */
    getParkingZones: function() {
      return this.config.parkingZones || [];
    },

    /**
     * Deep merge two objects
     */
    deepMerge: function(target, source) {
      const output = Object.assign({}, target);

      if (this.isObject(target) && this.isObject(source)) {
        Object.keys(source).forEach(key => {
          if (this.isObject(source[key])) {
            if (!(key in target)) {
              Object.assign(output, { [key]: source[key] });
            } else {
              output[key] = this.deepMerge(target[key], source[key]);
            }
          } else {
            Object.assign(output, { [key]: source[key] });
          }
        });
      }

      return output;
    },

    /**
     * Check if value is an object
     */
    isObject: function(item) {
      return item && typeof item === 'object' && !Array.isArray(item);
    },

    /**
     * Debug: Log full configuration
     */
    debug: function() {
      console.log('=== CONFIGURATION DEBUG ===');
      console.log('Full config:', JSON.parse(JSON.stringify(this.config)));
      console.log('Business:', this.getBusinessConfig());
      console.log('Map:', this.getMapConfig());
      console.log('Attractions:', this.getAttractionsConfig());
      console.log('Transportation:', this.getTransportationConfig());
      console.log('Parking:', this.getParkingConfig());
      console.log('Custom markers:', this.getCustomMarkers().length);
      console.log('Parking zones:', this.getParkingZones().length);
    }
  };

  // Export to global scope
  window.ConfigLoader = ConfigLoader;

  // Debug function
  window.debugConfig = function() {
    if (window.ConfigLoader) {
      window.ConfigLoader.debug();
    } else {
      console.error('ConfigLoader not available');
    }
  };

})();
