(function ($, Drupal, once, drupalSettings) {
  'use strict';

  Drupal.behaviors.mapLeafletPositionSelector = {
    attach: function (context, settings) {
      // Use drupalSettings instead of Drupal.settings
      var mapSettings = drupalSettings.map || {};
      var debug = mapSettings.debug || false;

      // Debug log to check if script is loading
      if (debug) {
        console.log('Leaflet Position Selector: Attaching behavior');
        console.log('drupalSettings:', drupalSettings);
      }

      // Check if Leaflet is loaded
      if (typeof L === 'undefined') {
        console.error('Leaflet library not loaded! Make sure the Leaflet module is enabled.');
        // Try to load Leaflet dynamically if not available
        if (debug) {
          console.log('Attempting to detect Leaflet loading state...');
        }
        return;
      }

      if (debug) {
        console.log('Leaflet library is loaded:', typeof L);
      }

      // Initialize all leaflet maps using once()
      $(once('leaflet-position-selector', '.leaflet-map', context)).each(function () {
        var $mapContainer = $(this);
        var mapId = this.id;

        if (debug) {
          console.log('Found map element:', mapId);
          console.log('Map element:', this);
        }

        // Make sure the element has dimensions
        if ($mapContainer.height() === 0) {
          $mapContainer.css('height', '300px');
        }

        var latFieldId, lngFieldId, displayId;

        // Determine which fields this map controls
        switch (mapId) {
          case 'custom-coordinates-map':
            latFieldId = 'custom-coordinates-lat';
            lngFieldId = 'custom-coordinates-lng';
            displayId = 'custom-coordinates-display';
            break;
          case 'highway-exit-map':
            latFieldId = 'highway-exit-lat';
            lngFieldId = 'highway-exit-lng';
            displayId = 'highway-exit-coordinates-display';
            break;
          case 'train-station-map':
            latFieldId = 'train-station-lat';
            lngFieldId = 'train-station-lng';
            displayId = 'train-station-coordinates-display';
            break;
          case 'airport-map':
            latFieldId = 'airport-lat';
            lngFieldId = 'airport-lng';
            displayId = 'airport-coordinates-display';
            break;
          case 'main-parking-map':
            latFieldId = 'main-parking-lat';
            lngFieldId = 'main-parking-lng';
            displayId = 'main-parking-coordinates-display';
            break;
          default:
            if (debug) {
              console.warn('Unknown map ID:', mapId);
            }
            // Try to guess field IDs based on map ID pattern
            var baseId = mapId.replace('-map', '');
            latFieldId = baseId + '-lat';
            lngFieldId = baseId + '-lng';
            displayId = baseId + '-coordinates-display';
            if (debug) {
              console.log('Guessing field IDs:', latFieldId, lngFieldId, displayId);
            }
        }

        // Check if fields exist
        var $latField = $('#' + latFieldId);
        var $lngField = $('#' + lngFieldId);
        var $display = $('#' + displayId);

        if (debug) {
          console.log('Looking for fields:', latFieldId, lngFieldId, displayId);
          console.log('Found lat field:', $latField.length);
          console.log('Found lng field:', $lngField.length);
          console.log('Found display:', $display.length);
        }

        if (!$latField.length || !$lngField.length) {
          console.warn('Coordinate fields not found for map:', mapId);
          // Try to find fields by name instead
          $latField = $('[name="' + latFieldId.replace('custom-coordinates-', 'custom_lat') + '"]');
          $lngField = $('[name="' + lngFieldId.replace('custom-coordinates-', 'custom_lng') + '"]');

          if (!$latField.length || !$lngField.length) {
            console.error('Could not find coordinate fields for map:', mapId);
            return;
          }
        }

        // Get initial coordinates from hidden fields
        var initialLat = parseFloat($latField.val());
        var initialLng = parseFloat($lngField.val());

        // Default to Rome if no coordinates
        if (isNaN(initialLat)) initialLat = 41.9028;
        if (isNaN(initialLng)) initialLng = 12.4964;

        if (debug) {
          console.log('Initial coordinates:', initialLat, initialLng);
        }

        try {
          // Create the map
          var map = L.map(mapId, {
            center: [initialLat, initialLng],
            zoom: 14,
            zoomControl: true,
            attributionControl: false
          });

          // Add OpenStreetMap tile layer
          L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
            maxZoom: 19
          }).addTo(map);

          // Add attribution control
          L.control.attribution({
            position: 'bottomright'
          }).addTo(map);

          // Create a marker
          var marker = L.marker([initialLat, initialLng], {
            draggable: true,
            title: 'Drag me or click map to set location'
          }).addTo(map);

          // Bind popup to marker
          marker.bindPopup('Selected location<br>' + initialLat.toFixed(6) + ', ' + initialLng.toFixed(6)).openPopup();

          // Update coordinates when marker is dragged
          marker.on('dragend', function (e) {
            var position = marker.getLatLng();
            updateCoordinates(position.lat, position.lng);
            marker.getPopup().setContent('Selected location<br>' + position.lat.toFixed(6) + ', ' + position.lng.toFixed(6));
            marker.openPopup();
          });

          // Update coordinates when map is clicked
          map.on('click', function (e) {
            marker.setLatLng(e.latlng);
            updateCoordinates(e.latlng.lat, e.latlng.lng);
            marker.getPopup().setContent('Selected location<br>' + e.latlng.lat.toFixed(6) + ', ' + e.latlng.lng.toFixed(6));
            marker.openPopup();
          });

          // Function to update coordinates in hidden fields and display
          function updateCoordinates(lat, lng) {
            $latField.val(lat.toFixed(6));
            $lngField.val(lng.toFixed(6));
            if ($display.length) {
              $display.text(lat.toFixed(6) + ', ' + lng.toFixed(6));
            }
          }

          // Also update display when marker is manually dragged
          marker.on('drag', function (e) {
            var position = marker.getLatLng();
            if ($display.length) {
              $display.text(position.lat.toFixed(6) + ', ' + position.lng.toFixed(6));
            }
          });

          // Force map to render after a short delay
          setTimeout(function() {
            map.invalidateSize();
            if (debug) {
              console.log('Map invalidateSize called for:', mapId);
            }
          }, 100);

          // Also invalidate size when tab becomes visible
          $('.vertical-tabs__pane').on('click', function() {
            setTimeout(function() {
              map.invalidateSize();
            }, 50);
          });

          if (debug) {
            console.log('Map initialized successfully:', mapId);
          }

        } catch (error) {
          console.error('Error initializing map ' + mapId + ':', error);
          $mapContainer.html('<div class="map-error" style="color: red; padding: 20px; text-align: center; border: 2px solid red;">Error loading map: ' + error.message +
            '<br>Please check browser console for details.</div>');
        }
      });
    }
  };

})(jQuery, Drupal, once, drupalSettings);
