/**
 * @file
 * Business Location Map - Core Module
 */

(function (Drupal, drupalSettings) {
  'use strict';

  console.log('=== Map Core Module Loading ===');
  // console.log(drupalSettings);

  Drupal.behaviors.businessMap = {
    mapInitialized: false,
    mapInstance: null,
    currentMarkers: [],

    // Template system for infoPanel
    templates: {
      /**
       * Generic template renderer
       */
      render: function(templateName, data) {
        const template = this[templateName];
        if (typeof template === 'function') {
          return template(data);
        }
        return `<div class="error-template">Template "${templateName}" not found</div>`;
      },

      /**
       * WHERE Template - Business Location
       */
      where: function(data) {

        // console.log(data);
        const address = data.address || {};
        // const countryName = this.getCountryName(address.country);
        const countryName = address.country;

        return `
          <div class="location-info">
            <p>${Drupal.t('Our headquarters is located in @city, offering easy access and modern facilities.', {
              '@city': address.city || ''
            })}</p>
            <div class="contact-details">
              <p><strong>📍 ${Drupal.t('Address:')}</strong><br>
                ${address.street || ''}<br>
                ${address.street2 ? address.street2 + '<br>' : ''}
                ${address.postalCode || ''} ${address.city || ''}<br>
                ${address.state ? address.state + '<br>' : ''}
                ${countryName}
              </p>
              ${address.notes ? `<p><strong>📋 ${Drupal.t('Access Notes:')}</strong> ${address.notes}</p>` : ''}
              <p><strong>📞 ${Drupal.t('Phone:')}</strong> ${address.country}</p>
              <p><strong>✉️ ${Drupal.t('Email:')}</strong> ${address.city}</p>
              <p><strong>🕒 ${Drupal.t('Hours:')}</strong><br>${address.country}
              </p>
            </div>
          </div>
        `;
      },

      /**
       * CAR Template - Driving Directions
       */
      car: function(data) {
        const transportation = data.transportation || {};
        const parking = data.parking || {};
        const highwayExit = transportation.highwayExit || {};

        return `
          <div class="transport-info">
            <h3><strong>${Drupal.t('Driving Directions:')}</strong></h3>
            ${highwayExit.name ? `<h4>${highwayExit.name}</h4>` : ''}
            ${highwayExit.description ? `<p>${highwayExit.description}</p>` : ''}
            ${transportation.carRental ? `
              <div class="car-rental-info">
                <h4><i class="fas fa-car"></i> ${Drupal.t('Car Rental:')}</h4>
                <p>${transportation.carRental}</p>
              </div>
            ` : ''}
          </div>
        `;
      },

      /**
       * PUBLIC TRANSPORT Template
       */
      publicTransport: function(data) {
        const transportation = data.transportation || {};

        let transportOptions = '';
        if (transportation.transportModeOptions) {
          transportOptions = Object.values(transportation.transportModeOptions)
            .map(option => `<li>${Drupal.t(option)}</li>`)
            .join('');
        }

        return `
          <div class="transport-info">
            <p><strong>${Drupal.t('Available Transport Modes:')}</strong></p>
            ${transportOptions ? `<ul>${transportOptions}</ul>` : ''}

            ${transportation.airport ? `
              <div class="airport-info">
                <h4><i class="fas fa-plane"></i> ${Drupal.t('Nearest Airport:')}</h4>
                <p><strong>${transportation.airport.name || ''}</strong></p>
                ${transportation.airport.description ? `<p>${transportation.airport.description}</p>` : ''}
              </div>
            ` : ''}

            ${transportation.busStation && transportation.busStation.enabled ? `
              <div class="bus-station-info">
                <h4><i class="fas fa-bus"></i> ${Drupal.t('Bus Station:')}</h4>
                <p>${transportation.busStation.name || ''}</p>
              </div>
            ` : ''}

            <div class="default-transport">
              <p><strong>${Drupal.t('Recommended Transport:')}</strong>
                ${transportation.transportModeOptions ?
                  Drupal.t(transportation.transportModeOptions[transportation.defaultTransportMode] || '') : ''}
              </p>
            </div>
          </div>
        `;
      },

      /**
       * PARKING Template
       */
      parking: function(data) {
        const parking = data.parking || {};
        const capacity = parking.capacity || {};

        let capacityList = '';
        if (capacity.total) {
          capacityList = `
            <li><strong>${Drupal.t('Total Spaces:')}</strong> ${capacity.total}</li>
            ${capacity.disabled ? `<li><strong>${Drupal.t('Disabled Spaces:')}</strong> ${capacity.disabled}</li>` : ''}
            ${capacity.family ? `<li><strong>${Drupal.t('Family Spaces:')}</strong> ${capacity.family}</li>` : ''}
            ${capacity.ev ? `<li><strong>${Drupal.t('EV Charging:')}</strong> ${capacity.ev}</li>` : ''}
          `;
        }

        let parkingZones = '';
        if (parking.parkingZones && Array.isArray(parking.parkingZones)) {
          parkingZones = parking.parkingZones.map(zone => `
            <div class="parking-zone">
              <h4>${zone.name || ''}</h4>
              ${zone.description ? `<p>${zone.description}</p>` : ''}
              ${zone.spaces ? `<p><strong>${Drupal.t('Spaces:')}</strong> ${zone.spaces}</p>` : ''}
            </div>
          `).join('');
        }

        return `
          <div class="parking-info">
            ${capacityList ? `
              <p><strong>${Drupal.t('Parking Capacity:')}</strong></p>
              <ul>${capacityList}</ul>
            ` : ''}

            ${parking.hours ? `<p><strong>${Drupal.t('Hours:')}</strong> ${parking.hours}</p>` : ''}

            ${parking.feeType ? `
              <p><strong>${Drupal.t('Fee Type:')}</strong>
                ${parking.parkingFeeTypeOptions ?
                  Drupal.t(parking.parkingFeeTypeOptions[parking.feeType] || parking.feeType) :
                  Drupal.t(parking.feeType)}
              </p>
            ` : ''}

            ${parkingZones ? `
              <div class="parking-zones">
                <h4>${Drupal.t('Parking Zones:')}</h4>
                ${parkingZones}
              </div>
            ` : ''}

            ${parking.mainParking ? `
              <div class="main-parking">
                <h4>${Drupal.t('Main Parking:')}</h4>
                <p><strong>${parking.mainParking.name || ''}</strong></p>
                ${parking.mainParking.description ? `<p>${parking.mainParking.description}</p>` : ''}
              </div>
            ` : ''}
          </div>
        `;
      },

      attractions: function(data) {
        // Custom markers from drupalSettings
        const nearbyMarkers = data.nearbyMarkers || [];

        // Define the helper functions
        const getCategoryIcon = (category) => {
          const icons = {
            // museum: '🏛️',
            // park: '🌳',
            // restaurant: '🍽️',
            // hotel: '🏨',
            // shop: '🛍️',
          'restaurant': '<i class="fas fa-utensils"></i>',
          'shopping': '<i class="fas fa-shopping-bag"></i>',
          'culture': '<i class="fas fa-landmark"></i>',
          'hotel': '<i class="fas fa-hotel"></i>',
          'park': '<i class="fas fa-tree"></i>',
          'museum': '<i class="fas fa-museum"></i>',
          'cafe': '<i class="fas fa-coffee"></i>',
          'bar': '<i class="fas fa-glass-cheers"></i>'
            // default: '📍'
          };
          return icons[category] || icons.default;
        };
        //

        const getCategoryName = (category) => {
          const categoryNames = {
            // museum: Drupal.t('Museums'),
            // park: Drupal.t('Parks & Gardens'),
            // restaurant: Drupal.t('Restaurants'),
            // hotel: Drupal.t('Hotels'),
            // shop: Drupal.t('Shopping'),
                'restaurant': Drupal.t('Restaurants & Cafés'),
                'shopping': Drupal.t('Shopping'),
                'culture': Drupal.t('Cultural Sites'),
                'hotel': Drupal.t('Accommodation'),
                'park': Drupal.t('Parks & Recreation'),
                'museum': Drupal.t('Museums'),
                'cafe': Drupal.t('Cafés'),
                'bar': Drupal.t('Bars & Lounges')
            // default: category
          };
          return categoryNames[category] || categoryNames.default;
        };

        // getCategoryIcon: function(category) {
        //   const icons = {
        //   };
        //   return icons[category] || '<i class="fas fa-star"></i>';
        // },

        // getCategoryName: function(category) {
        //   const names = {
        //   };
        //   return names[category] || Drupal.t(category);
        // }

        let attractionsHtml = '';

        if (nearbyMarkers.length > 0) {
          const categories = {};

          // Group by category
          nearbyMarkers.forEach(marker => {
            if (marker.category && marker.enabled !== false) {
              if (!categories[marker.category]) {
                categories[marker.category] = [];
              }
              categories[marker.category].push(marker);
            }
          });

          // Generate HTML for each category
          attractionsHtml = Object.entries(categories).map(([category, items]) => {
            const icon = getCategoryIcon(category); // ✅ Now defined
            const itemsHtml = items.map(item => `
              <li>
                <strong>${item.name || ''}</strong>
                ${item.description ? `- ${item.description}` : ''}
                ${item.distance ? `<br><small>(${item.distance})</small>` : ''}
              </li>
            `).join('');

            return `
              <div class="attraction-category">
                <h4>${icon} ${getCategoryName(category)}</h4>
                <ul>${itemsHtml}</ul>
              </div>
            `;
          }).join('');
        }

        return `
          <div class="attractions-info">
            <p><strong>${Drupal.t('Discover what\'s around our location:')}</strong></p>
            ${attractionsHtml}
          </div>
        `;
      },



      /**
       * MY LOCATION Template
       */
      myLocation: function(data) {
        const position = data.position || {};

        return `
          <div class="location-info">
            <p><strong>${Drupal.t('Your current location has been found!')}</strong></p>
            <p>${Drupal.t('Latitude:')} ${position.latitude ? position.latitude.toFixed(6) : ''}</p>
            <p>${Drupal.t('Longitude:')} ${position.longitude ? position.longitude.toFixed(6) : ''}</p>
            ${position.accuracy ? `<p>${Drupal.t('Accuracy:')} ${Math.round(position.accuracy)} ${Drupal.t('meters')}</p>` : ''}
            <p>${Drupal.t('Click "Route to Business" on the map marker to get directions.')}</p>
          </div>
        `;
      },

      /**
       * Helper functions for templates
       */
      getCountryName: function(countryCode) {
        const countries = {
          'CH': Drupal.t('Switzerland'),
          'IT': Drupal.t('Italy'),
          'US': Drupal.t('United States'),
          'GB': Drupal.t('United Kingdom'),
          'DE': Drupal.t('Germany')
        };
        return countries[countryCode] || countryCode || '';
      },

      getPhoneNumber: function(countryCode) {

        return 'Get numbers from B.I. module';

        // const phones = {
        //   'CH': '+41 22 123 45 67',
        //   'IT': '+39 06 1234 5678',
        //   'US': '+1 (555) 123-4567'
        // };
        // return phones[countryCode] || '+39 06 1234 5678';
      },

      // getEmail: function(city) {
      //   if (city) {
      //     return `info@${city.toLowerCase().replace(/\s+/g, '')}.com`;
      //   }
      //   return 'info@business.com';
      // },

      // getBusinessHours: function(countryCode) {
      //   const hours = {
      //     'CH': Drupal.t('Mon-Fri: 8:00-17:30, Sat: 8:00-12:00'),
      //     'IT': Drupal.t('Mon-Fri: 9:00-18:00, Sat: 9:00-13:00'),
      //     'US': Drupal.t('Mon-Fri: 8:30-17:30, Sat: 9:00-13:00')
      //   };
      //   return hours[countryCode] || Drupal.t('Mon-Fri: 9:00-18:00, Sat: 9:00-13:00');
      // },


    },

    attach: function (context, settings) {
      if (this.mapInitialized) {
        return;
      }

      // Wait for DOM
      if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
          this.initializeMap();
        });
      } else {
        this.initializeMap();
      }
    },

    /**
     * Initialize the map
     */
    initializeMap: function() {
      const mapContainer = document.getElementById('map');
      if (!mapContainer) {
        console.error('❌ Map container not found');
        return;
      }

      // Check Leaflet
      if (typeof L === 'undefined') {
        console.error('❌ Leaflet not loaded');
        return;
      }

      try {
        // console.log('🗺️ Creating map...');

        // Get initial location from drupalSettings
        const initialLocation = drupalSettings?.map?.config?.center || [41.9028, 12.4964];
        const initialZoom = drupalSettings?.map?.config?.minZoom || 12;

        // Create map
        this.mapInstance = L.map(mapContainer, {
          zoomControl: false,
          scrollWheelZoom: true
        }).setView(initialLocation, initialZoom);

        // console.log('✅ Map created');

        // Add tile layer based on map type
        const mapType = drupalSettings?.map?.config?.mapType || 'openstreetmap';
        this.addTileLayer(mapType);

        // Add zoom control
        L.control.zoom({
          position: 'bottomright'
        }).addTo(this.mapInstance);

        // Add business marker using coordinates from settings
        this.addBusinessMarker();

        // Initialize Route Manager if available
        if (typeof window.RouteManager !== 'undefined') {
          // console.log('🚀 Initializing Route Manager...');
          window.RouteManager.initialize(this.mapInstance);
        }

        // Setup UI
        this.setupUI();

        // Handle window resize
        window.addEventListener('resize', () => {
          this.mapInstance.invalidateSize();
        });

        // Force initial resize
        setTimeout(() => {
          this.mapInstance.invalidateSize();
        }, 100);

        this.mapInitialized = true;
        console.log('✅ Map system fully initialized');

      } catch (error) {
        console.error('❌ Error initializing map:', error);
        console.error(error.stack);
      }
    },

    /**
     * Add tile layer based on map type
     */
    addTileLayer: function(mapType) {
      const tileLayers = {
        'openstreetmap': {
          url: 'https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png',
          attribution: '<a href="https://www.drupal.org/u/afagioli">afagioli</a> | © <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
          maxZoom: 19
        },
        'google': {
          url: 'https://mt1.google.com/vt/lyrs=m&x={x}&y={y}&z={z}',
          attribution: '© Google Maps',
          maxZoom: 20
        }
      };

      const config = tileLayers[mapType] || tileLayers.openstreetmap;
      L.tileLayer(config.url, {
        attribution: config.attribution,
        maxZoom: drupalSettings?.map?.config?.maxZoom || config.maxZoom
      }).addTo(this.mapInstance);
    },

    /**
     * Add business marker
     */
    addBusinessMarker: function() {
      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      const businessMarker = L.marker(businessLocation, {
        icon: L.divIcon({
          className: 'marker-pulse',
          html: '<i class="fas fa-building"></i>',
          iconSize: [26, 26]
        })
      }).addTo(this.mapInstance);

      return businessMarker;
    },

    /**
     * Setup UI and event handlers
     */
    setupUI: function() {
      // console.log('🎨 Setting up UI...');

      // Navigation buttons
      const navButtons = document.querySelectorAll('.nav-button');
      // console.log('Found nav buttons:', navButtons.length);

      navButtons.forEach(button => {
        button.addEventListener('click', (e) => {
          e.preventDefault();
          e.stopPropagation();

          const target = button.dataset.target;
          // console.log('Nav button clicked:', target);

          this.handleNavigation(target);
        });
      });

      // Location button
      const locationBtn = document.getElementById('locationBtn');
      if (locationBtn) {
        locationBtn.addEventListener('click', (e) => {
          e.preventDefault();
          this.handleMyLocation();
        });
      }

      // Close info panel
      const closeBtn = document.getElementById('closeInfoPanel');
      if (closeBtn) {
        closeBtn.addEventListener('click', () => {
          document.getElementById('infoPanel').classList.add('hidden');
        });
      }

      // Share button
      const shareBtn = document.getElementById('shareBtn');
      if (shareBtn) {
        shareBtn.addEventListener('click', (e) => {
          e.preventDefault();
          this.shareLocation();
        });
      }

      // Home button
      const homeBtn = document.getElementById('homeBtn');
      if (homeBtn) {
        homeBtn.addEventListener('click', (e) => {
          e.preventDefault();
          window.location.href = '/';
        });
      }

      // Show initial category
      setTimeout(() => {
        this.handleNavigation('where');
      }, 500);
    },

    /**
     * Handle navigation button clicks
     */
    handleNavigation: function(category) {
      // console.log('📍 Navigation to:', category);

      // Update active button
      document.querySelectorAll('.nav-button').forEach(btn => {
        btn.classList.remove('active');
        if (btn.dataset.target === category) {
          btn.classList.add('active');
        }
      });

      // Update info panel using template system
      const infoPanel = document.getElementById('infoPanel');
      const infoTitle = document.getElementById('infoTitle');
      const infoContent = document.getElementById('infoContent');

      if (infoPanel) infoPanel.classList.remove('hidden');

      // Prepare data for template
      const templateData = {
        address: drupalSettings?.map?.config?.address,
        transportation: drupalSettings?.map?.config?.transportation,
        parking: drupalSettings?.map?.config?.parking,


        // customMarkers: drupalSettings?.map?.config?.customMarkers,
        nearbyMarkers: drupalSettings?.map?.config?.nearbyMarkers,

      };

      // Set title and content based on category
      switch(category) {
        case 'where':
          if (infoTitle) infoTitle.innerHTML = "<i class=\"fas fa-building\"></i> " + Drupal.t('Our Location');
          // if (infoContent) infoContent.innerHTML = drupalSettings?.map?.business_identity_settings?.where_are_we;
          if (infoContent) infoContent.innerHTML = this.templates.render('where', templateData);

          this.showWhereCategory();
          break;

        case 'car':
          if (infoTitle) infoTitle.innerHTML = "<i class=\"fas fa-car icon-car\" aria-hidden=\"true\"></i>" + Drupal.t("By Car");
          if (infoContent) infoContent.innerHTML = this.templates.render('car', templateData);
          this.showCarCategory();
          break;

        case 'publicTransport':
          if (infoTitle) infoTitle.innerHTML =  "<i class=\"fas fa-bus icon-transport\" aria-hidden=\"true\"></i>" + Drupal.t("Public Transport");
          if (infoContent) infoContent.innerHTML = this.templates.render('publicTransport', templateData);
          this.showPublicTransportCategory();
          break;

        case 'parking':
          if (infoTitle) infoTitle.innerHTML = "<i class=\"fas fa-parking icon-parking\" aria-hidden=\"true\"></i>" + Drupal.t("Parking");
          if (infoContent) infoContent.innerHTML = this.templates.render('parking', templateData);
          this.showParkingCategory();
          break;

        case 'attractions':
          if (infoTitle) infoTitle.innerHTML = "<i class=\"fas fa-star icon-attractions\" aria-hidden=\"true\"></i>" + Drupal.t("Nearby Attractions");
          if (infoContent) infoContent.innerHTML = this.templates.render('attractions', templateData);
          this.showAttractionsCategory();
          break;

        default:
          console.warn('Unknown category:', category);
      }
    },

    /**
     * Show "Our Location" category (map actions only)
     */
    showWhereCategory: function() {
      this.clearMarkers();

      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      this.mapInstance.flyTo(businessLocation, 14, {
        duration: 1.2
      });

      if (window.RouteManager && typeof window.RouteManager.removeAllRoutes === 'function') {
        window.RouteManager.removeAllRoutes();
      }
    },

    /**
     * Show "By Car" category (map actions only)
     */
    showCarCategory: function() {
      this.clearMarkers();
      this.addCarMarkers();

      if (window.RouteManager && typeof window.RouteManager.showCarRoute === 'function') {
        setTimeout(() => {
          window.RouteManager.showCarRoute();
        }, 300);
      }
    },

    /**
     * Show "Public Transport" category (map actions only)
     */
    showPublicTransportCategory: function() {
      this.clearMarkers();
      this.addPublicTransportMarkers();

      if (window.RouteManager && typeof window.RouteManager.showPublicTransportRoute === 'function') {
        setTimeout(() => {
          window.RouteManager.showPublicTransportRoute();
        }, 300);
      }
    },

    /**
     * Show "Parking" category (map actions only)
     */
    showParkingCategory: function() {
      this.clearMarkers();
      this.addParkingMarkers();

      if (window.RouteManager && typeof window.RouteManager.removeAllRoutes === 'function') {
        window.RouteManager.removeAllRoutes();
      }

      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      this.mapInstance.flyTo(businessLocation, 15, {
        duration: 1.2
      });
    },

    /**
     * Show "Nearby" (Attractions) category (map actions only)
     */
    showAttractionsCategory: function() {
      this.clearMarkers();
      this.addAttractionMarkers();

      if (window.RouteManager && typeof window.RouteManager.removeAllRoutes === 'function') {
        window.RouteManager.removeAllRoutes();
      }

      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      this.mapInstance.flyTo(businessLocation, 14, {
        duration: 1.2
      });
    },

    /**
     * Add car-related markers (using customMarkers from settings)
     */
    addCarMarkers: function() {
      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      this.addMarker(businessLocation, 'main', Drupal.t('Your Business Headquarters'));

      // Add custom markers with category 'highway' or 'toll'
      const customMarkers = drupalSettings?.map?.config?.customMarkers || [];
      customMarkers.forEach(marker => {
        if (marker.enabled !== false && (marker.category === 'highway' || marker.category === 'toll')) {
          this.addMarker([marker.lat, marker.lng], 'toll',
            `${marker.name || ''}<br>${marker.description || ''}`);
        }
      });

      // Fit bounds to show all markers
      this.fitMapToMarkers();
    },

    /**
     * Add public transport markers (using customMarkers from settings)
     */
    addPublicTransportMarkers: function() {
      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      this.addMarker(businessLocation, 'main', Drupal.t('Your Business Headquarters'));

      // Add custom markers with category 'transport'
      const customMarkers = drupalSettings?.map?.config?.customMarkers || [];
      customMarkers.forEach(marker => {
        if (marker.enabled !== false && marker.category === 'transport') {
          const iconType = marker.subcategory || 'bus';
          this.addMarker([marker.lat, marker.lng], iconType,
            `${marker.name || ''}<br>${marker.description || ''}`);
        }
      });

      // Fit bounds to show all markers
      this.fitMapToMarkers();
    },

    /**
     * Add parking markers (using customMarkers from settings)
     */
    addParkingMarkers: function() {
      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      this.addMarker(businessLocation, 'main', Drupal.t('Your Business Headquarters'));

      // Add custom markers with category 'parking'
      const customMarkers = drupalSettings?.map?.config?.customMarkers || [];
      customMarkers.forEach(marker => {
        if (marker.enabled !== false && marker.category === 'parking') {
          this.addMarker([marker.lat, marker.lng], 'parking',
            `${marker.name || ''}<br>${marker.description || ''}`);
        }
      });

      // Fit bounds to show all markers
      this.fitMapToMarkers();
    },

    /**
     * Add attraction markers (using customMarkers from settings)
     */
    addAttractionMarkers: function() {
      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      this.addMarker(businessLocation, 'main', Drupal.t('Your Business Headquarters'));

      // Add custom markers with other categories (not highway, transport, parking)
      // const customMarkers = drupalSettings?.map?.config?.customMarkers || [];
      const customMarkers = drupalSettings?.map?.config?.nearbyMarkers || [];


      const excludedCategories = ['highway', 'transport', 'parking', 'toll'];

      customMarkers.forEach(marker => {
        if (marker.enabled !== false && !excludedCategories.includes(marker.category)) {
          this.addMarker([marker.lat, marker.lng], 'attraction',
            `${marker.name || ''}<br>${marker.description || ''}`);
        }
      });

      // Fit bounds to show all markers
      this.fitMapToMarkers();
    },

    // Method to fit map
    fitMapToMarkers: function() {

      if (!this.mapInstance) {
        console.error('Map instance not available');
        return;
      }

      if (this.currentMarkers.length === 0) {
        console.warn('No markers to fit');
        return;
      }

      // Create or clear feature group
      if (this.markerGroup) {
        this.mapInstance.removeLayer(this.markerGroup);
      }

      this.markerGroup = L.featureGroup();

      // Add markers to group
      this.currentMarkers.forEach(marker => {
        if (marker) {
          this.markerGroup.addLayer(marker);
        }
      });

      this.mapInstance.addLayer(this.markerGroup);

      // Fit bounds with a slight delay
      setTimeout(() => {
        const bounds = this.markerGroup.getBounds();

        if (bounds && bounds.isValid()) {
          console.log('Fitting to bounds:', bounds);

          this.mapInstance.fitBounds(bounds, {
            padding: [50, 50],
            maxZoom: 16,
            animate: true
          });
        }
      }, 100);
    },















    /**
     * Add a marker with custom icon
     */
    addMarker: function(coords, type, popupContent) {
      const icon = this.getMarkerIcon(type);
      const marker = L.marker(coords, { icon: icon }).addTo(this.mapInstance);

      if (popupContent) {
        marker.bindPopup(popupContent);
      }

      this.currentMarkers.push(marker);
      return marker;
    },

    /**
     * Get marker icon based on type
     */
    getMarkerIcon: function(type) {
      let className = 'custom-marker';
      let html = '';
      let iconSize = [28, 28];

      switch(type) {
        case 'main':
          className = 'marker-pulse';
          html = '<i class="fas fa-building"></i>';
          iconSize = [26, 26];
          break;
        case 'toll':
        case 'highway':
          className = 'custom-marker marker-toll';
          html = '<i class="fas fa-road"></i>';
          break;
        case 'train':
          className = 'custom-marker marker-transport';
          html = '<i class="fas fa-train"></i>';
          break;
        case 'bus':
          className = 'custom-marker marker-transport';
          html = '<i class="fas fa-bus"></i>';
          break;
        case 'subway':
        case 'metro':
          className = 'custom-marker marker-transport';
          html = '<i class="fas fa-subway"></i>';
          break;
        case 'parking':
          className = 'custom-marker marker-parking';
          html = '<i class="fas fa-parking"></i>';
          break;
        case 'attraction':
          className = 'custom-marker marker-attraction';
          html = '<i class="fas fa-star"></i>';
          break;
        case 'user':
          className = 'custom-marker marker-user';
          html = '<i class="fas fa-user"></i>';
          iconSize = [26, 26];
          break;
      }

      return L.divIcon({
        className: className,
        html: html,
        iconSize: iconSize
      });
    },

    /**
     * Clear all markers
     */
    clearMarkers: function() {
      this.currentMarkers.forEach(marker => {
        if (marker && marker.remove) {
          this.mapInstance.removeLayer(marker);
        }
      });
      this.currentMarkers = [];
    },

    /**
     * Handle My Location button
     */
    handleMyLocation: function() {
      if (!navigator.geolocation) {
        alert(Drupal.t('Geolocation not supported'));
        return;
      }

      const locationBtn = document.getElementById('locationBtn');
      if (locationBtn) {
        locationBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>' + Drupal.t('Locating...') + '</span>';
        locationBtn.disabled = true;
      }

      navigator.geolocation.getCurrentPosition(
        (position) => {
          const userLat = position.coords.latitude;
          const userLng = position.coords.longitude;

          console.log('📍 User location found:', userLat, userLng);

          // Clear existing markers
          this.clearMarkers();

          // Add user marker
          this.addMarker([userLat, userLng], 'user', Drupal.t('Your Current Location'));

          // Add business marker
          const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
          this.addMarker(businessLocation, 'main', Drupal.t('Your Business Headquarters'));

          // Fly to user location
          this.mapInstance.flyTo([userLat, userLng], 14, {
            duration: 1.5
          });

          // Show route to business
          if (window.RouteManager && typeof window.RouteManager.showRouteToBusiness === 'function') {
            setTimeout(() => {
              window.RouteManager.showRouteToBusiness(userLat, userLng);
            }, 1000);
          }

          // Update UI using template
          const infoTitle = document.getElementById('infoTitle');
          const infoContent = document.getElementById('infoContent');

          if (infoTitle) infoTitle.textContent = Drupal.t("My Location");
          if (infoContent) {
            const templateData = {
              position: {
                latitude: userLat,
                longitude: userLng,
                accuracy: position.coords.accuracy
              }
            };
            infoContent.innerHTML = this.templates.render('myLocation', templateData);
          }

          // Reset button
          if (locationBtn) {
            locationBtn.innerHTML = '<i class="fas fa-location-arrow"></i><span>' + Drupal.t('My Location') + '</span>';
            locationBtn.disabled = false;
            locationBtn.classList.add('active');
          }
        },
        (error) => {
          console.error('Geolocation error:', error);
          if (locationBtn) {
            locationBtn.innerHTML = '<i class="fas fa-location-arrow"></i><span>' + Drupal.t('My Location') + '</span>';
            locationBtn.disabled = false;
          }
          alert(Drupal.t('Unable to get your location. Please check browser permissions.'));
        },
        {
          enableHighAccuracy: true,
          timeout: 10000
        }
      );
    },

    /**
     * Share location
     */
    shareLocation: function() {
      const shareModal = document.getElementById('shareModal');
      if (shareModal) {
        shareModal.classList.remove('hidden');

        // Setup close button
        const cancelBtn = document.getElementById('cancelShare');
        if (cancelBtn) {
          cancelBtn.onclick = () => {
            shareModal.classList.add('hidden');
          };
        }

        // Setup share buttons
        const shareButtons = document.querySelectorAll('.share-btn');
        shareButtons.forEach(button => {
          button.onclick = (e) => {
            e.preventDefault();
            const app = button.dataset.app;
            this.shareViaApp(app);
          };
        });
      }
    },

    /**
     * Share via specific app
     */
    shareViaApp: function(app) {
      const address = drupalSettings?.map?.config?.address || {};
      const businessName = address.street || Drupal.t('Your Business');
      const addressText = `${address.street || ''}, ${address.postalCode || ''} ${address.city || ''}, ${address.country || ''}`;

      const businessLocation = drupalSettings?.map?.config?.center || [41.8349, 12.4700];
      const googleMapsUrl = `https://www.google.com/maps/search/?api=1&query=${businessLocation[0]},${businessLocation[1]}`;

      const message = `📍 ${businessName}\n${addressText}\n\n${Drupal.t('Find us here:')} ${googleMapsUrl}`;

      switch(app) {
        case 'whatsapp':
          window.open(`https://wa.me/?text=${encodeURIComponent(message)}`, '_blank');
          break;
        case 'email':
          window.open(`mailto:?subject=${encodeURIComponent(businessName + ' ' + Drupal.t('Location'))}&body=${encodeURIComponent(message)}`);
          break;
        case 'sms':
          window.open(`sms:?body=${encodeURIComponent(message)}`);
          break;
        case 'link':
          navigator.clipboard.writeText(googleMapsUrl).then(() => {
            alert(Drupal.t('📍 Location link copied to clipboard!'));
          }).catch(() => {
            const tempInput = document.createElement('input');
            tempInput.value = googleMapsUrl;
            document.body.appendChild(tempInput);
            tempInput.select();
            document.execCommand('copy');
            document.body.removeChild(tempInput);
            alert(Drupal.t('📍 Location link copied to clipboard!'));
          });
          break;
      }

      const shareModal = document.getElementById('shareModal');
      if (shareModal) {
        shareModal.classList.add('hidden');
      }
    },

    /**
     * Get map instance
     */
    getMap: function() {
      return this.mapInstance;
    }
  };

  // Debug function
  window.debugNavigation = function() {
    // console.log('=== Navigation Debug ===');
    // console.log('Nav buttons found:', document.querySelectorAll('.nav-button').length);

    document.querySelectorAll('.nav-button').forEach((btn, i) => {
      console.log(`Button ${i}:`, {
        id: btn.id,
        dataset: btn.dataset,
        text: btn.textContent,
        onclick: btn.onclick
      });
    });
  };

  // Function to zoom to all markers
  function zoomToAllMarkers() {
      if (allMarkers.length === 0) return;

      var group = L.featureGroup(allMarkers);
      map.fitBounds(group.getBounds(), {
          padding: [30, 30],
          maxZoom: 18
      });
  }



})(Drupal, drupalSettings);
