/**
 * @file
 * Route Manager - Handles route calculation and display
 */

(function(Drupal, drupalSettings) {
  'use strict';

  const RouteManager = {
    map: null,
    carRouteControl: null,
    publicTransportRouteControl: null,
    userRouteControl: null,
    currentPolyline: null,

    // Configuration from drupalSettings
    config: {
      businessLocation: null,
      highwayExit: null,
      airport: null,
      busStation: null,
      routeService: null,
      routing: null,
      transportModeOptions: null,
      customMarkers: null
    },

    /**
     * Initialize Route Manager
     */
    initialize: function(mapInstance) {
      console.log('🚀 Initializing Route Manager...');

      if (!mapInstance || typeof mapInstance.addLayer !== 'function') {
        console.error('❌ Invalid map instance provided to RouteManager');
        console.error('Map instance:', mapInstance);
        return;
      }

      this.map = mapInstance;

      // Load configuration from drupalSettings
      this.loadConfiguration();

      // Check if Leaflet Routing Machine is available
      this.checkRoutingAvailability();

      console.log('✅ Route Manager initialized with configuration:', this.config);
    },

    /**
     * Load configuration from drupalSettings
     */
    loadConfiguration: function() {
      console.log('📋 Loading configuration from drupalSettings...');

      // Get main configuration
      const mapConfig = drupalSettings?.map?.config || {};

      // Business location from center coordinates
      this.config.businessLocation = mapConfig.center || [41.8349, 12.4700];

      // Transportation data
      const transportation = mapConfig.transportation || {};
      this.config.highwayExit = transportation.highwayExit || null;
      this.config.airport = transportation.airport || null;
      this.config.busStation = transportation.busStation || null;
      this.config.defaultTransportMode = transportation.defaultTransportMode || 'driving';
      this.config.carRental = transportation.carRental || '';

      // Transport mode options
      this.config.transportModeOptions = transportation.transportModeOptions || {
        driving: Drupal.t('Driving'),
        walking: Drupal.t('Walking'),
        cycling: Drupal.t('Cycling'),
        transit: Drupal.t('Public Transport')
      };

      // Routing configuration
      this.config.routing = mapConfig.routing || {
        service: 'simulated',
        googleApiKey: '',
        mapboxToken: ''
      };

      // Route service options
      this.config.routeServiceOptions = mapConfig.routeServiceOptions || {
        simulated: Drupal.t('Simulated Routes'),
        osrm: Drupal.t('OSRM (Open Source)'),
        mapbox: Drupal.t('Mapbox Directions'),
        google: Drupal.t('Google Directions API')
      };

      // Custom markers for waypoints
      this.config.customMarkers = mapConfig.customMarkers || [];

      console.log('✅ Configuration loaded:', this.config);
    },

    /**
     * Check if Leaflet Routing Machine is loaded
     */
    checkRoutingAvailability: function() {
      console.log('🔍 Checking routing availability...');

      if (typeof L.Routing === 'undefined') {
        console.warn('⚠️ Leaflet Routing Machine not loaded');
        console.warn('Make sure leaflet-routing-machine.js is loaded AFTER leaflet.js');
        return false;
      }

      if (typeof L.Routing.control !== 'function') {
        console.warn('⚠️ L.Routing.control is not a function');
        return false;
      }

      console.log('✅ Leaflet Routing Machine is available');
      return true;
    },

    /**
     * Get coordinates for a location type
     */
    getCoordinates: function(locationType) {
      switch(locationType) {
        case 'business':
          return this.config.businessLocation;

        case 'highwayExit':
          if (this.config.highwayExit && this.config.highwayExit.lat && this.config.highwayExit.lng) {
            return [this.config.highwayExit.lat, this.config.highwayExit.lng];
          }
          return [42.0044, 12.5128]; // Default Rome coordinates

        case 'airport':
          if (this.config.airport && this.config.airport.lat && this.config.airport.lng) {
            return [this.config.airport.lat, this.config.airport.lng];
          }
          return [41.8518, 12.4940]; // Default station coordinates

        case 'busStation':
          if (this.config.busStation && this.config.busStation.enabled &&
              this.config.busStation.lat && this.config.busStation.lng) {
            return [this.config.busStation.lat, this.config.busStation.lng];
          }
          return [41.8330, 12.4740]; // Default bus station

        default:
          console.warn('Unknown location type:', locationType);
          return this.config.businessLocation;
      }
    },

    /**
     * Show car route from highway exit to business
     */
    showCarRoute: function() {
      console.log('🚗 Showing car route...');

      // Remove existing routes
      this.removeAllRoutes();

      // Check routing availability
      if (!this.isRoutingAvailable()) {
        console.warn('Routing not available, showing alternative');
        this.showAlternativeRoute('car');
        return;
      }

      // Update UI
      this.updateRouteDetails(Drupal.t('Calculating car route...'), true);

      try {
        console.log('Creating car route control...');

        // Get coordinates from configuration
        const startPoint = this.getCoordinates('highwayExit');
        const endPoint = this.getCoordinates('business');

        // Create waypoints
        const waypoints = [
          L.latLng(startPoint[0], startPoint[1]),
          L.latLng(endPoint[0], endPoint[1])
        ];

        console.log('Car route waypoints:', {
          from: startPoint,
          to: endPoint,
          highwayExit: this.config.highwayExit
        });

        // Get routing service configuration
        const routingService = this.config.routing.service || 'simulated';
        const router = this.getRouter(routingService, 'driving');

        // Create routing control
        this.carRouteControl = L.Routing.control({
          waypoints: waypoints,
          router: router,
          routeWhileDragging: false,
          showAlternatives: false,
          lineOptions: {
            styles: [{
              color: '#3498db',
              weight: 6,
              opacity: 0.8,
              className: 'route-line'
            }]
          },
          createMarker: function(i, waypoint, n) {
            // Return null to use default markers or false to skip markers
            return false;
          },
          addWaypoints: false,
          draggableWaypoints: false,
          fitSelectedRoutes: false,
          show: false,
          collapsible: false,
          autoRoute: true
        });

        console.log('Route control created, adding to map...');

        // Add to map
        this.map.addControl(this.carRouteControl);

        // Handle route found event
        this.carRouteControl.on('routesfound', (e) => {
          console.log('✅ Car route found!', e);
          this.handleCarRouteFound(e);

          // Fit bounds to show the route
          if (e.routes && e.routes[0]) {
            const route = e.routes[0];
            const coordinates = route.coordinates || [];
            if (coordinates.length > 0) {
              const latlngs = coordinates.map(coord => [coord.lat, coord.lng]);
              const routeBounds = L.latLngBounds(latlngs);
              this.map.fitBounds(routeBounds, { padding: [50, 50] });
            }
          }
        });

        // Handle errors
        this.carRouteControl.on('routingerror', (e) => {
          console.error('❌ Routing error:', e.error);
          this.showAlternativeRoute('car');
        });

      } catch (error) {
        console.error('❌ Error showing car route:', error);
        console.error('Stack:', error.stack);
        this.showAlternativeRoute('car');
      }
    },

    /**
     * Show public transport route
     */
    showPublicTransportRoute: function() {
      console.log('🚇 Showing public transport route...');

      // Remove existing routes FIRST
      this.removeAllRoutes();

      // Check routing availability
      if (!this.isRoutingAvailable()) {
        console.warn('Routing not available, showing alternative');
        this.showAlternativeRoute('publicTransport');
        return;
      }

      // Update UI
      this.updateRouteDetails(Drupal.t('Calculating public transport route...'), true);

      try {
        // Use airport as starting point if available, otherwise use default station
        const startPoint = this.getCoordinates('airport');
        const endPoint = this.getCoordinates('business');

        const waypoints = [
          L.latLng(startPoint[0], startPoint[1]),
          L.latLng(endPoint[0], endPoint[1])
        ];

        console.log('Public transport waypoints:', {
          from: startPoint,
          to: endPoint,
          airport: this.config.airport
        });

        // Get routing service - use walking profile for public transport
        const routingService = this.config.routing.service || 'simulated';
        const router = this.getRouter(routingService, 'walking');

        this.publicTransportRouteControl = L.Routing.control({
          waypoints: waypoints,
          router: router,
          routeWhileDragging: false,
          showAlternatives: false,
          lineOptions: {
            styles: [{
              color: '#2ecc71',
              weight: 6,
              opacity: 0.8,
              dashArray: '10, 10',
              className: 'route-line'
            }]
          },
          createMarker: function(i, waypoint, n) {
            return false;
          },
          addWaypoints: false,
          draggableWaypoints: false,
          fitSelectedRoutes: false,
          show: false,
          collapsible: false,
          autoRoute: true
        });

        // Add to map
        this.map.addControl(this.publicTransportRouteControl);

        this.publicTransportRouteControl.on('routesfound', (e) => {
          console.log('✅ Public transport route found!', e);
          this.handlePublicTransportRouteFound(e);
        });

        this.publicTransportRouteControl.on('routingerror', (e) => {
          console.error('❌ Routing error:', e.error);
          this.showAlternativeRoute('publicTransport');
        });

      } catch (error) {
        console.error('❌ Error showing public transport route:', error);
        this.showAlternativeRoute('publicTransport');
      }
    },

    /**
     * Show route from user location to business
     */
    showRouteToBusiness: function(userLat, userLng) {
      console.log('📍 Showing route from user to business...');

      this.removeAllRoutes();

      if (!this.isRoutingAvailable()) {
        console.warn('Routing not available for user route');
        return;
      }

      try {
        const waypoints = [
          L.latLng(userLat, userLng),
          L.latLng(this.config.businessLocation[0], this.config.businessLocation[1])
        ];

        // Get routing service
        const routingService = this.config.routing.service || 'simulated';
        const router = this.getRouter(routingService, 'driving');

        this.userRouteControl = L.Routing.control({
          waypoints: waypoints,
          router: router,
          routeWhileDragging: false,
          showAlternatives: false,
          lineOptions: {
            styles: [{
              color: '#9b59b6',
              weight: 6,
              opacity: 0.8,
              dashArray: '5, 5',
              className: 'route-line'
            }]
          },
          createMarker: function(i, waypoint, n) {
            return false;
          },
          addWaypoints: false,
          draggableWaypoints: false,
          fitSelectedRoutes: false,
          show: false,
          collapsible: false,
          autoRoute: true
        });

        // Add to map
        this.map.addControl(this.userRouteControl);

        this.userRouteControl.on('routesfound', (e) => {
          console.log('✅ User route found!', e);
          this.handleUserRouteFound(e, userLat, userLng);
        });

        this.userRouteControl.on('routingerror', (e) => {
          console.error('❌ User routing error:', e.error);
        });

      } catch (error) {
        console.error('❌ Error showing user route:', error);
      }
    },

    /**
     * Get router based on service configuration
     */
    getRouter: function(service, profile) {
      console.log('🔄 Getting router for service:', service, 'profile:', profile);

      switch(service) {
        case 'osrm':
          return L.Routing.osrmv1({
            serviceUrl: 'https://router.project-osrm.org/route/v1',
            profile: profile,
            timeout: 10000
          });

        case 'mapbox':
          const token = this.config.routing.mapboxToken || '';
          if (!token) {
            console.warn('Mapbox token not configured, falling back to OSRM');
            return L.Routing.osrmv1({
              serviceUrl: 'https://router.project-osrm.org/route/v1',
              profile: profile,
              timeout: 10000
            });
          }
          return L.Routing.mapbox(token, {
            profile: profile === 'transit' ? 'driving-traffic' : profile,
            timeout: 10000
          });

        case 'google':
          const apiKey = this.config.routing.googleApiKey || '';
          if (!apiKey) {
            console.warn('Google API key not configured, falling back to OSRM');
            return L.Routing.osrmv1({
              serviceUrl: 'https://router.project-osrm.org/route/v1',
              profile: profile,
              timeout: 10000
            });
          }
          // Note: Leaflet Routing Machine may need additional plugin for Google
          return L.Routing.osrmv1({
            serviceUrl: 'https://router.project-osrm.org/route/v1',
            profile: profile,
            timeout: 10000
          });

        case 'simulated':
        default:
          // Return a simulated router for fallback
          return L.Routing.osrmv1({
            serviceUrl: 'https://router.project-osrm.org/route/v1',
            profile: profile,
            timeout: 10000
          });
      }
    },

    /**
     * Check if routing is available
     */
    isRoutingAvailable: function() {
      const isAvailable = typeof L !== 'undefined' &&
                         typeof L.Routing !== 'undefined' &&
                         typeof L.Routing.control === 'function';

      if (!isAvailable) {
        console.warn('⚠️ Routing features are not available');
      }

      return isAvailable;
    },

    /**
     * Handle car route found
     */
    handleCarRouteFound: function(e) {
      console.log('📊 Car routes found event:', e);

      if (!e.routes || e.routes.length === 0) {
        console.warn('No routes found');
        this.showAlternativeRoute('car');
        return;
      }

      const route = e.routes[0];

      let distance = 'N/A';
      let time = 'N/A';

      if (route.summary) {
        distance = (route.summary.totalDistance / 1000).toFixed(1);
        time = Math.round(route.summary.totalTime / 60);
      }

      // Get highway exit name from configuration
      const highwayExitName = this.config.highwayExit ?
        this.config.highwayExit.name || Drupal.t('Highway Exit') :
        Drupal.t('Highway Exit');

      const highwayExitDesc = this.config.highwayExit ?
        this.config.highwayExit.description || '' : '';

      this.updateRouteDetails(`
        <div class="route-info">
          <p><strong>🚗 ${Drupal.t('Car Route from')} ${highwayExitName}:</strong></p>
          ${highwayExitDesc ? `<p><em>${highwayExitDesc}</em></p>` : ''}
          <p>📏 <strong>${Drupal.t('Distance:')}</strong> ${distance} km</p>
          <p>⏱️ <strong>${Drupal.t('Estimated time:')}</strong> ${time} ${Drupal.t('minutes')}</p>
          <p>🛣️ <strong>${Drupal.t('Route:')}</strong> ${Drupal.t('Highway to business location')}</p>
          <button class="remove-route-btn" onclick="window.RouteManager.removeAllRoutes()">
            <i class="fas fa-times"></i> ${Drupal.t('Remove Route')}
          </button>
        </div>
      `);
    },

    /**
     * Handle public transport route found
     */
    handlePublicTransportRouteFound: function(e) {
      if (!e.routes || e.routes.length === 0) {
        this.showAlternativeRoute('publicTransport');
        return;
      }

      const route = e.routes[0];

      let distance = 'N/A';
      let time = 'N/A';

      if (route.summary) {
        distance = (route.summary.totalDistance / 1000).toFixed(1);
        time = Math.round(route.summary.totalTime / 60);
      }

      // Get airport info from configuration
      const airportName = this.config.airport ?
        this.config.airport.name || Drupal.t('Airport') :
        Drupal.t('Airport');

      const airportDesc = this.config.airport ?
        this.config.airport.description || '' : '';

      this.updateRouteDetails(`
        <div class="route-info">
          <p><strong>🚇 ${Drupal.t('Public Transport Route from')} ${airportName}:</strong></p>
          ${airportDesc ? `<p><em>${airportDesc}</em></p>` : ''}
          <p>📏 <strong>${Drupal.t('Walking distance:')}</strong> ${distance} km</p>
          <p>⏱️ <strong>${Drupal.t('Walking time:')}</strong> ${time} ${Drupal.t('minutes')}</p>
          <p>🚌 <strong>${Drupal.t('Recommended transport:')}</strong>
            ${this.config.transportModeOptions[this.config.defaultTransportMode] || Drupal.t('Public Transport')}
          </p>
          ${this.config.carRental ? `<p>🚗 <strong>${Drupal.t('Car rental:')}</strong> ${this.config.carRental}</p>` : ''}
          <button class="remove-route-btn" onclick="window.RouteManager.removeAllRoutes()">
            <i class="fas fa-times"></i> ${Drupal.t('Remove Route')}
          </button>
        </div>
      `);
    },

    /**
     * Handle user route found
     */
    handleUserRouteFound: function(e, userLat, userLng) {
      if (!e.routes || e.routes.length === 0) {
        return;
      }

      const route = e.routes[0];

      let distance = 'N/A';
      let time = 'N/A';

      if (route.summary) {
        distance = (route.summary.totalDistance / 1000).toFixed(1);
        time = Math.round(route.summary.totalTime / 60);
      }

      this.updateRouteDetails(`
        <div class="route-info">
          <p><strong>📍 ${Drupal.t('Route to Our Business:')}</strong></p>
          <p>📏 <strong>${Drupal.t('Distance:')}</strong> ${distance} km</p>
          <p>⏱️ <strong>${Drupal.t('Estimated time:')}</strong> ${time} ${Drupal.t('minutes')}</p>
          <p>🗺️ <strong>${Drupal.t('From your current location')}</strong></p>
          <p>${Drupal.t('Using')}: ${this.config.routeServiceOptions[this.config.routing.service] || Drupal.t('Simulated routing')}</p>
          <button class="remove-route-btn" onclick="window.RouteManager.removeAllRoutes()">
            <i class="fas fa-times"></i> ${Drupal.t('Remove Route')}
          </button>
        </div>
      `);
    },

    /**
     * Show alternative route when routing not available
     */
    showAlternativeRoute: function(type) {
      let content = '';
      let polylineCoords = [];
      let color = '#3498db';

      if (type === 'car') {
        const startPoint = this.getCoordinates('highwayExit');
        const endPoint = this.getCoordinates('business');
        const highwayExitName = this.config.highwayExit ?
          this.config.highwayExit.name || Drupal.t('Highway Exit') :
          Drupal.t('Highway Exit');

        content = `
          <div class="route-info">
            <p><strong>🚗 ${Drupal.t('Car Route from')} ${highwayExitName}:</strong></p>
            <p>📏 <strong>${Drupal.t('Distance:')}</strong> ${Drupal.t('Approximately')} 18 km</p>
            <p>⏱️ <strong>${Drupal.t('Estimated time:')}</strong> 25-35 ${Drupal.t('minutes')}</p>
            ${this.config.highwayExit && this.config.highwayExit.description ?
              `<p>🛣️ <strong>${Drupal.t('Directions:')}</strong> ${this.config.highwayExit.description}</p>` : ''}
            <p><em>${Drupal.t('Note: Live routing is currently unavailable. Showing static route information.')}</em></p>
            <p><small>${Drupal.t('Service:')} ${this.config.routeServiceOptions[this.config.routing.service] || Drupal.t('Simulated')}</small></p>
          </div>
        `;
        polylineCoords = [startPoint, endPoint];
        color = '#3498db';

      } else if (type === 'publicTransport') {
        const startPoint = this.getCoordinates('airport');
        const endPoint = this.getCoordinates('business');
        const airportName = this.config.airport ?
          this.config.airport.name || Drupal.t('Airport') :
          Drupal.t('Airport');

        content = `
          <div class="route-info">
            <p><strong>🚇 ${Drupal.t('Public Transport Route from')} ${airportName}:</strong></p>
            <p>📏 <strong>${Drupal.t('Distance:')}</strong> ${Drupal.t('Approximately')} 1.2 km ${Drupal.t('walking')}</p>
            <p>⏱️ <strong>${Drupal.t('Walking time:')}</strong> 15-20 ${Drupal.t('minutes')}</p>
            <p>🚌 <strong>${Drupal.t('Recommended:')}</strong> ${Drupal.t('Public transport to business location')}</p>
            ${this.config.airport && this.config.airport.description ?
              `<p><em>${this.config.airport.description}</em></p>` : ''}
            <p><em>${Drupal.t('Note: Live routing is currently unavailable. Showing static route information.')}</em></p>
            <p><small>${Drupal.t('Service:')} ${this.config.routeServiceOptions[this.config.routing.service] || Drupal.t('Simulated')}</small></p>
          </div>
        `;
        polylineCoords = [startPoint, endPoint];
        color = '#2ecc71';
      }

      this.updateRouteDetails(content);

      // Draw a simple polyline as fallback
      if (polylineCoords.length > 0 && this.map && typeof this.map.addLayer === 'function') {
        // Remove existing polyline
        if (this.currentPolyline && this.map.hasLayer(this.currentPolyline)) {
          this.map.removeLayer(this.currentPolyline);
        }

        this.currentPolyline = L.polyline(polylineCoords, {
          color: color,
          weight: 4,
          opacity: 0.6,
          dashArray: type === 'car' ? '10, 10' : '5, 5'
        }).addTo(this.map);

        // Fit bounds to show the polyline
        this.map.fitBounds(polylineCoords, { padding: [50, 50] });
      }
    },

    /**
     * Update route details display
     */
    updateRouteDetails: function(content, loading = false) {
      const routeDetails = document.getElementById('routeDetails');
      if (!routeDetails) {
        console.warn('routeDetails element not found');
        return;
      }

      if (loading) {
        routeDetails.innerHTML = `
          <div class="route-info">
            <p><i class="fas fa-spinner fa-spin"></i> ${content}</p>
          </div>
        `;
      } else {
        routeDetails.innerHTML = content;
      }
    },

    /**
     * Remove all routes from map - CORRETTA
     */
    removeAllRoutes: function() {
      console.log('🗑️ Removing all routes...');

      // Remove routing controls
      const controls = [this.carRouteControl, this.publicTransportRouteControl, this.userRouteControl];

      controls.forEach((control, index) => {
        if (control && this.map) {
          try {
            // Try different methods to remove control
            if (typeof this.map.removeControl === 'function') {
              this.map.removeControl(control);
              console.log(`Removed control ${index} with removeControl`);
            }
            // If removeControl doesn't exist, try to remove the container
            else if (control._container && control._container.parentNode) {
              control._container.parentNode.removeChild(control._container);
              console.log(`Removed control ${index} container`);
            }
            // Last resort: remove from DOM by class
            else {
              const routingContainers = document.querySelectorAll('.leaflet-routing-container');
              routingContainers.forEach(container => {
                container.parentNode.removeChild(container);
              });
              console.log(`Removed routing containers from DOM`);
            }
          } catch (e) {
            console.warn(`Error removing control ${index}:`, e);
          }
        }
      });

      // Reset references
      this.carRouteControl = null;
      this.publicTransportRouteControl = null;
      this.userRouteControl = null;

      // Remove polyline fallback
      if (this.currentPolyline && this.map && this.map.hasLayer) {
        try {
          if (this.map.hasLayer(this.currentPolyline)) {
            this.map.removeLayer(this.currentPolyline);
          }
        } catch (e) {
          console.warn('Error removing polyline:', e);
        }
        this.currentPolyline = null;
      }

      // Clear route details
      this.updateRouteDetails('');

      console.log('✅ All routes removed');
    },

    /**
     * Safe remove control method
     */
    safeRemoveControl: function(control) {
      if (!control || !this.map) return;

      try {
        // Method 1: Standard Leaflet method
        if (typeof this.map.removeControl === 'function') {
          this.map.removeControl(control);
          return true;
        }

        // Method 2: Remove container from DOM
        if (control._container && control._container.parentNode) {
          control._container.parentNode.removeChild(control._container);
          return true;
        }

        // Method 3: Remove by Leaflet internal ID
        if (control._leaflet_id && this.map._layers) {
          delete this.map._layers[control._leaflet_id];
          return true;
        }

        return false;
      } catch (error) {
        console.warn('Error in safeRemoveControl:', error);
        return false;
      }
    },

    /**
     * Test routing functionality
     */
    testRouting: function() {
      console.log('=== Testing Routing Functionality ===');

      console.log('1. Leaflet available:', typeof L !== 'undefined');
      console.log('2. L.Routing available:', typeof L.Routing !== 'undefined');
      console.log('3. L.Routing.control:', typeof L.Routing.control);

      console.log('4. Map instance:', this.map);
      console.log('5. Map methods:', this.map ? Object.keys(this.map).filter(k => typeof this.map[k] === 'function') : 'No map');

      console.log('6. Configuration:', this.config);
      console.log('7. Routing service:', this.config.routing.service);
      console.log('8. Highway exit:', this.config.highwayExit);
      console.log('9. Airport:', this.config.airport);

      // Test if we can create a simple control
      if (typeof L.Routing !== 'undefined') {
        try {
          const testControl = L.Routing.control({
            waypoints: [],
            show: false
          });
          console.log('✅ Test control created:', testControl);
          console.log('Control methods:', Object.keys(testControl).filter(k => typeof testControl[k] === 'function'));
        } catch (error) {
          console.error('❌ Error creating test control:', error);
        }
      }
    }
  };

  // Export to global scope
  window.RouteManager = RouteManager;

  // Debug functions
  window.testRouting = function() {
    if (window.RouteManager) {
      window.RouteManager.testRouting();
    }
  };

  window.debugMapInstance = function() {
    console.log('=== Map Instance Debug ===');
    console.log('RouteManager.map:', window.RouteManager ? window.RouteManager.map : 'No RouteManager');

    if (window.RouteManager && window.RouteManager.map) {
      const map = window.RouteManager.map;
      console.log('Map type:', typeof map);
      console.log('Map methods:', Object.keys(map).filter(k => typeof map[k] === 'function'));
      console.log('Has addControl?', typeof map.addControl === 'function');
      console.log('Has removeControl?', typeof map.removeControl === 'function');
      console.log('Has addLayer?', typeof map.addLayer === 'function');
      console.log('Has removeLayer?', typeof map.removeLayer === 'function');
    }
  };

})(Drupal, drupalSettings);
