<?php

namespace Drupal\map\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Url;
use Drupal\views\Views;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure map settings.
 */
class MapSettingsForm extends ConfigFormBase {

  /**
   * The business identity config.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $businessIdentityConfig;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->businessIdentityConfig = $instance->config('business_identity.settings');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['map.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'map_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('map.settings');

    // Attach admin CSS and JS
    $form['#attached']['library'][] = 'map/map-admin';
    $form['#attached']['library'][] = 'leaflet/leaflet';
    $form['#attached']['library'][] = 'map/leaflet-position-selector';

    // Main container with vertical tabs
    $form['#prefix'] = '<div class="layout-container map-settings-form">';
    $form['#suffix'] = '</div>';

    // Get business identity data
    $business_name = $this->businessIdentityConfig->get('company_name') ?: $this->businessIdentityConfig->get('legal_name');
    $business_address = $this->businessIdentityConfig->get('business_address');
    $address_lat = $this->businessIdentityConfig->get('address_lat');
    $address_lng = $this->businessIdentityConfig->get('address_lng');
    $headquarters_lat = $this->businessIdentityConfig->get('headquarters_lat');
    $headquarters_lng = $this->businessIdentityConfig->get('headquarters_lng');
    $point_of_sale_lat = $this->businessIdentityConfig->get('point_of_sale_lat');
    $point_of_sale_lng = $this->businessIdentityConfig->get('point_of_sale_lng');

    // Determine which coordinates to use (prioritize business address, then headquarters, then point of sale)
    $default_lat = $address_lat ?: $headquarters_lat ?: $point_of_sale_lat ?: 41.9028;
    $default_lng = $address_lng ?: $headquarters_lng ?: $point_of_sale_lng ?: 12.4964;

    // Vertical tabs container
    $form['vertical_tabs'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Map Configuration'),
      '#default_tab' => 'edit-business',
      '#attributes' => [
        'class' => ['map-settings-tabs'],
      ],
    ];

    // ==============================================
    // TAB 1: BUSINESS INFORMATION
    // ==============================================
    $form['business'] = [
      '#type' => 'details',
      '#title' => $this->t('Business Information'),
      '#description' => $this->t('Address and location information managed through Business Identity.'),
      '#group' => 'vertical_tabs',
      '#attributes' => [
        'id' => 'business-settings',
        'class' => ['business-info-tab'],
      ],
    ];

    // Business Identity integration information
    $form['business']['business_identity_info'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Business Details'),
      '#description' => $this->t('These details are managed through the <a href=":url">Business Identity settings</a>.', [
        ':url' => Url::fromRoute('business_identity.settings')->toString(),
      ]),
      '#attributes' => [
        'class' => ['business-identity-info'],
      ],
    ];

    // Display business information as read-only
    $form['business']['business_identity_info']['business_name'] = [
      '#type' => 'item',
      '#title' => $this->t('Business Name'),
      '#markup' => $business_name ?: $this->t('Not set'),
      '#wrapper_attributes' => [
        'class' => ['business-info-item'],
      ],
    ];

    if ($business_address) {
      $formatted_address = [];
      if (!empty($business_address['address_line1'])) {
        $formatted_address[] = $business_address['address_line1'];
      }
      if (!empty($business_address['address_line2'])) {
        $formatted_address[] = $business_address['address_line2'];
      }
      if (!empty($business_address['address_line3'])) {
        $formatted_address[] = $business_address['address_line3'];
      }
      if (!empty($business_address['postal_code'])) {
        $formatted_address[] = $business_address['postal_code'];
      }
      if (!empty($business_address['locality'])) {
        $formatted_address[] = $business_address['locality'];
      }
      if (!empty($business_address['administrative_area'])) {
        $formatted_address[] = $business_address['administrative_area'];
      }
      if (!empty($business_address['country_code'])) {
        $country_code = $business_address['country_code'];
        $country_name = $this->getCountryName($country_code);
        $formatted_address[] = $country_name ?: $country_code;
      }

      $form['business']['business_identity_info']['business_address'] = [
        '#type' => 'item',
        '#title' => $this->t('Business Address'),
        '#markup' => $formatted_address ? implode(', ', $formatted_address) : $this->t('Not set'),
        '#wrapper_attributes' => [
          'class' => ['business-info-item'],
        ],
      ];
    }

    // Display coordinates
    $coordinates_info = [];
    if ($address_lat && $address_lng) {
      $coordinates_info[] = $this->t('Business Address: @lat, @lng', [
        '@lat' => $address_lat,
        '@lng' => $address_lng,
      ]);
    }
    if ($headquarters_lat && $headquarters_lng) {
      $coordinates_info[] = $this->t('Headquarters: @lat, @lng', [
        '@lat' => $headquarters_lat,
        '@lng' => $headquarters_lng,
      ]);
    }
    if ($point_of_sale_lat && $point_of_sale_lng) {
      $coordinates_info[] = $this->t('Point of Sale: @lat, @lng', [
        '@lat' => $point_of_sale_lat,
        '@lng' => $point_of_sale_lng,
      ]);
    }

    if ($coordinates_info) {
      $form['business']['business_identity_info']['coordinates'] = [
        '#type' => 'item',
        '#title' => $this->t('Available Coordinates'),
        '#markup' => '<ul class="coordinates-list"><li>' . implode('</li><li>', $coordinates_info) . '</li></ul>',
        '#wrapper_attributes' => [
          'class' => ['business-info-item'],
        ],
      ];
    }

    // Map center selection
    $form['business']['map_center_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Map Center Source'),
      '#options' => [
        'business_address' => $this->t('Business Address Coordinates'),
        'headquarters' => $this->t('Headquarters Coordinates'),
        'point_of_sale' => $this->t('Point of Sale Coordinates'),
        'custom' => $this->t('Custom Coordinates'),
      ],
      '#default_value' => $config->get('map_center_source') ?: 'business_address',
      '#description' => $this->t('Select which coordinates to use as the center of the map.'),
      '#attributes' => [
        'class' => ['map-center-select'],
      ],
    ];

    // Custom coordinates field (only visible when custom is selected)
    $form['business']['custom_coordinates'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Custom Map Center'),
      '#states' => [
        'visible' => [
          ':input[name="map_center_source"]' => ['value' => 'custom'],
        ],
      ],
      '#attributes' => [
        'class' => ['custom-coordinates-fieldset'],
      ],
    ];

    // Custom coordinates map selector
    $form['business']['custom_coordinates']['custom_coordinates_map'] = [
      '#type' => 'item',
      '#title' => $this->t('Select Location on Map'),
      '#description' => $this->t('Click on the map to select the custom center coordinates.'),
      '#markup' => '<div id="custom-coordinates-map" class="leaflet-map" style="height: 400px; width: 100%;"></div>',
    ];

    $form['business']['custom_coordinates']['custom_lat'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('custom_lat') ?: $default_lat,
      '#attributes' => [
        'id' => 'custom-coordinates-lat',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['business']['custom_coordinates']['custom_lng'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('custom_lng') ?: $default_lng,
      '#attributes' => [
        'id' => 'custom-coordinates-lng',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['business']['custom_coordinates']['coordinates_display'] = [
      '#type' => 'item',
      '#title' => $this->t('Selected Coordinates'),
      '#markup' => '<div id="custom-coordinates-display">' .
                   ($config->get('custom_lat') ?: $default_lat) . ', ' .
                   ($config->get('custom_lng') ?: $default_lng) . '</div>',
    ];

    // ==============================================
    // TAB 2: VIEWS FOR NEARBY INTERESTS
    // ==============================================
    $form['nearby'] = [
      '#type' => 'details',
      '#title' => $this->t('Nearby'),
      '#description' => $this->t('Configure points of interest from Views to be displayed as Nearby Interests on the map.'),
      '#group' => 'vertical_tabs',
      '#attributes' => [
        'id' => 'views-settings',
        'class' => ['views-tab'],
      ],
    ];

    // Nearby interests section
    $form['nearby']['nearby_interests'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Nearby Interests'),
      '#description' => $this->t('Select Views that contain points of interest to display on the map as nearby locations.'),
      '#attributes' => [
        'class' => ['nearby-interests-fieldset'],
      ],
    ];

    // Get all available views with display types that can be used for mapping
    $views_options = $this->getViewsForMapping();

    $form['nearby']['nearby_interests']['nearby_interests_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Nearby Interests'),
      '#default_value' => $config->get('nearby_interests_enabled') ?: FALSE,
      '#description' => $this->t('Display points of interest from selected views on the map.'),
    ];

    $form['nearby']['nearby_interests']['nearby_interests_view'] = [
      '#type' => 'select',
      '#title' => $this->t('Select View for Nearby Interests'),
      '#options' => $views_options,
      '#default_value' => $config->get('nearby_interests_view') ?: [],
      '#description' => $this->t('Select a view that contains points of interest. Should have latitude and longitude fields.'),
      '#states' => [
        'visible' => [
          ':input[name="nearby_interests_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Distance settings for nearby interests
    $form['nearby']['nearby_interests']['nearby_interests_distance'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Distance Settings'),
      '#states' => [
        'visible' => [
          ':input[name="nearby_interests_enabled"]' => ['checked' => TRUE],
        ],
      ],
      '#attributes' => [
        'class' => ['distance-settings-fieldset'],
      ],
    ];

    $form['nearby']['nearby_interests']['nearby_interests_distance']['nearby_radius'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Radius (km)'),
      '#min' => 1,
      '#max' => 1000,
      '#step' => 1,
      '#default_value' => $config->get('nearby_radius') ?: 50,
      '#description' => $this->t('(@todo) Maximum distance from map center to show nearby interests.'),
    ];

    $form['nearby']['nearby_interests']['nearby_interests_distance']['nearby_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Number of Points'),
      '#min' => 1,
      '#max' => 100,
      '#step' => 1,
      '#default_value' => $config->get('nearby_limit') ?: 20,
      '#description' => $this->t('Maximum number of nearby points to display on the map.'),
    ];

    // Display settings for nearby interests
    $form['nearby']['nearby_interests']['nearby_interests_display'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display Settings'),
      '#states' => [
        'visible' => [
          ':input[name="nearby_interests_enabled"]' => ['checked' => TRUE],
        ],
      ],
      '#attributes' => [
        'class' => ['display-settings-fieldset'],
      ],
    ];

    $form['nearby']['nearby_interests']['nearby_interests_display']['nearby_cluster'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Cluster Nearby Points'),
      '#default_value' => $config->get('nearby_cluster') ?: TRUE,
      '#description' => $this->t('Group nearby points together when zoomed out for better visibility.'),
    ];

    $form['nearby']['nearby_interests']['nearby_interests_display']['nearby_popup'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Popup Information'),
      '#default_value' => $config->get('nearby_popup') ?: TRUE,
      '#description' => $this->t('Display detailed information in a popup when clicking on a point.'),
    ];

    $form['nearby']['nearby_interests']['nearby_interests_display']['nearby_categories'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Group by Categories'),
      '#default_value' => $config->get('nearby_categories') ?: FALSE,
      '#description' => $this->t('Group nearby points by categories if available in the view.'),
    ];

    // Legend settings
    $form['nearby']['nearby_interests']['nearby_interests_legend'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Legend Settings'),
      '#states' => [
        'visible' => [
          ':input[name="nearby_interests_enabled"]' => ['checked' => TRUE],
        ],
      ],
      '#attributes' => [
        'class' => ['legend-settings-fieldset'],
      ],
    ];

    $form['nearby']['nearby_interests']['nearby_interests_legend']['nearby_legend_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Legend'),
      '#default_value' => $config->get('nearby_legend_enabled') ?: TRUE,
      '#description' => $this->t('Display a legend explaining the different point types on the map.'),
    ];

    $form['nearby']['nearby_interests']['nearby_interests_legend']['nearby_legend_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Legend Position'),
      '#options' => [
        'topleft' => $this->t('Top Left'),
        'topright' => $this->t('Top Right'),
        'bottomleft' => $this->t('Bottom Left'),
        'bottomright' => $this->t('Bottom Right'),
      ],
      '#default_value' => $config->get('nearby_legend_position') ?: 'bottomright',
      '#description' => $this->t('Position of the legend on the map.'),
      '#states' => [
        'visible' => [
          ':input[name="nearby_legend_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Advanced view settings
    $form['nearby']['nearby_interests']['nearby_interests_advanced'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Advanced Settings'),
      '#states' => [
        'visible' => [
          ':input[name="nearby_interests_enabled"]' => ['checked' => TRUE],
        ],
      ],
      '#attributes' => [
        'class' => ['advanced-settings-fieldset'],
      ],
    ];

    $form['nearby']['nearby_interests']['nearby_interests_advanced']['nearby_cache'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Cache Results'),
      '#default_value' => $config->get('nearby_cache') ?: TRUE,
      '#description' => $this->t('Cache nearby interest results for better performance.'),
    ];

    $form['nearby']['nearby_interests']['nearby_interests_advanced']['nearby_cache_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache Duration (seconds)'),
      '#min' => 60,
      '#max' => 86400,
      '#step' => 60,
      '#default_value' => $config->get('nearby_cache_duration') ?: 3600,
      '#description' => $this->t('How long to cache nearby interest results.'),
      '#states' => [
        'visible' => [
          ':input[name="nearby_cache"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // ==============================================
    // TAB 3: GENERAL SETTINGS
    // ==============================================
    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General'),
      '#description' => $this->t('Basic map configuration and display settings.'),
      '#group' => 'vertical_tabs',
      '#attributes' => [
        'id' => 'general-settings',
      ],
    ];

    // Map type selection
    $form['general']['map_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Map Type'),
      '#options' => [
        'openstreetmap' => $this->t('OpenStreetMap'),
        'google' => $this->t('Google Maps'),
        'mapbox' => $this->t('Mapbox'),
        'cartodb' => $this->t('CartoDB'),
      ],
      '#default_value' => $config->get('map_type') ?: 'openstreetmap',
      '#description' => $this->t('Select the base map type to use.'),
    ];

    // Address display format
    $form['general']['address_format'] = [
      '#type' => 'radios',
      '#title' => $this->t('Address Display Format'),
      '#options' => [
        'single_line' => $this->t('Single line (Street, City, Country)'),
        'multi_line' => $this->t('Multi-line (US/International format)'),
        'localized' => $this->t('Localized format (based on country)'),
      ],
      '#default_value' => $config->get('address_format') ?: 'multi_line',
      '#description' => $this->t('How to display the business address on the map and in directions.'),
    ];

    // ==============================================
    // TAB 4: ZOOM & CONTROLS
    // ==============================================
    $form['zoom_controls'] = [
      '#type' => 'details',
      '#title' => $this->t('Zoom & Controls'),
      '#description' => $this->t('Configure zoom levels and map controls.'),
      '#group' => 'vertical_tabs',
      '#attributes' => [
        'id' => 'zoom-controls-settings',
      ],
    ];

    // Zoom settings
    $form['zoom_controls']['zoom_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Zoom Settings'),
    ];

    $form['zoom_controls']['zoom_settings']['default_zoom'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Zoom Level'),
      '#min' => 1,
      '#max' => 19,
      '#default_value' => $config->get('default_zoom') ?: 14,
      '#required' => TRUE,
    ];

    $form['zoom_controls']['zoom_settings']['min_zoom'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum Zoom Level'),
      '#min' => 1,
      '#max' => 19,
      '#default_value' => $config->get('min_zoom') ?: 5,
      '#required' => TRUE,
    ];

    $form['zoom_controls']['zoom_settings']['max_zoom'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Zoom Level'),
      '#min' => 1,
      '#max' => 19,
      '#default_value' => $config->get('max_zoom') ?: 18,
      '#required' => TRUE,
    ];

    // Control settings
    $form['zoom_controls']['control_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Control Settings'),
    ];

    $form['zoom_controls']['control_settings']['zoom_control'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Zoom Control'),
      '#default_value' => $config->get('zoom_control') ?: TRUE,
    ];

    $form['zoom_controls']['control_settings']['zoom_control_position'] = [
      '#type' => 'select',
      '#title' => $this->t('Zoom Control Position'),
      '#options' => [
        'topleft' => $this->t('Top Left'),
        'topright' => $this->t('Top Right'),
        'bottomleft' => $this->t('Bottom Left'),
        'bottomright' => $this->t('Bottom Right'),
      ],
      '#default_value' => $config->get('zoom_control_position') ?: 'topleft',
      '#states' => [
        'visible' => [
          ':input[name="zoom_control"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['zoom_controls']['control_settings']['attribution_control'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Attribution Control'),
      '#default_value' => $config->get('attribution_control') ?: TRUE,
    ];

    $form['zoom_controls']['control_settings']['fullscreen_control'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Fullscreen Control'),
      '#default_value' => $config->get('fullscreen_control') ?: TRUE,
    ];

    // ==============================================
    // TAB 5: PUBLIC TRANSPORT
    // ==============================================
    $form['public_transport'] = [
      '#type' => 'details',
      '#title' => $this->t('Public transport'),
      '#description' => $this->t('Configure public_transport points for route planning.'),
      '#group' => 'vertical_tabs',
      '#attributes' => [
        'id' => 'public-transport-settings',
      ],
    ];

    // Highway Exit/Toll Booth Configuration
    $form['public_transport']['highway_exit'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Highway Exit / Toll Booth'),
      '#description' => $this->t('Configure the nearest highway exit for driving directions.'),
      '#attributes' => [
        'class' => ['parent-fieldset'],
      ],
    ];

    $form['public_transport']['highway_exit']['highway_exit_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable highway exit routing'),
      '#default_value' => $config->get('highway_exit_enabled') ?: FALSE,
      '#description' => $this->t('Show driving route from highway exit to destination.'),
    ];

    $form['public_transport']['highway_exit']['highway_exit_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Exit Name'),
      '#default_value' => $config->get('highway_exit_name') ?: $this->t('Ponzano Romano Exit'),
      '#description' => $this->t('Name of the highway exit (e.g., "Ponzano Romano Exit", "Roma Nord Exit").'),
      '#states' => [
        'visible' => [
          ':input[name="highway_exit_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['public_transport']['highway_exit']['highway_exit_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Exit Description'),
      '#default_value' => $config->get('highway_exit_description') ?: $this->t('A1 Highway Exit - Rome North'),
      '#description' => $this->t('Description and instructions for the highway exit.'),
      '#rows' => 2,
      '#states' => [
        'visible' => [
          ':input[name="highway_exit_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['public_transport']['highway_exit']['highway_exit_coordinates'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Exit Coordinates'),
      '#states' => [
        'visible' => [
          ':input[name="highway_exit_enabled"]' => ['checked' => TRUE],
        ],
      ],
      '#attributes' => [
        'class' => ['nested-fieldset', 'smaller-fieldset'],
      ],
    ];

    $form['public_transport']['highway_exit']['highway_exit_coordinates']['highway_exit_map'] = [
      '#type' => 'item',
      '#title' => $this->t('Select Highway Exit Location'),
      '#description' => $this->t('Click on the map to select the highway exit coordinates.'),
      '#markup' => '<div id="highway-exit-map" class="leaflet-map" style="height: 300px; width: 100%;"></div>',
    ];

    $form['public_transport']['highway_exit']['highway_exit_coordinates']['highway_exit_lat'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('highway_exit_lat') ?: 42.2568,
      '#attributes' => [
        'id' => 'highway-exit-lat',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['public_transport']['highway_exit']['highway_exit_coordinates']['highway_exit_lng'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('highway_exit_lng') ?: 12.5734,
      '#attributes' => [
        'id' => 'highway-exit-lng',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['public_transport']['highway_exit']['highway_exit_coordinates']['highway_exit_coordinates_display'] = [
      '#type' => 'item',
      '#title' => $this->t('Selected Coordinates'),
      '#markup' => '<div id="highway-exit-coordinates-display">' .
                   ($config->get('highway_exit_lat') ?: 42.2568) . ', ' .
                   ($config->get('highway_exit_lng') ?: 12.5734) . '</div>',
    ];

    // Train Station Configuration
    $form['public_transport']['train_station'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Train Station'),
      '#description' => $this->t('Configure the nearest train station for public transport directions.'),
      '#attributes' => ['class' => ['parent-fieldset']],
    ];

    $form['public_transport']['train_station']['train_station_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable train station'),
      '#default_value' => $config->get('train_station_enabled') ?: FALSE,
      '#description' => $this->t('Show train station on the map.'),
    ];

    $form['public_transport']['train_station']['train_station_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Station Name'),
      '#default_value' => $config->get('train_station_name') ?: $this->t('Rieti Railway Station'),
      '#description' => $this->t('Name of the train station.'),
      '#states' => [
        'visible' => [
          ':input[name="train_station_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['public_transport']['train_station']['train_station_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Station Description'),
      '#default_value' => $config->get('train_station_description') ?: $this->t('Regional trains to Rome (Tiburtina Station). Connection via bus service.'),
      '#description' => $this->t('Train lines and connection information.'),
      '#rows' => 2,
      '#states' => [
        'visible' => [
          ':input[name="train_station_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['public_transport']['train_station']['train_station_coordinates'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Station Coordinates'),
      '#states' => [
        'visible' => [
          ':input[name="train_station_enabled"]' => ['checked' => TRUE],
        ],
      ],
      '#attributes' => [
        'class' => ['nested-fieldset', 'smaller-fieldset'],
      ],
    ];

    $form['public_transport']['train_station']['train_station_coordinates']['train_station_map'] = [
      '#type' => 'item',
      '#title' => $this->t('Select Train Station Location'),
      '#description' => $this->t('Click on the map to select the train station coordinates.'),
      '#markup' => '<div id="train-station-map" class="leaflet-map" style="height: 300px; width: 100%;"></div>',
    ];

    $form['public_transport']['train_station']['train_station_coordinates']['train_station_lat'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('train_station_lat') ?: 42.4012,
      '#attributes' => [
        'id' => 'train-station-lat',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['public_transport']['train_station']['train_station_coordinates']['train_station_lng'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('train_station_lng') ?: 12.8584,
      '#attributes' => [
        'id' => 'train-station-lng',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['public_transport']['train_station']['train_station_coordinates']['train_station_coordinates_display'] = [
      '#type' => 'item',
      '#title' => $this->t('Selected Coordinates'),
      '#markup' => '<div id="train-station-coordinates-display">' .
                   ($config->get('train_station_lat') ?: 42.4012) . ', ' .
                   ($config->get('train_station_lng') ?: 12.8584) . '</div>',
    ];

    // Airport Configuration
    $form['public_transport']['airport'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Airport'),
      '#description' => $this->t('Configure the nearest airport for travel directions.'),
      '#attributes' => ['class' => ['parent-fieldset']],
    ];

    $form['public_transport']['airport']['airport_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable airport'),
      '#default_value' => $config->get('airport_enabled') ?: FALSE,
      '#description' => $this->t('Show airport on the map.'),
    ];

    $form['public_transport']['airport']['airport_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Airport Name'),
      '#default_value' => $config->get('airport_name') ?: $this->t('Rome Fiumicino Airport (FCO)'),
      '#description' => $this->t('Name of the airport with IATA code.'),
      '#states' => [
        'visible' => [
          ':input[name="airport_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['public_transport']['airport']['airport_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Airport Description'),
      '#default_value' => $config->get('airport_description') ?: $this->t('Leonardo da Vinci International Airport. Direct train connection to Rome.'),
      '#description' => $this->t('Airport information and connection details.'),
      '#rows' => 2,
      '#states' => [
        'visible' => [
          ':input[name="airport_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['public_transport']['airport']['airport_coordinates'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Airport Coordinates'),
      '#states' => [
        'visible' => [
          ':input[name="airport_enabled"]' => ['checked' => TRUE],
        ],
      ],
      '#attributes' => [
        'class' => ['nested-fieldset', 'smaller-fieldset'],
      ],
    ];

    $form['public_transport']['airport']['airport_coordinates']['airport_map'] = [
      '#type' => 'item',
      '#title' => $this->t('Select Airport Location'),
      '#description' => $this->t('Click on the map to select the airport coordinates.'),
      '#markup' => '<div id="airport-map" class="leaflet-map" style="height: 300px; width: 100%;"></div>',
    ];

    $form['public_transport']['airport']['airport_coordinates']['airport_lat'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('airport_lat') ?: 41.8003,
      '#attributes' => [
        'id' => 'airport-lat',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['public_transport']['airport']['airport_coordinates']['airport_lng'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('airport_lng') ?: 12.2389,
      '#attributes' => [
        'id' => 'airport-lng',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['public_transport']['airport']['airport_coordinates']['airport_coordinates_display'] = [
      '#type' => 'item',
      '#title' => $this->t('Selected Coordinates'),
      '#markup' => '<div id="airport-coordinates-display">' .
                   ($config->get('airport_lat') ?: 41.8003) . ', ' .
                   ($config->get('airport_lng') ?: 12.2389) . '</div>',
    ];

    // ==============================================
    // TAB 6: PARKING AREAS
    // ==============================================
    $form['parking'] = [
      '#type' => 'details',
      '#title' => $this->t('Parking'),
      '#description' => $this->t('Configure parking areas with detailed information.'),
      '#group' => 'vertical_tabs',
      '#attributes' => [
        'id' => 'parking-settings',
      ],
    ];

    // Main parking settings
    $form['parking']['main_parking'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Main Parking Area'),
      '#description' => $this->t('Configure the primary parking area for visitors.'),
      '#attributes' => ['class' => ['parent-fieldset']],
    ];

    $form['parking']['main_parking']['main_parking_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable main parking area'),
      '#default_value' => $config->get('main_parking_enabled') ?: FALSE,
      '#description' => $this->t('Show main parking area on the map.'),
    ];

    $form['parking']['main_parking']['main_parking_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Parking Name'),
      '#default_value' => $config->get('main_parking_name') ?: $this->t('Main Visitor Parking'),
      '#description' => $this->t('Name of the main parking area.'),
      '#states' => [
        'visible' => [
          ':input[name="main_parking_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['parking']['main_parking']['main_parking_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Parking Description'),
      '#default_value' => $config->get('main_parking_description') ?: $this->t('Spacious parking area with 50+ spaces. Well-lit and secure.'),
      '#description' => $this->t('Detailed description of the parking facility.'),
      '#rows' => 2,
      '#states' => [
        'visible' => [
          ':input[name="main_parking_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['parking']['main_parking']['main_parking_coordinates'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Parking Coordinates'),
      '#states' => [
        'visible' => [
          ':input[name="main_parking_enabled"]' => ['checked' => TRUE],
        ],
      ],
      '#attributes' => [
        'class' => ['nested-fieldset', 'smaller-fieldset'],
      ],
    ];

    $form['parking']['main_parking']['main_parking_coordinates']['main_parking_map'] = [
      '#type' => 'item',
      '#title' => $this->t('Select Parking Location'),
      '#description' => $this->t('Click on the map to select the main parking area coordinates.'),
      '#markup' => '<div id="main-parking-map" class="leaflet-map" style="height: 300px; width: 100%;"></div>',
    ];

    $form['parking']['main_parking']['main_parking_coordinates']['main_parking_lat'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('main_parking_lat') ?: 42.4065,
      '#attributes' => [
        'id' => 'main-parking-lat',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['parking']['main_parking']['main_parking_coordinates']['main_parking_lng'] = [
      '#type' => 'hidden',
      '#default_value' => $config->get('main_parking_lng') ?: 12.6870,
      '#attributes' => [
        'id' => 'main-parking-lng',
        'class' => ['coordinate-input'],
      ],
    ];

    $form['parking']['main_parking']['main_parking_coordinates']['main_parking_coordinates_display'] = [
      '#type' => 'item',
      '#title' => $this->t('Selected Coordinates'),
      '#markup' => '<div id="main-parking-coordinates-display">' .
                   ($config->get('main_parking_lat') ?: 42.4065) . ', ' .
                   ($config->get('main_parking_lng') ?: 12.6870) . '</div>',
    ];

    // Parking capacity
    $form['parking']['parking_capacity'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Parking Capacity'),
      '#description' => $this->t('Configure parking capacity and special spaces.'),
    ];

    $form['parking']['parking_capacity']['parking_capacity_total'] = [
      '#type' => 'number',
      '#title' => $this->t('Total Capacity'),
      '#min' => 0,
      '#default_value' => $config->get('parking_capacity_total') ?: 50,
      '#description' => $this->t('Total number of parking spaces available.'),
    ];

    $form['parking']['parking_capacity']['parking_capacity_disabled'] = [
      '#type' => 'number',
      '#title' => $this->t('Disabled Parking Spaces'),
      '#min' => 0,
      '#default_value' => $config->get('parking_capacity_disabled') ?: 4,
      '#description' => $this->t('Number of parking spaces reserved for disabled visitors.'),
    ];

    $form['parking']['parking_capacity']['parking_capacity_family'] = [
      '#type' => 'number',
      '#title' => $this->t('Family Parking Spaces'),
      '#min' => 0,
      '#default_value' => $config->get('parking_capacity_family') ?: 2,
      '#description' => $this->t('Number of wider parking spaces for families.'),
    ];

    $form['parking']['parking_capacity']['parking_capacity_ev'] = [
      '#type' => 'number',
      '#title' => $this->t('EV Charging Spaces'),
      '#min' => 0,
      '#default_value' => $config->get('parking_capacity_ev') ?: 2,
      '#description' => $this->t('Number of electric vehicle charging spaces.'),
    ];

    // Parking rates and hours
    $form['parking']['parking_rates'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Parking Rates & Hours'),
    ];

    $form['parking']['parking_rates']['parking_fee_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Parking Fee'),
      '#options' => [
        'free' => $this->t('Free Parking'),
        'hourly' => $this->t('Hourly Rate'),
        'daily' => $this->t('Daily Rate'),
        'weekly' => $this->t('Weekly Rate'),
      ],
      '#default_value' => $config->get('parking_fee_type') ?: 'free',
      '#description' => $this->t('Type of parking fee structure.'),
    ];

    $form['parking']['parking_rates']['parking_hours'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Parking Hours'),
      '#default_value' => $config->get('parking_hours') ?: $this->t('24/7'),
      '#description' => $this->t('Parking facility opening hours.'),
      '#rows' => 2,
    ];

    // ==============================================
    // TAB 7: ROUTING & INTEGRATION
    // ==============================================
    $form['routing'] = [
      '#type' => 'details',
      '#title' => $this->t('Routing & Integration'),
      '#description' => $this->t('Configure route calculation and external service integration.'),
      '#group' => 'vertical_tabs',
      '#attributes' => [
        'id' => 'routing-settings',
      ],
    ];

    // Route calculation settings
    $form['routing']['route_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Route Calculation'),
    ];

    $form['routing']['route_settings']['route_service'] = [
      '#type' => 'select',
      '#title' => $this->t('Routing Service'),
      '#options' => [
        'simulated' => $this->t('Simulated Routes'),
        'osrm' => $this->t('OSRM (Open Source)'),
        'mapbox' => $this->t('Mapbox Directions'),
        'google' => $this->t('Google Directions API'),
      ],
      '#default_value' => $config->get('route_service') ?: 'simulated',
      '#description' => $this->t('Service to use for calculating routes.'),
    ];

    $form['routing']['route_settings']['mapbox_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mapbox Access Token'),
      '#default_value' => $config->get('mapbox_token') ?: '',
      '#description' => $this->t('Required if using Mapbox Directions API.'),
      '#states' => [
        'visible' => [
          ':input[name="route_service"]' => ['value' => 'mapbox'],
        ],
      ],
    ];

    $form['routing']['route_settings']['google_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Google Maps API Key'),
      '#default_value' => $config->get('google_api_key') ?: '',
      '#description' => $this->t('Required if using Google Directions API.'),
      '#states' => [
        'visible' => [
          ':input[name="route_service"]' => ['value' => 'google'],
        ],
      ],
    ];

    $form['routing']['route_settings']['default_transport_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Transport Mode'),
      '#options' => [
        'driving' => $this->t('Driving'),
        'walking' => $this->t('Walking'),
        'cycling' => $this->t('Cycling'),
        'transit' => $this->t('Public Transport'),
      ],
      '#default_value' => $config->get('default_transport_mode') ?: 'driving',
      '#description' => $this->t('Default mode of transport for route calculations.'),
    ];

    // Additional public_transport services
    $form['routing']['additional_services'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Additional Services'),
    ];

    $form['routing']['additional_services']['taxi_service'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Taxi Service Contact'),
      '#default_value' => $config->get('taxi_service') ?: '+39 0746 123456',
      '#description' => $this->t('Local taxi service phone number.'),
    ];

    $form['routing']['additional_services']['car_rental'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Car Rental Information'),
      '#default_value' => $config->get('car_rental') ?: $this->t('Available at Rome airports and major train stations'),
      '#description' => $this->t('Car rental availability information.'),
    ];

    $form['routing']['additional_services']['bus_station_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable bus station information'),
      '#default_value' => $config->get('bus_station_enabled') ?: FALSE,
      '#description' => $this->t('Show bus station on the map.'),
    ];

    // ==============================================
    // TAB 8: ADVANCED SETTINGS
    // ==============================================
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced'),
      '#description' => $this->t('Advanced configuration and custom settings.'),
      '#group' => 'vertical_tabs',
      '#attributes' => [
        'id' => 'advanced-settings',
      ],
    ];

    // Custom markers JSON
    $form['advanced']['custom_markers'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Custom Markers'),
    ];

    $form['advanced']['custom_markers']['custom_markers_json'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Markers JSON'),
      '#default_value' => $config->get('custom_markers_json') ?: '',
      '#description' => $this->t('Define custom markers in JSON format.'),
      '#rows' => 12,
      '#attributes' => [
        'style' => 'font-family: monospace; font-size: 12px;',
      ],
    ];

    // Custom parking zones JSON
    $form['advanced']['custom_parking'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Custom Parking Zones'),
    ];

    $form['advanced']['custom_parking']['parking_zones_json'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Parking Zones JSON'),
      '#default_value' => $config->get('parking_zones_json') ?: '',
      '#description' => $this->t('Define multiple parking zones in JSON format.'),
      '#rows' => 12,
      '#attributes' => [
        'style' => 'font-family: monospace; font-size: 12px;',
      ],
    ];

    // Debug settings
    $form['advanced']['debug'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Debug Settings'),
    ];

    $form['advanced']['debug']['enable_debug'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable debug mode'),
      '#default_value' => $config->get('enable_debug') ?: FALSE,
      '#description' => $this->t('Show debug information and console logs.'),
    ];

    $form['advanced']['debug']['log_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Log Level'),
      '#options' => [
        'error' => $this->t('Errors only'),
        'warn' => $this->t('Warnings and errors'),
        'info' => $this->t('All information'),
      ],
      '#default_value' => $config->get('log_level') ?: 'error',
      '#description' => $this->t('Level of detail for logging.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_debug"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Get all views that can be used for mapping points of interest.
   *
   * @return array
   *   Array of view options.
   */
  private function getViewsForMapping() {
    $views = Views::getEnabledViews();
    $options = [];

    foreach ($views as $view) {
      $view_id = $view->id();
      $view_label = $view->label();

      // Check if view has displays that could be used for mapping
      // You might want to add more specific checks based on your view configurations
      $displays = $view->get('display');

      // Add view to options if it has at least one display
      if (!empty($displays)) {
        $options[$view_id] = $this->t('@label (@id)', [
          '@label' => $view_label,
          '@id' => $view_id,
        ]);
      }
    }

    // Sort options alphabetically by label
    asort($options);

    return $options;
  }

  /**
   * Get country name from country code.
   *
   * @param string $country_code
   *   The country code.
   *
   * @return string|null
   *   The country name or null if not found.
   */
  private function getCountryName($country_code) {
    $countries = $this->getCountryOptions();
    return $countries[$country_code] ?? null;
  }

  /**
   * Get country options for select field.
   *
   * @return array
   *   Array of country options.
   */
  private function getCountryOptions() {
    return [
      'CH' => $this->t('Switzerland'),
      'IT' => $this->t('Italy'),
      'US' => $this->t('United States'),
      'GB' => $this->t('United Kingdom'),
      'DE' => $this->t('Germany'),
      'FR' => $this->t('France'),
      'ES' => $this->t('Spain'),
      'CA' => $this->t('Canada'),
      'AU' => $this->t('Australia'),
      'JP' => $this->t('Japan'),
      'CN' => $this->t('China'),
      'IN' => $this->t('India'),
      'BR' => $this->t('Brazil'),
      'MX' => $this->t('Mexico'),
      'RU' => $this->t('Russia'),
      'ZA' => $this->t('South Africa'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate custom coordinates if selected
    if ($form_state->getValue('map_center_source') === 'custom') {
      $lat = $form_state->getValue('custom_lat');
      if ($lat < -90 || $lat > 90) {
        $form_state->setErrorByName('custom_lat', $this->t('Latitude must be between -90 and 90.'));
      }

      $lng = $form_state->getValue('custom_lng');
      if ($lng < -180 || $lng > 180) {
        $form_state->setErrorByName('custom_lng', $this->t('Longitude must be between -180 and 180.'));
      }
    }

    // Validate zoom levels
    $min_zoom = $form_state->getValue('min_zoom');
    $max_zoom = $form_state->getValue('max_zoom');
    $default_zoom = $form_state->getValue('default_zoom');

    if ($min_zoom > $max_zoom) {
      $form_state->setErrorByName('min_zoom', $this->t('Minimum zoom cannot be greater than maximum zoom.'));
    }

    if ($default_zoom < $min_zoom || $default_zoom > $max_zoom) {
      $form_state->setErrorByName('default_zoom', $this->t('Default zoom must be between minimum and maximum zoom.'));
    }

    // Validate highway exit coordinates if enabled
    if ($form_state->getValue('highway_exit_enabled')) {
      $exit_lat = $form_state->getValue('highway_exit_lat');
      $exit_lng = $form_state->getValue('highway_exit_lng');

      if ($exit_lat && ($exit_lat < -90 || $exit_lat > 90)) {
        $form_state->setErrorByName('highway_exit_lat', $this->t('Highway exit latitude must be between -90 and 90.'));
      }

      if ($exit_lng && ($exit_lng < -180 || $exit_lng > 180)) {
        $form_state->setErrorByName('highway_exit_lng', $this->t('Highway exit longitude must be between -180 and 180.'));
      }
    }

    // Validate train station coordinates if enabled
    if ($form_state->getValue('train_station_enabled')) {
      $station_lat = $form_state->getValue('train_station_lat');
      $station_lng = $form_state->getValue('train_station_lng');

      if ($station_lat && ($station_lat < -90 || $station_lat > 90)) {
        $form_state->setErrorByName('train_station_lat', $this->t('Train station latitude must be between -90 and 90.'));
      }

      if ($station_lng && ($station_lng < -180 || $station_lng > 180)) {
        $form_state->setErrorByName('train_station_lng', $this->t('Train station longitude must be between -180 and 180.'));
      }
    }

    // Validate airport coordinates if enabled
    if ($form_state->getValue('airport_enabled')) {
      $airport_lat = $form_state->getValue('airport_lat');
      $airport_lng = $form_state->getValue('airport_lng');

      if ($airport_lat && ($airport_lat < -90 || $airport_lat > 90)) {
        $form_state->setErrorByName('airport_lat', $this->t('Airport latitude must be between -90 and 90.'));
      }

      if ($airport_lng && ($airport_lng < -180 || $airport_lng > 180)) {
        $form_state->setErrorByName('airport_lng', $this->t('Airport longitude must be between -180 and 180.'));
      }
    }

    // Validate main parking coordinates if enabled
    if ($form_state->getValue('main_parking_enabled')) {
      $main_parking_lat = $form_state->getValue('main_parking_lat');
      $main_parking_lng = $form_state->getValue('main_parking_lng');

      if ($main_parking_lat && ($main_parking_lat < -90 || $main_parking_lat > 90)) {
        $form_state->setErrorByName('main_parking_lat', $this->t('Main parking latitude must be between -90 and 90.'));
      }

      if ($main_parking_lng && ($main_parking_lng < -180 || $main_parking_lng > 180)) {
        $form_state->setErrorByName('main_parking_lng', $this->t('Main parking longitude must be between -180 and 180.'));
      }
    }

    // Validate custom markers JSON
    $custom_markers_json = $form_state->getValue('custom_markers_json');
    if (!empty(trim($custom_markers_json))) {
      json_decode($custom_markers_json);
      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setErrorByName('custom_markers_json',
          $this->t('Invalid JSON format for custom markers: %error', ['%error' => json_last_error_msg()])
        );
      }
    }

    // Validate parking zones JSON
    $parking_zones_json = $form_state->getValue('parking_zones_json');
    if (!empty(trim($parking_zones_json))) {
      json_decode($parking_zones_json);
      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setErrorByName('parking_zones_json',
          $this->t('Invalid JSON format for parking zones: %error', ['%error' => json_last_error_msg()])
        );
      }
    }

    // Validate nearby interests settings
    if ($form_state->getValue('nearby_interests_enabled')) {

      $selected_view = $form_state->getValue('nearby_interests_view');
        if (empty($selected_view)) {
          $form_state->setErrorByName('nearby_interests_view',
            $this->t('Please select one view when Nearby Interests is enabled.'));
      }

      $nearby_radius = $form_state->getValue('nearby_radius');
      if ($nearby_radius < 1 || $nearby_radius > 1000) {
        $form_state->setErrorByName('nearby_radius',
          $this->t('Nearby radius must be between 1 and 1000 km.')
        );
      }

      $nearby_limit = $form_state->getValue('nearby_limit');
      if ($nearby_limit < 1 || $nearby_limit > 100) {
        $form_state->setErrorByName('nearby_limit',
          $this->t('Nearby limit must be between 1 and 100 points.')
        );
      }

      if ($form_state->getValue('nearby_cache')) {
        $cache_duration = $form_state->getValue('nearby_cache_duration');
        if ($cache_duration < 60 || $cache_duration > 86400) {
          $form_state->setErrorByName('nearby_cache_duration',
            $this->t('Cache duration must be between 60 and 86400 seconds.')
          );
        }
      }
    }

    // Check if business identity has address configured when business address is selected
    if ($form_state->getValue('map_center_source') === 'business_address') {
      $business_address = $this->businessIdentityConfig->get('business_address');
      $address_lat = $this->businessIdentityConfig->get('address_lat');
      $address_lng = $this->businessIdentityConfig->get('address_lng');

      if (empty($business_address['address_line1']) && empty($business_address['locality'])) {
        $form_state->setErrorByName('map_center_source',
          $this->t('Business address is not configured in Business Identity settings. Please configure it first or select a different map center source.')
        );
      }

      if (empty($address_lat) || empty($address_lng)) {
        $this->messenger()->addWarning(
          $this->t('Business address coordinates are not configured in Business Identity. The map will use approximate coordinates.')
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('map.settings');

    // Save all configuration values
    $config
      // Business identity integration
      ->set('map_center_source', $form_state->getValue('map_center_source'))
      ->set('custom_lat', $form_state->getValue('custom_lat'))
      ->set('custom_lng', $form_state->getValue('custom_lng'))
      // Views configuration for nearby interests
      ->set('nearby_interests_enabled', $form_state->getValue('nearby_interests_enabled'))
      ->set('nearby_interests_view', $form_state->getValue('nearby_interests_view'))
      ->set('nearby_radius', $form_state->getValue('nearby_radius'))
      ->set('nearby_limit', $form_state->getValue('nearby_limit'))
      ->set('nearby_cluster', $form_state->getValue('nearby_cluster'))
      ->set('nearby_popup', $form_state->getValue('nearby_popup'))
      ->set('nearby_categories', $form_state->getValue('nearby_categories'))
      ->set('nearby_legend_enabled', $form_state->getValue('nearby_legend_enabled'))
      ->set('nearby_legend_position', $form_state->getValue('nearby_legend_position'))
      ->set('nearby_cache', $form_state->getValue('nearby_cache'))
      ->set('nearby_cache_duration', $form_state->getValue('nearby_cache_duration'))
      // General settings
      ->set('map_type', $form_state->getValue('map_type'))
      ->set('address_format', $form_state->getValue('address_format'))
      // Zoom & Controls
      ->set('default_zoom', $form_state->getValue('default_zoom'))
      ->set('min_zoom', $form_state->getValue('min_zoom'))
      ->set('max_zoom', $form_state->getValue('max_zoom'))
      ->set('zoom_control', $form_state->getValue('zoom_control'))
      ->set('zoom_control_position', $form_state->getValue('zoom_control_position'))
      ->set('attribution_control', $form_state->getValue('attribution_control'))
      ->set('fullscreen_control', $form_state->getValue('fullscreen_control'))
      // Public transport
      ->set('highway_exit_enabled', $form_state->getValue('highway_exit_enabled'))
      ->set('highway_exit_name', $form_state->getValue('highway_exit_name'))
      ->set('highway_exit_description', $form_state->getValue('highway_exit_description'))
      ->set('highway_exit_lat', $form_state->getValue('highway_exit_lat'))
      ->set('highway_exit_lng', $form_state->getValue('highway_exit_lng'))
      ->set('train_station_enabled', $form_state->getValue('train_station_enabled'))
      ->set('train_station_name', $form_state->getValue('train_station_name'))
      ->set('train_station_description', $form_state->getValue('train_station_description'))
      ->set('train_station_lat', $form_state->getValue('train_station_lat'))
      ->set('train_station_lng', $form_state->getValue('train_station_lng'))
      ->set('airport_enabled', $form_state->getValue('airport_enabled'))
      ->set('airport_name', $form_state->getValue('airport_name'))
      ->set('airport_description', $form_state->getValue('airport_description'))
      ->set('airport_lat', $form_state->getValue('airport_lat'))
      ->set('airport_lng', $form_state->getValue('airport_lng'))
      // Parking
      ->set('main_parking_enabled', $form_state->getValue('main_parking_enabled'))
      ->set('main_parking_name', $form_state->getValue('main_parking_name'))
      ->set('main_parking_description', $form_state->getValue('main_parking_description'))
      ->set('main_parking_lat', $form_state->getValue('main_parking_lat'))
      ->set('main_parking_lng', $form_state->getValue('main_parking_lng'))
      ->set('parking_capacity_total', $form_state->getValue('parking_capacity_total'))
      ->set('parking_capacity_disabled', $form_state->getValue('parking_capacity_disabled'))
      ->set('parking_capacity_family', $form_state->getValue('parking_capacity_family'))
      ->set('parking_capacity_ev', $form_state->getValue('parking_capacity_ev'))
      ->set('parking_fee_type', $form_state->getValue('parking_fee_type'))
      ->set('parking_hours', $form_state->getValue('parking_hours'))
      // Routing & Integration
      ->set('route_service', $form_state->getValue('route_service'))
      ->set('mapbox_token', $form_state->getValue('mapbox_token'))
      ->set('google_api_key', $form_state->getValue('google_api_key'))
      ->set('default_transport_mode', $form_state->getValue('default_transport_mode'))
      ->set('taxi_service', $form_state->getValue('taxi_service'))
      ->set('car_rental', $form_state->getValue('car_rental'))
      ->set('bus_station_enabled', $form_state->getValue('bus_station_enabled'))
      // Advanced
      ->set('custom_markers_json', $form_state->getValue('custom_markers_json'))
      ->set('parking_zones_json', $form_state->getValue('parking_zones_json'))
      ->set('enable_debug', $form_state->getValue('enable_debug'))
      ->set('log_level', $form_state->getValue('log_level'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
