/**
 * @file
 * Configuration Manager - Centralized access to drupalSettings
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Configuration Manager
   */
  window.BusinessMapConfigManager = {
    /**
     * Get configuration with fallback and validation
     * @param {string} path - Dot notation path to config
     * @param {*} defaultValue - Default value if not found
     * @returns {*} The configuration value
     */
    get: function(path, defaultValue = null) {
      const keys = path.split('.');
      let value = drupalSettings;

      for (const key of keys) {
        if (value && typeof value === 'object' && key in value) {
          value = value[key];
        } else {
          return defaultValue;
        }
      }

      return value !== undefined ? value : defaultValue;
    },

    /**
     * Get nested configuration safely
     * @param {string} basePath - Base path to config
     * @param {Object} defaults - Default structure with fallback values
     * @returns {Object} Merged configuration
     */
    getNested: function(basePath, defaults = {}) {
      const config = this.get(basePath, {});
      return {
        ...defaults,
        ...config
      };
    },

    /**
     * Check if configuration exists
     * @param {string} path - Dot notation path
     * @returns {boolean} True if configuration exists
     */
    has: function(path) {
      const value = this.get(path, undefined);
      return value !== undefined;
    },

    /**
     * Check if setup is complete
     * @returns {Object} Contains status and missing configurations
     */
    checkSetupComplete: function() {
      const missingConfigs = [];
      const businessIdentity = this.get('map.business_identity_settings', {});
      const mapConfig = this.get('map.config', {});

      console.log('check degli essenziali');
      console.log(businessIdentity);
      console.log('Parking config:', mapConfig.parking);

      // Check business identity
      if (!businessIdentity || Object.keys(businessIdentity).length === 0) {
        missingConfigs.push({
          name: 'Business Identity',
          path: '/admin/config/business/identity',
          description: Drupal.t('Configure business address and contact information')
        });
      } else if (!businessIdentity.address_lat || !businessIdentity.address_lng ||
                 businessIdentity.address_lat === 41.9028 || businessIdentity.address_lng === 12.4964) {
        // Check if using default Rome coordinates
        missingConfigs.push({
          name: 'Business Identity',
          path: '/admin/config/business/identity',
          description: Drupal.t('Update business address from default location')
        });
      }

      // Check map configuration
      if (!mapConfig || Object.keys(mapConfig).length === 0) {
        missingConfigs.push({
          name: 'Map Configuration',
          path: '/admin/config/map/settings',
          description: Drupal.t('[M1]. Configure map settings, markers, and transportation options')
        });
      } else {
        // Check for specific required map configurations
        if (!mapConfig.mapType) {
          missingConfigs.push({
            name: 'Map Configuration',
            path: '/admin/config/map/settings',
            description: Drupal.t('[M2]. Customize map type and display settings')
          });
        }

        // Check parking configuration (if enabled)
        if (mapConfig.parking && mapConfig.parking.main_parking && mapConfig.parking.main_parking.enabled) {
          if (!mapConfig.parking.main_parking.coordinates ||
              !mapConfig.parking.main_parking.coordinates.lat ||
              !mapConfig.parking.main_parking.coordinates.lng) {
            missingConfigs.push({
              name: 'Parking Configuration',
              path: '/admin/config/map/settings',
              description: Drupal.t('[P1]. Parking is enabled but coordinates are missing')
            });
          }
        }
      }

      return {
        isComplete: missingConfigs.length === 0,
        missingConfigs: missingConfigs,
        hasBusinessIdentity: !missingConfigs.some(item => item.name === 'Business Identity'),
        hasMapConfig: !missingConfigs.some(item => item.name === 'Map Configuration'),
        hasParkingConfig: mapConfig.parking && mapConfig.parking.main_parking && mapConfig.parking.main_parking.enabled
      };
    },

    /**
     * Get map-specific configuration
     */
    map: {
      getCenter: function() {
        const lat = window.BusinessMapConfigManager.get('map.business_identity_settings.address_lat', 41.9028);
        const lng = window.BusinessMapConfigManager.get('map.business_identity_settings.address_lng', 12.4964);
        return [lat, lng];
      },

      getZoom: function(type = 'minZoom') {
        return window.BusinessMapConfigManager.get(`map.config.${type}`, 12);
      },

      getType: function() {
        return window.BusinessMapConfigManager.get('map.config.mapType', 'openstreetmap');
      },

      getAddress: function() {
        return window.BusinessMapConfigManager.getNested('map.config.address', {
          street: '',
          street2: '',
          city: '',
          postalCode: '',
          state: '',
          country: '',
          notes: ''
        });
      },

      getTransportation: function() {
        return window.BusinessMapConfigManager.getNested('map.config.transportation', {
          highwayExit: {},
          transportModeOptions: {},
          defaultTransportMode: '',
          airport: {},
          busStation: {}
        });
      },

      getParking: function() {
        return window.BusinessMapConfigManager.getNested('map.config.parking', {
          main_parking: {
            enabled: false,
            name: '',
            description: '',
            coordinates: {
              lat: null,
              lng: null
            }
          },
          capacity: {
            total: 0,
            disabled: 0,
            family: 0,
            ev: 0
          },
          rates: {
            fee_type: 'free',
            hours: '24/7'
          }
        });
      },

      // Convenience methods for parking
      parking: {
        /**
         * Check if main parking is enabled
         * @returns {boolean}
         */
        isEnabled: function() {
          const parkingConfig = window.BusinessMapConfigManager.map.getParking();
          return parkingConfig.main_parking && parkingConfig.main_parking.enabled === true;
        },

        /**
         * Get main parking coordinates
         * @returns {Object|null} {lat: number, lng: number} or null if not enabled
         */
        getCoordinates: function() {
          const parkingConfig = window.BusinessMapConfigManager.map.getParking();
          if (parkingConfig.main_parking &&
              parkingConfig.main_parking.enabled &&
              parkingConfig.main_parking.coordinates) {
            return {
              lat: parseFloat(parkingConfig.main_parking.coordinates.lat),
              lng: parseFloat(parkingConfig.main_parking.coordinates.lng)
            };
          }
          return null;
        },

        /**
         * Get main parking as Leaflet LatLng array
         * @returns {Array|null} [lat, lng] or null
         */
        getLatLng: function() {
          const coords = this.getCoordinates();
          return coords ? [coords.lat, coords.lng] : null;
        },

        /**
         * Get parking name
         * @returns {string}
         */
        getName: function() {
          const parkingConfig = window.BusinessMapConfigManager.map.getParking();
          return parkingConfig.main_parking && parkingConfig.main_parking.name
            ? parkingConfig.main_parking.name
            : Drupal.t('Parking Area');
        },

        /**
         * Get parking description
         * @returns {string}
         */
        getDescription: function() {
          const parkingConfig = window.BusinessMapConfigManager.map.getParking();
          return parkingConfig.main_parking && parkingConfig.main_parking.description
            ? parkingConfig.main_parking.description
            : '';
        },

        /**
         * Get parking capacity
         * @returns {Object}
         */
        getCapacity: function() {
          const parkingConfig = window.BusinessMapConfigManager.map.getParking();
          return parkingConfig.capacity || {
            total: 0,
            disabled: 0,
            family: 0,
            ev: 0
          };
        },

        /**
         * Get parking rates and hours
         * @returns {Object}
         */
        getRates: function() {
          const parkingConfig = window.BusinessMapConfigManager.map.getParking();
          return parkingConfig.rates || {
            fee_type: 'free',
            hours: '24/7'
          };
        },

        /**
         * Get parking fee type (translated)
         * @returns {string}
         */
        getFeeType: function() {
          const rates = this.getRates();
          const feeTypes = {
            'free': Drupal.t('Free Parking'),
            'hourly': Drupal.t('Hourly Rate'),
            'daily': Drupal.t('Daily Rate'),
            'weekly': Drupal.t('Weekly Rate')
          };
          return feeTypes[rates.fee_type] || Drupal.t('Free Parking');
        },

        /**
         * Get parking hours
         * @returns {string}
         */
        getHours: function() {
          const rates = this.getRates();
          return rates.hours || Drupal.t('24/7');
        },

        /**
         * Get formatted parking info for display
         * @returns {Object}
         */
        getInfo: function() {
          const capacity = this.getCapacity();
          const rates = this.getRates();

          return {
            name: this.getName(),
            description: this.getDescription(),
            coordinates: this.getCoordinates(),
            capacity: capacity,
            totalSpaces: capacity.total,
            disabledSpaces: capacity.disabled,
            familySpaces: capacity.family,
            evSpaces: capacity.ev,
            feeType: rates.fee_type,
            feeTypeDisplay: this.getFeeType(),
            hours: rates.hours
          };
        },

        /**
         * Check if parking has valid coordinates
         * @returns {boolean}
         */
        hasValidCoordinates: function() {
          const coords = this.getCoordinates();
          return coords &&
                 !isNaN(coords.lat) &&
                 !isNaN(coords.lng) &&
                 coords.lat >= -90 && coords.lat <= 90 &&
                 coords.lng >= -180 && coords.lng <= 180;
        }
      },

      getCustomMarkers: function() {
        return window.BusinessMapConfigManager.get('map.config.customMarkers', []);
      },

      getNearbyMarkers: function() {
        return window.BusinessMapConfigManager.get('map.config.nearbyMarkers', []);
      },

      getFullConfig: function() {
        return {
          center: this.getCenter(),
          zoom: this.getZoom(),
          mapType: this.getType(),
          address: this.getAddress(),
          transportation: this.getTransportation(),
          parking: this.getParking(),
          parkingInfo: this.parking.getInfo(),
          customMarkers: this.getCustomMarkers(),
          nearbyMarkers: this.getNearbyMarkers()
        };
      }
    },

    /**
     * Utility methods for working with parking data
     */
    utils: {
      /**
       * Parse coordinates from string or number
       * @param {string|number} coord
       * @returns {number|null}
       */
      parseCoordinate: function(coord) {
        if (coord === null || coord === undefined) return null;
        const parsed = parseFloat(coord);
        return isNaN(parsed) ? null : parsed;
      },

      /**
       * Format coordinates for display
       * @param {number} lat
       * @param {number} lng
       * @param {number} decimals - Number of decimal places
       * @returns {string}
       */
      formatCoordinates: function(lat, lng, decimals = 6) {
        const latFormatted = lat.toFixed(decimals);
        const lngFormatted = lng.toFixed(decimals);
        return `${latFormatted}, ${lngFormatted}`;
      },

      /**
       * Calculate distance between two coordinates (Haversine formula)
       * @param {number} lat1
       * @param {number} lon1
       * @param {number} lat2
       * @param {number} lon2
       * @returns {number} Distance in kilometers
       */
      calculateDistance: function(lat1, lon1, lat2, lon2) {
        const R = 6371; // Earth's radius in km
        const dLat = this.toRad(lat2 - lat1);
        const dLon = this.toRad(lon2 - lon1);
        const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                Math.cos(this.toRad(lat1)) * Math.cos(this.toRad(lat2)) *
                Math.sin(dLon/2) * Math.sin(dLon/2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
        return R * c;
      },

      /**
       * Convert degrees to radians
       * @param {number} degrees
       * @returns {number}
       */
      toRad: function(degrees) {
        return degrees * Math.PI / 180;
      },

      /**
       * Get distance from business to parking
       * @returns {number|null} Distance in km or null if parking not enabled
       */
      getBusinessToParkingDistance: function() {
        const businessCoords = window.BusinessMapConfigManager.map.getCenter();
        const parkingCoords = window.BusinessMapConfigManager.map.parking.getLatLng();

        if (!parkingCoords || !window.BusinessMapConfigManager.map.parking.isEnabled()) {
          return null;
        }

        return this.calculateDistance(
          businessCoords[0],
          businessCoords[1],
          parkingCoords[0],
          parkingCoords[1]
        );
      }
    }
  };

  // Add some convenience aliases
  window.BMConfig = window.BusinessMapConfigManager;
  window.BMParking = window.BusinessMapConfigManager.map.parking;
  window.BMUtils = window.BusinessMapConfigManager.utils;

})(Drupal, drupalSettings);
