/**
 * @file
 * Event Manager - Handles all event listeners
 */

(function() {
  'use strict';

  const EventManager = {
    mapManager: null,

    /**
     * Initialize Event Manager
     */
    initialize: function(mapManager) {
      console.log('Initializing Event Manager...');
      this.mapManager = mapManager;
      this.setupEventHandlers();
      console.log('Event Manager initialized');
    },

    /**
     * Setup all event handlers
     */
    setupEventHandlers: function() {
      // Use event delegation for all clicks
      document.addEventListener('click', this.handleClick.bind(this));

      // Map click to hide info panel
      const map = this.mapManager.getMap();
      if (map) {
        map.on('click', () => {
          const infoPanel = document.getElementById('infoPanel');
          if (infoPanel) {
            infoPanel.classList.add('hidden');
          }
        });
      }

      // Escape key handler
      document.addEventListener('keydown', this.handleKeydown.bind(this));

      // Close modal on outside click
      this.setupModalCloseHandlers();
    },

    /**
     * Handle click events
     */
    handleClick: function(event) {
      const target = event.target;

      // Close info panel
      if (target.closest('#closeInfoPanel')) {
        this.closeInfoPanel();
        return;
      }

      // Home button
      if (target.closest('#homeBtn')) {
        this.handleHomeButton();
        return;
      }

      // Share button
      if (target.closest('#shareBtn')) {
        this.handleShareButton();
        return;
      }

      // Location button
      if (target.closest('#locationBtn')) {
        this.handleLocationButton();
        return;
      }

      // Cancel share
      if (target.closest('#cancelShare')) {
        this.closeShareModal();
        return;
      }

      // Copy link
      if (target.closest('#copyLinkBtn')) {
        this.handleCopyLink();
        return;
      }

      // Navigation buttons
      const navButton = target.closest('.nav-button');
      if (navButton && navButton.id !== 'locationBtn' && navButton.dataset.target) {
        this.handleNavButton(navButton.dataset.target);
        return;
      }

      // Share buttons in modal
      if (target.closest('.share-btn')) {
        const button = target.closest('.share-btn');
        this.handleShareApp(button.dataset.app);
        return;
      }

      // Remove route button
      if (target.closest('.remove-route-btn')) {
        this.handleRemoveRoute();
        return;
      }
    },

    /**
     * Handle keydown events
     */
    handleKeydown: function(event) {
      if (event.key === 'Escape') {
        this.closeInfoPanel();
        this.closeShareModal();
      }
    },

    /**
     * Setup modal close handlers
     */
    setupModalCloseHandlers: function() {
      const shareModal = document.getElementById('shareModal');
      if (shareModal) {
        shareModal.addEventListener('click', (e) => {
          if (e.target === shareModal) {
            this.closeShareModal();
          }
        });
      }
    },

    /**
     * Close info panel
     */
    closeInfoPanel: function() {
      const infoPanel = document.getElementById('infoPanel');
      if (infoPanel) {
        infoPanel.classList.add('hidden');
      }
    },

    /**
     * Handle home button click
     */
    handleHomeButton: function() {
      window.location.href = 'https://yourbusiness.com';
    },

    /**
     * Handle share button click
     */
    handleShareButton: function() {
      if (window.ShareManager) {
        window.ShareManager.shareLocation();
      }
    },

    /**
     * Handle location button click
     */
    handleLocationButton: function() {
      if (window.LocationManager) {
        window.LocationManager.findUserLocation();
      }
      
      // Update UI
      const uiManager = window.UIManager;
      if (uiManager) {
        uiManager.updateInfoPanel(
          "My Location",
          `<div class="location-info">
            <p>Finding your current location...</p>
            <p><i class="fas fa-spinner fa-spin"></i> Please allow location access when prompted.</p>
          </div>`
        );
      }
    },

    /**
     * Handle navigation button click
     */
    handleNavButton: function(target) {
      if (window.UIManager) {
        window.UIManager.showCategory(target);
      }
    },

    /**
     * Handle copy link
     */
    handleCopyLink: function() {
      if (window.ShareManager) {
        window.ShareManager.shareViaApp('link');
      }
    },

    /**
     * Close share modal
     */
    closeShareModal: function() {
      const shareModal = document.getElementById('shareModal');
      if (shareModal) {
        shareModal.classList.add('hidden');
      }
    },

    /**
     * Handle share via specific app
     */
    handleShareApp: function(app) {
      if (window.ShareManager) {
        window.ShareManager.shareViaApp(app);
      }
      this.closeShareModal();
    },

    /**
     * Handle remove route button
     */
    handleRemoveRoute: function() {
      if (window.RouteManager) {
        window.RouteManager.removeAllRoutes();
      }
    }
  };

  // Export to global scope
  window.EventManager = EventManager;

})();