/**
 * @file
 * Location Manager - Handles user location and geolocation
 */

(function() {
  'use strict';

  const LocationManager = {
    mapManager: null,
    uiManager: null,
    userLocationMarker: null,
    userLocationCircle: null,

    /**
     * Initialize Location Manager
     */
    initialize: function(mapManager, uiManager) {
      console.log('Initializing Location Manager...');
      this.mapManager = mapManager;
      this.uiManager = uiManager;
      console.log('Location Manager initialized');
    },

    /**
     * Find user's current location
     */
    findUserLocation: function() {
      if (!navigator.geolocation) {
        this.uiManager.showNotification('Geolocation is not supported by your browser', 'error');
        return;
      }

      // Show loading state
      this.uiManager.showLocationLoading();

      navigator.geolocation.getCurrentPosition(
        this.handleLocationSuccess.bind(this),
        this.handleLocationError.bind(this),
        {
          enableHighAccuracy: true,
          timeout: 10000,
          maximumAge: 0
        }
      );
    },

    /**
     * Handle successful location retrieval
     */
    handleLocationSuccess: function(position) {
      const userLat = position.coords.latitude;
      const userLng = position.coords.longitude;

      // Remove previous user markers
      this.removeUserMarkers();

      // Add user marker
      this.addUserMarker(userLat, userLng, position.coords.accuracy);

      // Add accuracy circle
      this.addAccuracyCircle(userLat, userLng, position.coords.accuracy);

      // Show notification
      this.showUserLocationNotification();

      // Fly to user location
      this.mapManager.flyToLocation([userLat, userLng], 14, 1.5);

      // Reset button state
      this.uiManager.resetLocationButton();
      this.uiManager.activateLocationButton();

      // Show route to business
      this.showRouteToBusiness(userLat, userLng);

      // Update UI
      this.uiManager.showNotification('Your location has been found! Click "Route to Business" to get directions.', 'success');
    },

    /**
     * Handle location error
     */
    handleLocationError: function(error) {
      console.error('Geolocation error:', error);
      this.uiManager.resetLocationButton();

      let message = 'Unable to retrieve your location. ';
      switch(error.code) {
        case error.PERMISSION_DENIED:
          message += 'Please enable location permissions in your browser settings.';
          break;
        case error.POSITION_UNAVAILABLE:
          message += 'Location information is unavailable.';
          break;
        case error.TIMEOUT:
          message += 'The request to get your location timed out.';
          break;
        default:
          message += 'An unknown error occurred.';
      }
      
      this.uiManager.showNotification(message, 'error');
    },

    /**
     * Remove user markers and circle
     */
    removeUserMarkers: function() {
      if (this.userLocationMarker) {
        this.mapManager.getMap().removeLayer(this.userLocationMarker);
        this.userLocationMarker = null;
      }
      
      if (this.userLocationCircle) {
        this.mapManager.getMap().removeLayer(this.userLocationCircle);
        this.userLocationCircle = null;
      }
    },

    /**
     * Add user marker to map
     */
    addUserMarker: function(lat, lng, accuracy) {
      this.userLocationMarker = this.mapManager.addMarker(
        [lat, lng],
        'user',
        `
          <div class="user-popup">
            <h4 style="margin: 0 0 5px 0;"><i class="fas fa-user-circle"></i> Your Current Location</h4>
            <p style="margin: 0; font-size: 0.9em;">Latitude: ${lat.toFixed(6)}</p>
            <p style="margin: 0; font-size: 0.9em;">Longitude: ${lng.toFixed(6)}</p>
            <p style="margin: 5px 0; font-size: 0.9em; color: #666;">Accuracy: ${Math.round(accuracy)} meters</p>
            <button onclick="window.RouteManager.showRouteToBusiness(${lat}, ${lng})"
                    style="margin-top: 8px; padding: 8px 12px; background: #9b59b6; color: white; border: none; border-radius: 6px; cursor: pointer; font-size: 0.9em; display: flex; align-items: center; gap: 5px;">
              <i class="fas fa-route"></i> Route to Business
            </button>
          </div>
        `
      );
      
      this.userLocationMarker.openPopup();
    },

    /**
     * Add accuracy circle around user location
     */
    addAccuracyCircle: function(lat, lng, accuracy) {
      this.userLocationCircle = L.circle([lat, lng], {
        color: '#9b59b6',
        fillColor: '#9b59b6',
        fillOpacity: 0.2,
        radius: accuracy
      }).addTo(this.mapManager.getMap());
    },

    /**
     * Show user location notification
     */
    showUserLocationNotification: function() {
      const userLocationNotification = document.getElementById('userLocationNotification');
      if (userLocationNotification) {
        userLocationNotification.classList.remove('hidden');
      }
    },

    /**
     * Show route from user location to business
     */
    showRouteToBusiness: function(userLat, userLng) {
      if (window.RouteManager) {
        window.RouteManager.showRouteToBusiness(userLat, userLng);
      }
    },

    /**
     * Get user location
     */
    getUserLocation: function() {
      return new Promise((resolve, reject) => {
        if (!navigator.geolocation) {
          reject(new Error('Geolocation not supported'));
          return;
        }

        navigator.geolocation.getCurrentPosition(
          (position) => {
            resolve({
              lat: position.coords.latitude,
              lng: position.coords.longitude,
              accuracy: position.coords.accuracy
            });
          },
          (error) => {
            reject(error);
          },
          {
            enableHighAccuracy: true,
            timeout: 10000
          }
        );
      });
    }
  };

  // Export to global scope
  window.LocationManager = LocationManager;

})();