/**
 * @file
 * Map Core - Main map initialization and core functionality
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Debug helper
   */
  const debug = {
    enabled: true,
    log: function(...args) {
      if (this.enabled) {
        console.log('[Map Core]', ...args);
      }
    },
    warn: function(...args) {
      if (this.enabled) {
        console.warn('[Map Core]', ...args);
      }
    },
    error: function(...args) {
      console.error('[Map Core]', ...args);
    }
  };

  /**
   * Map Utilities
   */
  const MapUtils = {
    /**
     * Setup resize handler
     */
    setupResizeHandler: function(mapInstance) {
      const resizeHandler = () => {
        if (mapInstance) {
          mapInstance.invalidateSize();
        }
      };

      window.addEventListener('resize', resizeHandler);
      setTimeout(resizeHandler, 100);
    },

    /**
     * Add tile layer based on map type
     */
    addTileLayer: function(mapInstance, mapType, maxZoom) {
      const tileLayers = {
        'openstreetmap': {
          url: 'https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png',
          attribution: '<a href="https://www.drupal.org/u/afagioli">afagioli</a> | © <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
          defaultMaxZoom: 19
        },
        'google': {
          url: 'https://mt1.google.com/vt/lyrs=m&x={x}&y={y}&z={z}',
          attribution: '© Google Maps',
          defaultMaxZoom: 20
        }
      };

      const config = tileLayers[mapType] || tileLayers.openstreetmap;
      const finalMaxZoom = maxZoom || config.defaultMaxZoom;

      L.tileLayer(config.url, {
        attribution: config.attribution,
        maxZoom: finalMaxZoom
      }).addTo(mapInstance);
    },

    /**
     * Add business marker
     */
    addBusinessMarker: function(mapInstance, location) {
      return L.marker(location, {
        icon: L.divIcon({
          className: 'marker-pulse',
          html: '<i class="fas fa-building"></i>',
          iconSize: [26, 26]
        })
      }).addTo(mapInstance);
    },

    /**
     * Get marker icon
     */
    getMarkerIcon: function(type) {
      const iconConfig = {
        main: {
          className: 'marker-pulse',
          html: '<i class="fas fa-building"></i>',
          size: [26, 26]
        },
        toll: {
          className: 'custom-marker marker-toll',
          html: '<i class="fas fa-road"></i>',
          size: [28, 28]
        },
        highway: {
          className: 'custom-marker marker-toll',
          html: '<i class="fas fa-road"></i>',
          size: [28, 28]
        },
        train: {
          className: 'custom-marker marker-transport',
          html: '<i class="fas fa-train"></i>',
          size: [28, 28]
        },
        bus: {
          className: 'custom-marker marker-transport',
          html: '<i class="fas fa-bus"></i>',
          size: [28, 28]
        },
        subway: {
          className: 'custom-marker marker-transport',
          html: '<i class="fas fa-subway"></i>',
          size: [28, 28]
        },
        metro: {
          className: 'custom-marker marker-transport',
          html: '<i class="fas fa-subway"></i>',
          size: [28, 28]
        },
        parking: {
          className: 'custom-marker marker-parking',
          html: '<i class="fas fa-parking"></i>',
          size: [28, 28]
        },
        attraction: {
          className: 'custom-marker marker-attraction',
          html: '<i class="fas fa-star"></i>',
          size: [28, 28]
        },
        user: {
          className: 'custom-marker marker-user',
          html: '<i class="fas fa-user"></i>',
          size: [26, 26]
        }
      };

      const config = iconConfig[type] || {
        className: 'custom-marker',
        html: '<i class="fas fa-map-marker"></i>',
        size: [28, 28]
      };

      return L.divIcon({
        className: config.className,
        html: config.html,
        iconSize: config.size
      });
    }
  };

  /**
   * Map UI Manager
   */
  const MapUIManager = {
    /**
     * Setup navigation buttons
     */
    setupNavigation: function(callback) {
      const navButtons = document.querySelectorAll('.nav-button');

      navButtons.forEach(button => {
        button.addEventListener('click', (e) => {
          e.preventDefault();
          e.stopPropagation();

          const target = button.dataset.target;
          debug.log('Nav button clicked:', target);

          if (callback && typeof callback === 'function') {
            callback(target);
          }
        });
      });
    },

    /**
     * Setup location button
     */
    setupLocationButton: function(callback) {
      const locationBtn = document.getElementById('locationBtn');
      if (locationBtn) {
        locationBtn.addEventListener('click', (e) => {
          e.preventDefault();
          if (callback && typeof callback === 'function') {
            callback();
          }
        });
      }
    },

    /**
     * Setup close button
     */
    setupCloseButton: function() {
      const closeBtn = document.getElementById('closeInfoPanel');
      if (closeBtn) {
        closeBtn.addEventListener('click', () => {
          document.getElementById('infoPanel').classList.add('hidden');
        });
      }
    },

    /**
     * Setup share button
     */
    setupShareButton: function(callback) {
      const shareBtn = document.getElementById('shareBtn');
      if (shareBtn) {
        shareBtn.addEventListener('click', (e) => {
          e.preventDefault();
          if (callback && typeof callback === 'function') {
            callback();
          }
        });
      }
    },

    /**
     * Setup home button
     */
    setupHomeButton: function() {
      const homeBtn = document.getElementById('homeBtn');
      if (homeBtn) {
        homeBtn.addEventListener('click', (e) => {
          e.preventDefault();
          window.location.href = '/';
        });
      }
    },

    /**
     * Update active navigation button
     */
    updateActiveButton: function(activeCategory) {
      document.querySelectorAll('.nav-button').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.target === activeCategory);
      });
    },

    /**
     * Update location button state
     */
    updateLocationButtonState: function(button, isLoading) {
      if (!button) return;

      if (isLoading) {
        button.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>' + Drupal.t('Locating...') + '</span>';
        button.disabled = true;
      } else {
        button.innerHTML = '<i class="fas fa-location-arrow"></i><span>' + Drupal.t('My Location') + '</span>';
        button.disabled = false;
      }
    }
  };

  /**
   * Map Marker Manager
   */
  const MapMarkerManager = {
    markers: [],
    markerGroup: null,

    /**
     * Clear all markers
     */
    clearMarkers: function(mapInstance) {
      this.markers.forEach(marker => {
        if (marker && marker.remove) {
          mapInstance.removeLayer(marker);
        }
      });
      this.markers = [];

      if (this.markerGroup) {
        mapInstance.removeLayer(this.markerGroup);
        this.markerGroup = null;
      }
    },

    /**
     * Add a marker
     */
    addMarker: function(mapInstance, coords, type, popupContent) {
      const icon = MapUtils.getMarkerIcon(type);
      const marker = L.marker(coords, { icon: icon }).addTo(mapInstance);

      if (popupContent) {
        marker.bindPopup(popupContent);
      }

      this.markers.push(marker);
      return marker;
    },

    /**
     * Fit map to markers
     */
    fitMapToMarkers: function(mapInstance) {
      if (!mapInstance || this.markers.length === 0) {
        return;
      }

      this.markerGroup = L.featureGroup(this.markers);
      mapInstance.addLayer(this.markerGroup);

      setTimeout(() => {
        const bounds = this.markerGroup.getBounds();
        if (bounds && bounds.isValid()) {
          mapInstance.fitBounds(bounds, {
            padding: [50, 50],
            maxZoom: 16,
            animate: true
          });
        }
      }, 100);
    },

    /**
     * Add markers by category
     */
    addCategoryMarkers: function(mapInstance, categories) {
      const businessLocation = window.BusinessMapConfigManager.map.getCenter();
      this.addMarker(mapInstance, businessLocation, 'main', Drupal.t('Your Business Headquarters'));

      const customMarkers = window.BusinessMapConfigManager.map.getCustomMarkers();
      customMarkers.forEach(marker => {
        if (marker.enabled !== false && categories.includes(marker.category)) {
          const iconType = marker.subcategory || marker.category;
          this.addMarker(mapInstance, [marker.lat, marker.lng], iconType,
            `${marker.name || ''}<br>${marker.description || ''}`);
        }
      });

      this.fitMapToMarkers(mapInstance);
    },

    /**
     * Add attraction markers
     */
    addAttractionMarkers: function(mapInstance) {
      const businessLocation = window.BusinessMapConfigManager.map.getCenter();
      this.addMarker(mapInstance, businessLocation, 'main', Drupal.t('Your Business Headquarters'));

      const excludedCategories = ['highway', 'transport', 'parking', 'toll'];
      const nearbyMarkers = window.BusinessMapConfigManager.map.getNearbyMarkers();

      nearbyMarkers.forEach(marker => {
        if (marker.enabled !== false && !excludedCategories.includes(marker.category)) {
          this.addMarker(mapInstance, [marker.lat, marker.lng], 'attraction',
            `${marker.name || ''}<br>${marker.description || ''}`);
        }
      });

      this.fitMapToMarkers(mapInstance);
    }
  };

  /**
   * Location Service
   */
  const LocationService = {
    /**
     * Handle "My Location" button
     */
    handleMyLocation: function(onSuccess, onError, onButtonUpdate) {
      if (!navigator.geolocation) {
        alert(Drupal.t('Geolocation not supported'));
        return;
      }

      const locationBtn = document.getElementById('locationBtn');
      if (onButtonUpdate && typeof onButtonUpdate === 'function') {
        onButtonUpdate(locationBtn, true);
      }

      navigator.geolocation.getCurrentPosition(
        (position) => this.onLocationSuccess(position, locationBtn, onSuccess, onButtonUpdate),
        (error) => this.onLocationError(error, locationBtn, onError, onButtonUpdate),
        {
          enableHighAccuracy: true,
          timeout: 10000
        }
      );
    },

    /**
     * Handle location success
     */
    onLocationSuccess: function(position, locationBtn, onSuccess, onButtonUpdate) {
      const userLat = position.coords.latitude;
      const userLng = position.coords.longitude;

      debug.log('User location found:', userLat, userLng);

      if (onSuccess && typeof onSuccess === 'function') {
        onSuccess(userLat, userLng, position);
      }

      if (onButtonUpdate && typeof onButtonUpdate === 'function') {
        onButtonUpdate(locationBtn, false);
      }

      if (locationBtn) {
        locationBtn.classList.add('active');
      }
    },

    /**
     * Handle location error
     */
    onLocationError: function(error, locationBtn, onError, onButtonUpdate) {
      debug.error('Geolocation error:', error);

      if (onButtonUpdate && typeof onButtonUpdate === 'function') {
        onButtonUpdate(locationBtn, false);
      }

      if (onError && typeof onError === 'function') {
        onError(error);
      } else {
        alert(Drupal.t('Unable to get your location. Please check browser permissions.'));
      }
    },

    /**
     * Share location
     */
    shareLocation: function() {
      const shareModal = document.getElementById('shareModal');
      if (!shareModal) return;

      shareModal.classList.remove('hidden');
      this.setupShareModal(shareModal);
    },

    /**
     * Setup share modal
     */
    setupShareModal: function(modal) {
      const cancelBtn = document.getElementById('cancelShare');
      if (cancelBtn) {
        cancelBtn.onclick = () => modal.classList.add('hidden');
      }

      const shareButtons = document.querySelectorAll('.share-btn');
      shareButtons.forEach(button => {
        button.onclick = (e) => {
          e.preventDefault();
          this.shareViaApp(button.dataset.app);
          modal.classList.add('hidden');
        };
      });
    },

    /**
     * Share via specific app
     */
    shareViaApp: function(app) {
      const address = window.BusinessMapConfigManager.map.getAddress();
      const businessName = address.street || Drupal.t('Your Business');
      const addressText = this.formatShareAddress(address);

      const businessLocation = window.BusinessMapConfigManager.map.getCenter();
      const googleMapsUrl = `https://www.google.com/maps/search/?api=1&query=${businessLocation[0]},${businessLocation[1]}`;

      const message = `📍 ${businessName}\n${addressText}\n\n${Drupal.t('Find us here:')} ${googleMapsUrl}`;

      const shareActions = {
        whatsapp: () => window.open(`https://wa.me/?text=${encodeURIComponent(message)}`, '_blank'),
        email: () => window.open(`mailto:?subject=${encodeURIComponent(businessName + ' ' + Drupal.t('Location'))}&body=${encodeURIComponent(message)}`),
        sms: () => window.open(`sms:?body=${encodeURIComponent(message)}`),
        link: () => this.copyToClipboard(googleMapsUrl)
      };

      if (shareActions[app]) {
        shareActions[app]();
      }
    },

    /**
     * Format address for sharing
     */
    formatShareAddress: function(address) {
      const parts = [];
      if (address.street) parts.push(address.street);
      if (address.city) parts.push(address.city);
      if (address.country) parts.push(window.BusinessMapTemplateSystem.getCountryName(address.country));
      return parts.join(', ');
    },

    /**
     * Copy text to clipboard
     */
    copyToClipboard: function(text) {
      navigator.clipboard.writeText(text).then(() => {
        alert(Drupal.t('📍 Location link copied to clipboard!'));
      }).catch(() => {
        const tempInput = document.createElement('input');
        tempInput.value = text;
        document.body.appendChild(tempInput);
        tempInput.select();
        document.execCommand('copy');
        document.body.removeChild(tempInput);
        alert(Drupal.t('📍 Location link copied to clipboard!'));
      });
    }
  };

  /**
   * Main Map Behavior
   */
  Drupal.behaviors.businessMap = {
    mapInitialized: false,
    mapInstance: null,

    attach: function (context, settings) {
      if (this.mapInitialized) {
        return;
      }

      if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
          this.initializeMap();
        });
      } else {
        this.initializeMap();
      }
    },

    /**
     * Initialize the map
     */
    initializeMap: function() {
      const mapContainer = document.getElementById('map');
      if (!mapContainer) {
        debug.error('Map container not found');
        return;
      }

      if (typeof L === 'undefined') {
        debug.error('Leaflet not loaded');
        return;
      }

      try {
        // Check setup status
        const setupStatus = window.BusinessMapConfigManager.checkSetupComplete();
        this.setupComplete = setupStatus.isComplete;

        const initialLocation = window.BusinessMapConfigManager.map.getCenter();
        const initialZoom = window.BusinessMapConfigManager.map.getZoom();
        const maxZoom = window.BusinessMapConfigManager.get('map.config.maxZoom');

        debug.log('Creating map at:', initialLocation, 'zoom:', initialZoom);

        // Create map instance
        this.mapInstance = L.map(mapContainer, {
          zoomControl: false,
          scrollWheelZoom: true
        }).setView(initialLocation, initialZoom);

        // Add tile layer
        MapUtils.addTileLayer(this.mapInstance, window.BusinessMapConfigManager.map.getType(), maxZoom);

        // Add zoom control
        L.control.zoom({
          position: 'bottomright'
        }).addTo(this.mapInstance);

        // Add business marker
        MapUtils.addBusinessMarker(this.mapInstance, initialLocation);

        // Initialize RouteManager if available
        if (typeof window.RouteManager !== 'undefined') {
          window.RouteManager.initialize(this.mapInstance);
        }

        // Setup UI
        this.setupUI();

        // Setup resize handler
        MapUtils.setupResizeHandler(this.mapInstance);

        this.mapInitialized = true;
        debug.log('Map system fully initialized');

        // Show initial category
        setTimeout(() => {
          this.handleNavigation('where');
        }, 500);

      } catch (error) {
        debug.error('Error initializing map:', error);
      }
    },

    /**
     * Setup UI and event handlers
     */
    setupUI: function() {
      // Setup navigation
      MapUIManager.setupNavigation((target) => {
        this.handleNavigation(target);
      });

      // Setup location button
      MapUIManager.setupLocationButton(() => {
        this.handleMyLocation();
      });

      // Setup other UI components
      MapUIManager.setupCloseButton();
      MapUIManager.setupShareButton(() => {
        LocationService.shareLocation();
      });
      MapUIManager.setupHomeButton();
    },

    /**
     * Handle navigation
     */
    handleNavigation: function(category) {
      debug.log('Navigation to:', category);

      MapUIManager.updateActiveButton(category);
      this.showInfoPanel(category);
    },

    /**
     * Show info panel with content
     */
    showInfoPanel: function(category) {
      const infoPanel = document.getElementById('infoPanel');
      const infoTitle = document.getElementById('infoTitle');
      const infoContent = document.getElementById('infoContent');

      if (infoPanel) infoPanel.classList.remove('hidden');

      const templateData = {
        address: window.BusinessMapConfigManager.map.getAddress(),
        transportation: window.BusinessMapConfigManager.map.getTransportation(),
        parking: window.BusinessMapConfigManager.map.getParking(),
        nearbyMarkers: window.BusinessMapConfigManager.map.getNearbyMarkers()
      };

      const categoryConfig = {
        where: {
          title: "<i class=\"fas fa-building\"></i> " + Drupal.t('Our Location'),
          template: 'where',
          action: () => this.showWhereCategory()
        },
        car: {
          title: "<i class=\"fas fa-car icon-car\" aria-hidden=\"true\"></i>" + Drupal.t("By Car"),
          template: 'car',
          action: () => this.showCarCategory()
        },
        publicTransport: {
          title: "<i class=\"fas fa-bus icon-transport\" aria-hidden=\"true\"></i>" + Drupal.t("Public Transport"),
          template: 'publicTransport',
          action: () => this.showPublicTransportCategory()
        },
        parking: {
          title: "<i class=\"fas fa-parking icon-parking\" aria-hidden=\"true\"></i>" + Drupal.t("Parking"),
          template: 'parking',
          action: () => this.showParkingCategory()
        },
        attractions: {
          title: "<i class=\"fas fa-star icon-attractions\" aria-hidden=\"true\"></i>" + Drupal.t("Nearby Attractions"),
          template: 'attractions',
          action: () => this.showAttractionsCategory()
        }
      };

      const config = categoryConfig[category];
      if (!config) {
        debug.warn('Unknown category:', category);
        return;
      }

      if (infoTitle) infoTitle.innerHTML = config.title;
      if (infoContent) {
        infoContent.innerHTML = window.BusinessMapTemplateSystem.render(config.template, templateData);
      }

      config.action();
    },

    /**
     * Show "Our Location" category
     */
    showWhereCategory: function() {
      MapMarkerManager.clearMarkers(this.mapInstance);

      const businessLocation = window.BusinessMapConfigManager.map.getCenter();
      this.mapInstance.flyTo(businessLocation, 14, {
        duration: 2.4
      });

      if (window.RouteManager && typeof window.RouteManager.removeAllRoutes === 'function') {
        window.RouteManager.removeAllRoutes();
      }
    },

    /**
     * Show "By Car" category
     */
    showCarCategory: function() {
      MapMarkerManager.clearMarkers(this.mapInstance);
      MapMarkerManager.addCategoryMarkers(this.mapInstance, ['highway', 'toll']);

      if (window.RouteManager && typeof window.RouteManager.showCarRoute === 'function') {
        setTimeout(() => {
          try {
            window.RouteManager.showCarRoute();
          } catch (error) {
            debug.error('Error in RouteManager.showCarRoute():', error);
            this.showCarFallback();
          }
        }, 300);
      } else {
        this.showCarFallback();
      }
    },

    /**
     * Fallback for car category when RouteManager fails
     */
    showCarFallback: function() {
      const businessLocation = window.BusinessMapConfigManager.map.getCenter();

      // Add a simple polyline as route visualization
      const highwayExit = window.BusinessMapConfigManager.map.getTransportation().highwayExit;
      if (highwayExit && highwayExit.lat && highwayExit.lng) {
        const polyline = L.polyline([
          [highwayExit.lat, highwayExit.lng],
          businessLocation
        ], {
          color: '#3498db',
          weight: 4,
          opacity: 0.7,
          dashArray: '10, 10'
        }).addTo(this.mapInstance);

        // Fit map to show both points
        const bounds = L.latLngBounds([
          [highwayExit.lat, highwayExit.lng],
          businessLocation
        ]);
        this.mapInstance.fitBounds(bounds, { padding: [50, 50] });
      } else {
        // Just zoom to business location
        this.mapInstance.flyTo(businessLocation, 14, {
          duration: 1.2
        });
      }
    },

    /**
     * Show "Public Transport" category
     */
    showPublicTransportCategory: function() {
      MapMarkerManager.clearMarkers(this.mapInstance);
      MapMarkerManager.addCategoryMarkers(this.mapInstance, ['transport']);

      if (window.RouteManager && typeof window.RouteManager.showPublicTransportRoute === 'function') {
        setTimeout(() => {
          window.RouteManager.showPublicTransportRoute();
        }, 300);
      }
    },

    /**
     * Show "Parking" category
     */
    showParkingCategory: function() {
      MapMarkerManager.clearMarkers(this.mapInstance);
      MapMarkerManager.addCategoryMarkers(this.mapInstance, ['parking']);

      if (window.RouteManager && typeof window.RouteManager.removeAllRoutes === 'function') {
        window.RouteManager.removeAllRoutes();
      }

      const businessLocation = window.BusinessMapConfigManager.map.getCenter();
      this.mapInstance.flyTo(businessLocation, 15, {
        duration: 1.2
      });
    },

    /**
     * Show "Attractions" category
     */
    showAttractionsCategory: function() {
      MapMarkerManager.clearMarkers(this.mapInstance);
      MapMarkerManager.addAttractionMarkers(this.mapInstance);

      if (window.RouteManager && typeof window.RouteManager.removeAllRoutes === 'function') {
        window.RouteManager.removeAllRoutes();
      }

      const businessLocation = window.BusinessMapConfigManager.map.getCenter();
      this.mapInstance.flyTo(businessLocation, 14, {
        duration: 1.2
      });
    },

    /**
     * Handle "My Location" button
     */
    handleMyLocation: function() {
      LocationService.handleMyLocation(
        // onSuccess callback
        (userLat, userLng, position) => {
          MapMarkerManager.clearMarkers(this.mapInstance);
          MapMarkerManager.addMarker(this.mapInstance, [userLat, userLng], 'user', Drupal.t('Your Current Location'));

          const businessLocation = window.BusinessMapConfigManager.map.getCenter();
          MapMarkerManager.addMarker(this.mapInstance, businessLocation, 'main', Drupal.t('Your Business Headquarters'));

          this.mapInstance.flyTo([userLat, userLng], 14, {
            duration: 1.5
          });

          if (window.RouteManager && typeof window.RouteManager.showRouteToBusiness === 'function') {
            setTimeout(() => {
              window.RouteManager.showRouteToBusiness(userLat, userLng);
            }, 1000);
          }

          this.updateMyLocationInfo(position);
        },
        // onError callback
        (error) => {
          debug.error('Geolocation error:', error);
        },
        // onButtonUpdate callback
        MapUIManager.updateLocationButtonState
      );
    },

    /**
     * Update my location info panel
     */
    updateMyLocationInfo: function(position) {
      const infoTitle = document.getElementById('infoTitle');
      const infoContent = document.getElementById('infoContent');

      if (infoTitle) infoTitle.textContent = Drupal.t("My Location");
      if (infoContent) {
        const templateData = {
          position: {
            latitude: position.coords.latitude,
            longitude: position.coords.longitude,
            accuracy: position.coords.accuracy
          }
        };
        infoContent.innerHTML = window.BusinessMapTemplateSystem.render('myLocation', templateData);
      }
    },

    /**
     * Get map instance
     */
    getMap: function() {
      return this.mapInstance;
    }
  };

})(Drupal, drupalSettings);
