/**
 * @file
 * Map Manager - Handles Leaflet map initialization and operations
 */

(function() {
  'use strict';

  const MapManager = {
    map: null,
    businessMarker: null,
    currentMarkers: [],
    businessLocation: [41.8349, 12.4700],
    
    /**
     * Initialize the map
     */
    initialize: function(container, settings) {
      try {
        console.log('Initializing Map Manager...');
        
        // Initialize map
        this.map = L.map(container, {
          zoomControl: false
        }).setView([41.9028, 12.4964], 12);

        // Add custom zoom control
        L.control.zoom({
          position: 'bottomright'
        }).addTo(this.map);

        // Add OpenStreetMap tile layer
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
          attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
        }).addTo(this.map);

        // Initialize business marker
        this.initBusinessMarker();

        // Add sample transportation markers
        this.addSampleMarkers();

        // Handle window resize
        window.addEventListener('resize', () => {
          if (this.map) {
            this.map.invalidateSize();
          }
        });

        console.log('Map Manager initialized successfully');
      } catch (error) {
        console.error('Error in MapManager.initialize:', error);
        throw error;
      }
    },

    /**
     * Initialize the main business marker
     */
    initBusinessMarker: function() {
      this.businessMarker = L.marker(this.businessLocation, {
        icon: L.divIcon({
          className: 'marker-pulse',
          iconSize: [26, 26]
        })
      }).addTo(this.map);

      this.businessMarker.bindPopup(`
        <div class="business-popup">
          <h3 style="margin: 0 0 10px 0; color: #2c3e50;">
            <i class="fas fa-building"></i> Your Business Headquarters
          </h3>
          <p style="margin: 5px 0;">
            <strong>📍 Address:</strong><br>
            Business Street, 123<br>
            00100 Rome, Italy
          </p>
          <p style="margin: 5px 0;"><strong>📞 Phone:</strong> +39 06 1234 5678</p>
          <p style="margin: 5px 0;"><strong>✉️ Email:</strong> info@yourbusiness.com</p>
          <p style="margin: 5px 0;">
            <strong>🕒 Hours:</strong><br>
            Mon-Fri: 9AM-6PM, Sat: 9AM-1PM
          </p>
        </div>
      `);

      return this.businessMarker;
    },

    /**
     * Add sample markers for demonstration
     */
    addSampleMarkers: function() {
      const romaNordLocation = [42.0044, 12.5128];
      const trainStationLocation = [41.8518, 12.4940];
      
      // Toll booth marker
      const tollMarker = L.marker(romaNordLocation, {
        icon: this.getMarkerIcon('toll')
      }).addTo(this.map);
      tollMarker.bindPopup("<b>Roma Nord Toll Booth</b><br>Highway A1 - Exit 5");
      this.currentMarkers.push(tollMarker);

      // Train station marker
      const trainMarker = L.marker(trainStationLocation, {
        icon: this.getMarkerIcon('train')
      }).addTo(this.map);
      trainMarker.bindPopup("<b>Roma Termini Station</b><br>Main train station");
      this.currentMarkers.push(trainMarker);
    },

    /**
     * Get custom marker icon based on type
     */
    getMarkerIcon: function(type) {
      let className = 'custom-marker';
      let html = '';

      switch(type) {
        case 'main':
          className = 'marker-pulse';
          html = '<i class="fas fa-building" style="color: white; font-size: 12px;"></i>';
          break;
        case 'toll':
          className = 'custom-marker marker-toll';
          html = '<i class="fas fa-road" style="color: white; font-size: 12px;"></i>';
          break;
        case 'train':
          className = 'custom-marker marker-transport';
          html = '<i class="fas fa-train" style="color: white; font-size: 12px;"></i>';
          break;
        case 'bus':
          className = 'custom-marker marker-transport';
          html = '<i class="fas fa-bus" style="color: white; font-size: 12px;"></i>';
          break;
        case 'subway':
          className = 'custom-marker marker-transport';
          html = '<i class="fas fa-subway" style="color: white; font-size: 12px;"></i>';
          break;
        case 'parking':
          className = 'custom-marker marker-parking';
          html = '<i class="fas fa-parking" style="color: white; font-size: 12px;"></i>';
          break;
        case 'attraction':
          className = 'custom-marker marker-attraction';
          html = '<i class="fas fa-star" style="color: white; font-size: 12px;"></i>';
          break;
        case 'user':
          className = 'custom-marker marker-user';
          html = '<i class="fas fa-user" style="color: white; font-size: 12px;"></i>';
          break;
      }

      return L.divIcon({
        className: className,
        html: html,
        iconSize: type === 'main' || type === 'user' ? [26, 26] : [28, 28]
      });
    },

    /**
     * Clear all markers
     */
    clearMarkers: function() {
      this.currentMarkers.forEach(marker => {
        if (marker && marker.remove) {
          this.map.removeLayer(marker);
        }
      });
      this.currentMarkers = [];
    },

    /**
     * Add marker to map
     */
    addMarker: function(coords, type, popupContent) {
      const marker = L.marker(coords, {
        icon: this.getMarkerIcon(type)
      }).addTo(this.map);

      if (popupContent) {
        marker.bindPopup(popupContent);
      }

      this.currentMarkers.push(marker);
      return marker;
    },

    /**
     * Fly to location with animation
     */
    flyToLocation: function(coords, zoom, duration = 1.2) {
      if (this.map) {
        this.map.flyTo(coords, zoom, {
          duration: duration,
          easeLinearity: 0.25
        });
      }
    },

    /**
     * Fit bounds
     */
    fitBounds: function(bounds, padding = [50, 50]) {
      if (this.map) {
        this.map.fitBounds(bounds, { padding: padding });
      }
    },

    /**
     * Get map instance
     */
    getMap: function() {
      return this.map;
    },

    /**
     * Get business location
     */
    getBusinessLocation: function() {
      return this.businessLocation;
    },

    /**
     * Get business marker
     */
    getBusinessMarker: function() {
      return this.businessMarker;
    }
  };

  // Export to global scope
  window.MapManager = MapManager;

})();
