/**
 * @file
 * Route Manager - Handles route calculation and display
 */

(function(Drupal, drupalSettings) {
  'use strict';

  const RouteManager = {
    map: null,
    carRouteControl: null,
    publicTransportRouteControl: null,
    userRouteControl: null,
    currentPolyline: null,

    // Configuration from drupalSettings
    config: {
      businessLocation: null,
      highwayExit: null,
      airport: null,
      busStation: null,
      routeService: null,
      routing: null,
      transportModeOptions: null,
      customMarkers: null
    },

    /**
     * Initialize Route Manager
     */
    initialize: function(mapInstance) {
      if (!mapInstance || typeof mapInstance.addLayer !== 'function') {
        console.error('Invalid map instance provided to RouteManager');
        return;
      }

      this.map = mapInstance;
      this.loadConfiguration();
    },

    /**
     * Load configuration from drupalSettings
     */
    loadConfiguration: function() {
      // Get main configuration
      const mapConfig = drupalSettings?.map?.config || {};

      // Business location from center coordinates
      this.config.businessLocation = mapConfig.center || [41.8349, 12.4700];

      // Transportation data
      const transportation = mapConfig.transportation || {};
      this.config.highwayExit = transportation.highwayExit || null;
      this.config.airport = transportation.airport || null;
      this.config.busStation = transportation.busStation || null;
      this.config.defaultTransportMode = transportation.defaultTransportMode || 'driving';
      this.config.carRental = transportation.carRental || '';

      // Transport mode options
      this.config.transportModeOptions = transportation.transportModeOptions || {
        driving: Drupal.t('Driving'),
        walking: Drupal.t('Walking'),
        cycling: Drupal.t('Cycling'),
        transit: Drupal.t('Public Transport')
      };

      // Routing configuration
      this.config.routing = mapConfig.routing || {
        service: 'simulated',
        googleApiKey: '',
        mapboxToken: ''
      };

      // Custom markers for waypoints
      this.config.customMarkers = mapConfig.customMarkers || [];
    },

    /**
     * Get coordinates for a location type
     */
    getCoordinates: function(locationType) {
      switch(locationType) {
        case 'business':
          return this.config.businessLocation;

        case 'highwayExit':
          if (this.config.highwayExit && this.config.highwayExit.lat && this.config.highwayExit.lng) {
            return [this.config.highwayExit.lat, this.config.highwayExit.lng];
          }
          if (this.config.businessLocation) {
            return [
              this.config.businessLocation[0] + 0.01,
              this.config.businessLocation[1] + 0.01
            ];
          }
          return [42.0044, 12.5128];

        case 'airport':
          if (this.config.airport && this.config.airport.lat && this.config.airport.lng) {
            return [this.config.airport.lat, this.config.airport.lng];
          }
          if (this.config.businessLocation) {
            return [
              this.config.businessLocation[0] - 0.01,
              this.config.businessLocation[1] - 0.01
            ];
          }
          return [41.8518, 12.4940];

        default:
          return this.config.businessLocation;
      }
    },

    /**
     * Show car route from highway exit to business
     */
    showCarRoute: function() {
      // Remove existing routes
      this.removeAllRoutes();

      // Check if Leaflet Routing Machine is available
      if (typeof L === 'undefined' || typeof L.Routing === 'undefined') {
        this.showAlternativeRoute('car');
        return;
      }

      try {
        // Get coordinates from configuration
        const startPoint = this.getCoordinates('highwayExit');
        const endPoint = this.getCoordinates('business');

        // Create waypoints
        const waypoints = [
          L.latLng(startPoint[0], startPoint[1]),
          L.latLng(endPoint[0], endPoint[1])
        ];

        // Get routing service configuration
        const routingService = this.config.routing.service || 'simulated';
        const router = this.getRouter(routingService, 'driving');

        // Create routing control
        this.carRouteControl = L.Routing.control({
          waypoints: waypoints,
          router: router,
          routeWhileDragging: false,
          showAlternatives: false,
          lineOptions: {
            styles: [{
              color: '#3498db',
              weight: 6,
              opacity: 0.8
            }]
          },
          createMarker: function(i, waypoint, n) {
            return false;
          },
          addWaypoints: false,
          draggableWaypoints: false,
          fitSelectedRoutes: true,
          show: false,
          collapsible: false,
          autoRoute: true
        });

        // Add to map
        this.carRouteControl.addTo(this.map);

        // Handle route found event
        this.carRouteControl.on('routesfound', (e) => {
          this.handleCarRouteFound(e);
        });

        // Handle errors
        this.carRouteControl.on('routingerror', (e) => {
          this.showAlternativeRoute('car');
        });

      } catch (error) {
        this.showAlternativeRoute('car');
      }
    },

    /**
     * Show public transport route
     */
    showPublicTransportRoute: function() {
      this.removeAllRoutes();

      if (typeof L === 'undefined' || typeof L.Routing === 'undefined') {
        this.showAlternativeRoute('publicTransport');
        return;
      }

      try {
        const startPoint = this.getCoordinates('airport');
        const endPoint = this.getCoordinates('business');

        const waypoints = [
          L.latLng(startPoint[0], startPoint[1]),
          L.latLng(endPoint[0], endPoint[1])
        ];

        const routingService = this.config.routing.service || 'simulated';
        const router = this.getRouter(routingService, 'walking');

        this.publicTransportRouteControl = L.Routing.control({
          waypoints: waypoints,
          router: router,
          routeWhileDragging: false,
          showAlternatives: false,
          lineOptions: {
            styles: [{
              color: '#2ecc71',
              weight: 6,
              opacity: 0.8
            }]
          },
          createMarker: function(i, waypoint, n) {
            return false;
          },
          addWaypoints: false,
          draggableWaypoints: false,
          fitSelectedRoutes: true,
          show: false,
          collapsible: false,
          autoRoute: true
        });

        this.publicTransportRouteControl.addTo(this.map);

        this.publicTransportRouteControl.on('routesfound', (e) => {
          this.handlePublicTransportRouteFound(e);
        });

        this.publicTransportRouteControl.on('routingerror', (e) => {
          this.showAlternativeRoute('publicTransport');
        });

      } catch (error) {
        this.showAlternativeRoute('publicTransport');
      }
    },

    /**
     * Show route from user location to business
     */
    showRouteToBusiness: function(userLat, userLng) {
      this.removeAllRoutes();

      if (typeof L === 'undefined' || typeof L.Routing === 'undefined') {
        return;
      }

      try {
        const waypoints = [
          L.latLng(userLat, userLng),
          L.latLng(this.config.businessLocation[0], this.config.businessLocation[1])
        ];

        const routingService = this.config.routing.service || 'simulated';
        const router = this.getRouter(routingService, 'driving');

        this.userRouteControl = L.Routing.control({
          waypoints: waypoints,
          router: router,
          routeWhileDragging: false,
          showAlternatives: false,
          lineOptions: {
            styles: [{
              color: '#9b59b6',
              weight: 6,
              opacity: 0.8
            }]
          },
          createMarker: function(i, waypoint, n) {
            return false;
          },
          addWaypoints: false,
          draggableWaypoints: false,
          fitSelectedRoutes: true,
          show: false,
          collapsible: false,
          autoRoute: true
        });

        this.userRouteControl.addTo(this.map);

      } catch (error) {
        console.error('Error showing user route:', error);
      }
    },

    /**
     * Get router based on service configuration
     */
    getRouter: function(service, profile) {
      if (typeof L.Routing === 'undefined') {
        return null;
      }

      switch(service) {
        case 'osrm':
          return L.Routing.osrmv1({
            serviceUrl: 'https://router.project-osrm.org/route/v1',
            profile: profile,
            timeout: 10000
          });

        case 'mapbox':
          const token = this.config.routing.mapboxToken || '';
          if (!token) {
            return L.Routing.osrmv1({
              serviceUrl: 'https://router.project-osrm.org/route/v1',
              profile: profile,
              timeout: 10000
            });
          }
          return L.Routing.mapbox(token, {
            profile: profile === 'transit' ? 'driving-traffic' : profile,
            timeout: 10000
          });

        default:
          return L.Routing.osrmv1({
            serviceUrl: 'https://router.project-osrm.org/route/v1',
            profile: profile,
            timeout: 10000
          });
      }
    },

    /**
     * Handle car route found
     */
    handleCarRouteFound: function(e) {
      if (!e.routes || e.routes.length === 0) {
        this.showAlternativeRoute('car');
        return;
      }

      const route = e.routes[0];
      let distance = 'N/A';
      let time = 'N/A';

      if (route.summary) {
        distance = (route.summary.totalDistance / 1000).toFixed(1);
        time = Math.round(route.summary.totalTime / 60);
      }

      const highwayExitName = this.config.highwayExit ?
        this.config.highwayExit.name || Drupal.t('Highway Exit') :
        Drupal.t('Highway Exit');

      const highwayExitDesc = this.config.highwayExit ?
        this.config.highwayExit.description || '' : '';

      this.updateRouteDetails(`
        <div class="route-info">
          <p><strong>🚗 ${Drupal.t('Car Route from')} ${highwayExitName}:</strong></p>
          ${highwayExitDesc ? `<p><em>${highwayExitDesc}</em></p>` : ''}
          <p>📏 <strong>${Drupal.t('Distance:')}</strong> ${distance} km</p>
          <p>⏱️ <strong>${Drupal.t('Estimated time:')}</strong> ${time} ${Drupal.t('minutes')}</p>
          <button class="remove-route-btn" onclick="window.RouteManager.removeAllRoutes()">
            <i class="fas fa-times"></i> ${Drupal.t('Remove Route')}
          </button>
        </div>
      `);
    },

    /**
     * Handle public transport route found
     */
    handlePublicTransportRouteFound: function(e) {
      if (!e.routes || e.routes.length === 0) {
        this.showAlternativeRoute('publicTransport');
        return;
      }

      const route = e.routes[0];
      let distance = 'N/A';
      let time = 'N/A';

      if (route.summary) {
        distance = (route.summary.totalDistance / 1000).toFixed(1);
        time = Math.round(route.summary.totalTime / 60);
      }

      const airportName = this.config.airport ?
        this.config.airport.name || Drupal.t('Airport') :
        Drupal.t('Airport');

      const airportDesc = this.config.airport ?
        this.config.airport.description || '' : '';

      this.updateRouteDetails(`
        <div class="route-info">
          <p><strong>🚇 ${Drupal.t('Public Transport Route from')} ${airportName}:</strong></p>
          ${airportDesc ? `<p><em>${airportDesc}</em></p>` : ''}
          <p>📏 <strong>${Drupal.t('Walking distance:')}</strong> ${distance} km</p>
          <p>⏱️ <strong>${Drupal.t('Walking time:')}</strong> ${time} ${Drupal.t('minutes')}</p>
          <button class="remove-route-btn" onclick="window.RouteManager.removeAllRoutes()">
            <i class="fas fa-times"></i> ${Drupal.t('Remove Route')}
          </button>
        </div>
      `);
    },

    /**
     * Show alternative route when routing not available
     */
    showAlternativeRoute: function(type) {
      let content = '';
      let polylineCoords = [];
      let color = '#3498db';

      if (type === 'car') {
        const startPoint = this.getCoordinates('highwayExit');
        const endPoint = this.getCoordinates('business');
        const highwayExitName = this.config.highwayExit ?
          this.config.highwayExit.name || Drupal.t('Highway Exit') :
          Drupal.t('Highway Exit');

        content = `
          <div class="route-info">
            <p><strong>🚗 ${Drupal.t('Car Route from')} ${highwayExitName}:</strong></p>
            <p>📏 <strong>${Drupal.t('Distance:')}</strong> ${Drupal.t('Approximately')} 18 km</p>
            <p>⏱️ <strong>${Drupal.t('Estimated time:')}</strong> 25-35 ${Drupal.t('minutes')}</p>
            ${this.config.highwayExit && this.config.highwayExit.description ?
              `<p>🛣️ <strong>${Drupal.t('Directions:')}</strong> ${this.config.highwayExit.description}</p>` : ''}
            <button class="remove-route-btn" onclick="window.RouteManager.removeAllRoutes()">
              <i class="fas fa-times"></i> ${Drupal.t('Remove Route')}
            </button>
          </div>
        `;
        polylineCoords = [startPoint, endPoint];
        color = '#3498db';
      }

      this.updateRouteDetails(content);

      // Draw a simple polyline as fallback
      if (polylineCoords.length > 0 && this.map) {
        if (this.currentPolyline && this.map.hasLayer(this.currentPolyline)) {
          this.map.removeLayer(this.currentPolyline);
        }

        this.currentPolyline = L.polyline(polylineCoords, {
          color: color,
          weight: 4,
          opacity: 0.6,
          dashArray: '10, 10'
        }).addTo(this.map);

        if (polylineCoords.length >= 2) {
          this.map.fitBounds(polylineCoords, { padding: [50, 50] });
        }
      }
    },

    /**
     * Update route details display
     */
    updateRouteDetails: function(content) {
      const routeDetails = document.getElementById('routeDetails');
      if (!routeDetails) {
        const infoContent = document.getElementById('infoContent');
        if (infoContent) {
          infoContent.innerHTML += content;
        }
        return;
      }

      routeDetails.innerHTML = content;
    },

    /**
     * Remove all routes from map
     */
    removeAllRoutes: function() {
      // Remove routing controls
      const controls = [
        { name: 'carRouteControl', control: this.carRouteControl },
        { name: 'publicTransportRouteControl', control: this.publicTransportRouteControl },
        { name: 'userRouteControl', control: this.userRouteControl }
      ];

      controls.forEach(item => {
        if (item.control && this.map) {
          try {
            if (typeof item.control.remove === 'function') {
              item.control.remove();
            } else if (typeof item.control.removeFrom === 'function') {
              item.control.removeFrom(this.map);
            } else if (typeof this.map.removeControl === 'function') {
              this.map.removeControl(item.control);
            } else if (item.control._container && item.control._container.parentNode) {
              item.control._container.parentNode.removeChild(item.control._container);
            }
          } catch (e) {
            // Ignore errors
          }

          this[item.name] = null;
        }
      });

      // Remove polyline fallback
      if (this.currentPolyline && this.map) {
        try {
          if (this.map.hasLayer(this.currentPolyline)) {
            this.map.removeLayer(this.currentPolyline);
          }
        } catch (e) {
          // Ignore errors
        }
        this.currentPolyline = null;
      }

      // Clear route details
      this.updateRouteDetails('');
    }
  };

  // Export to global scope
  window.RouteManager = RouteManager;

})(Drupal, drupalSettings);
