/**
 * @file
 * Share Manager - Handles location sharing functionality
 */

(function() {
  'use strict';

  const ShareManager = {
    mapManager: null,
    uiManager: null,
    
    // Business information
    businessName: "Your Business",
    address: "Business Street, 123 - 00100 Rome, Italy",
    businessLocation: [41.8349, 12.4700],

    /**
     * Initialize Share Manager
     */
    initialize: function(mapManager, uiManager) {
      console.log('Initializing Share Manager...');
      this.mapManager = mapManager;
      this.uiManager = uiManager;
      console.log('Share Manager initialized');
    },

    /**
     * Share location - opens share modal
     */
    shareLocation: function() {
      const shareModal = document.getElementById('shareModal');
      if (shareModal) {
        shareModal.classList.remove('hidden');
      }
    },

    /**
     * Share via specific app
     */
    shareViaApp: function(app) {
      const googleMapsUrl = this.getGoogleMapsUrl();
      const message = this.getShareMessage();

      switch(app) {
        case 'whatsapp':
          this.shareViaWhatsApp(message);
          break;
        case 'email':
          this.shareViaEmail(message);
          break;
        case 'sms':
          this.shareViaSMS(message);
          break;
        case 'link':
          this.copyLinkToClipboard(googleMapsUrl);
          break;
      }

      this.closeShareModal();
    },

    /**
     * Get Google Maps URL
     */
    getGoogleMapsUrl: function() {
      const [lat, lng] = this.businessLocation;
      return `https://www.google.com/maps/search/?api=1&query=${lat},${lng}`;
    },

    /**
     * Get share message
     */
    getShareMessage: function() {
      const googleMapsUrl = this.getGoogleMapsUrl();
      return `📍 ${this.businessName}\n${this.address}\n\nFind us here: ${googleMapsUrl}`;
    },

    /**
     * Share via WhatsApp
     */
    shareViaWhatsApp: function(message) {
      window.open(`https://wa.me/?text=${encodeURIComponent(message)}`, '_blank');
    },

    /**
     * Share via Email
     */
    shareViaEmail: function(message) {
      window.open(`mailto:?subject=${encodeURIComponent(this.businessName + ' Location')}&body=${encodeURIComponent(message)}`);
    },

    /**
     * Share via SMS
     */
    shareViaSMS: function(message) {
      window.open(`sms:?body=${encodeURIComponent(message)}`);
    },

    /**
     * Copy link to clipboard
     */
    copyLinkToClipboard: function(url) {
      if (navigator.clipboard && window.isSecureContext) {
        // Use modern clipboard API
        navigator.clipboard.writeText(url).then(() => {
          this.uiManager.showNotification('📍 Location link copied to clipboard!', 'success');
        }).catch((err) => {
          console.error('Clipboard API error:', err);
          this.fallbackCopyToClipboard(url);
        });
      } else {
        // Fallback for older browsers
        this.fallbackCopyToClipboard(url);
      }
    },

    /**
     * Fallback copy method
     */
    fallbackCopyToClipboard: function(url) {
      try {
        const tempInput = document.createElement('input');
        tempInput.value = url;
        document.body.appendChild(tempInput);
        tempInput.select();
        tempInput.setSelectionRange(0, 99999); // For mobile devices
        
        const successful = document.execCommand('copy');
        document.body.removeChild(tempInput);
        
        if (successful) {
          this.uiManager.showNotification('📍 Location link copied to clipboard!', 'success');
        } else {
          throw new Error('Copy command failed');
        }
      } catch (err) {
        console.error('Fallback copy failed:', err);
        this.uiManager.showNotification('Failed to copy link. Please copy manually.', 'error');
        // Show URL for manual copy
        prompt('Copy this link:', url);
      }
    },

    /**
     * Close share modal
     */
    closeShareModal: function() {
      const shareModal = document.getElementById('shareModal');
      if (shareModal) {
        shareModal.classList.add('hidden');
      }
    },

    /**
     * Generate QR code for location
     */
    generateQRCode: function(elementId) {
      // This method can be extended to generate QR codes
      // Currently returns the Google Maps URL for QR generation
      return this.getGoogleMapsUrl();
    },

    /**
     * Get embed code for map
     */
    getEmbedCode: function() {
      const [lat, lng] = this.businessLocation;
      return `<iframe src="https://www.google.com/maps/embed/v1/place?key=YOUR_API_KEY&q=${lat},${lng}" width="600" height="450" style="border:0;" allowfullscreen="" loading="lazy"></iframe>`;
    }
  };

  // Export to global scope
  window.ShareManager = ShareManager;

})();